/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * TextWindow.cpp
 *
 *  Created on: Jan 7, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/shared_ptr.hpp>

#include "Menu/Menu.hpp"
#include "Menu/MenuDescription.hpp"
#include "Menu/TextMenu.hpp"
#include "Menu/ActionMenuItem.hpp"
#include "Menu/SeperatorItem.hpp"
#include "Menu/DisplayMenuItem.hpp"
#include "Menu/SubMenuItem.hpp"
#include "TextUI/TextStatusIndicator.hpp"
#include "TextUI/TextWindow.hpp"
#include "Actions/MethodAction.hpp"
#include "Actions/ErrorAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/ActionTraits.hpp"
#include "Parser/ChangeTracker.hpp"
#include "Views/StreamStringView.hpp"
#include "Views/MethodStringView.hpp"

#include "defs.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"

// all needed due to config::SaveAll()
#include "config.hpp"
#include "periodentafel.hpp"

// config::SaveAll() and enumerate()
#include "molecule.hpp"

#include <iostream>
#include <map>

// TODO: see what code can be moved to a base class for Graphic and Text Windows
TextWindow::TextWindow()
{
  map <std::string, TextMenu *> NametoTextMenuMap;
  std::set <char> ShortcutList;
  // reserve s for save and q for quite
  ShortcutList.insert('s');
  ShortcutList.insert('q');

  // build the main menu
  main_menu = new TextMenu(Log() << Verbose(0), "Main Menu");

  moleculeView = new StreamStringView(boost::bind(&MoleculeListClass::Enumerate,World::getInstance().getMolecules(),_1));
  new DisplayMenuItem(main_menu,moleculeView,"Molecule List");

  new SeperatorItem(main_menu);

  Action* undoAction = ActionRegistry::getInstance().getActionByName("undo");
  new ActionMenuItem(getSuitableShortForm(ShortcutList,"Undo last operation"),"Undo last operation",main_menu,undoAction);

  Action* redoAction = ActionRegistry::getInstance().getActionByName("redo");
  new ActionMenuItem(getSuitableShortForm(ShortcutList,"Redo last operation"),"Redo last operation",main_menu,redoAction);

  new SeperatorItem(main_menu);

  MenuDescription menudescriptions;
  for(MenuDescription::const_iterator iter = menudescriptions.getBeginIter(); iter != menudescriptions.getEndIter(); ++iter) {
    const std::string &MenuName = iter->first;
    TextMenu *Menu = new TextMenu((ostream &)std::cout, menudescriptions.getDescription(MenuName));
    NametoTextMenuMap.insert( pair <std::string, TextMenu *> (MenuName, Menu) );
    new SubMenuItem(getSuitableShortForm(ShortcutList,MenuName),menudescriptions.getDescription(MenuName).c_str(),main_menu,Menu);
  }

  new SeperatorItem(main_menu);

  // save has reserved key 's'
  Action *saveConfigAction = ActionRegistry::getInstance().getActionByName("output");
  new ActionMenuItem('s',"save current setup to config files",main_menu,saveConfigAction);

  ActionTraits quitTrait(OptionTrait("quitAction", &typeid(void), "quits the program"));
  quitAction = new MethodAction(quitTrait,boost::bind(&TextMenu::doQuit,main_menu),false);
  new ActionMenuItem('q',"quit",main_menu,quitAction);

  // quit has reserved key 'q'
  // go through all menus and create them
  for (map <std::string, TextMenu *>::iterator MenuRunner = NametoTextMenuMap.begin(); MenuRunner != NametoTextMenuMap.end(); ++MenuRunner)
    populateMenu(MenuRunner->second, MenuRunner->first);

  // Add status indicators etc...

  statusIndicator = new TextStatusIndicator();
}

TextWindow::~TextWindow()
{
  for (std::list<Action *>::iterator iter = returnFromActions.begin(); !returnFromActions.empty(); ++iter)
    delete (*iter);
  returnFromActions.clear();
  delete quitAction;
  delete moleculeView;
  delete statusIndicator;
  delete main_menu;
}

void TextWindow::display() {
  main_menu->display();
}

char TextWindow::getSuitableShortForm(std::set <char> &ShortcutList, const std::string name) const
{
  for (std::string::const_iterator CharRunner = name.begin(); CharRunner != name.end(); ++CharRunner) {
    if (ShortcutList.find(*CharRunner) == ShortcutList.end()) {
      ShortcutList.insert(*CharRunner);
      return *CharRunner;
    }
  }
  // if no letter matches, take digits
  int i=0;
  for (;i<10;++i) {
    if (ShortcutList.find((char)i + '0') == ShortcutList.end())
      break;
  }
  if (i != 10) {
    ShortcutList.insert((char)i + '0');
    return ((char)i + '0');
  } else {
    DoeLog(1) && (eLog() << Verbose(1) << "Could not find a suitable shortform for " << name << "." << endl);
    return '#';
  }
}

void TextWindow::populateMenu(TextMenu* Menu, const  std::string &MenuName)
{
  Action *ActionItem = NULL;
  set <char> ShortcutList;
  // reserve 'q' for quit
  ShortcutList.insert('q');
  // through all actions for this menu
  MenuDescription md;
  std::multimap <std::string, std::string> MenuItems = md.getMenuItemsMap();
  std::pair < std::multimap <std::string, std::string>::iterator, std::multimap <std::string, std::string>::iterator > MenuActions = MenuItems.equal_range(MenuName);
  for (std::multimap <std::string, std::string>::const_iterator MenuRunner = MenuActions.first; MenuRunner != MenuActions.second; ++MenuRunner) {
    ActionItem = ActionRegistry::getInstance().getActionByName(MenuRunner->second);
    new ActionMenuItem(getSuitableShortForm(ShortcutList, MenuRunner->second),ActionItem->Traits.getDescription().c_str(),Menu,ActionItem);
  }
  // finally add default quit item
  ActionTraits LeaveActionTraits("Leave menu: "+Menu->getTitle());
  Action *returnFromAction = new TextMenu::LeaveAction(Menu, LeaveActionTraits);
  //returnFromActions.push_back(returnFromAction);
  MenuItem *returnFromItem = new ActionMenuItem('q',"return to Main menu",Menu,returnFromAction);
  Menu->addDefault(returnFromItem);
}
