/*
 * Summator.hpp
 *
 *  Created on: 28.07.2012
 *      Author: heber
 */

#ifndef SUMMATOR_HPP_
#define SUMMATOR_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/fusion/sequence.hpp>

#include "Fragmentation/Summation/Summation.hpp"
#include "Fragmentation/Summation/SetValue.hpp"
#include "Jobs/MPQCDataMap.hpp"

/** Summator is a general class for making us of Summation.
 *
 * The idea is that we want to sum up not only one value but a whole bunch of.
 * However, in general the types of the values will all differ. We would like
 * to call the summation in a as simple a fashion as possible and at best in
 * a kind of for_each on each type/value.
 * For this purpose we require a list of all the types, represented  by the
 * MPQCDataMap_t map that uses the types enumerated in namespace MPQCDataFused,
 * see the simple example for the associative sequence in boost::fusion.
 * The MPQCDataMap_t then gives the essential mapping from the type to a specific
 * key. Via this key, which is a type and hence can be used for template
 * specification, we may gather all information for launching the
 * Summation. We only need to convert MPQCData into the above
 * MPQCDataMap_t instance and may then access the member variables via the above
 * key and also obtain the type at \b run-time through the key.
 *
 * Note that we then need a conversion from the type stored in the MPQCData,
 * e.g. a vector<double>, to the type in MPQCDataMap_t, e.g. Histogram.
 *
 * Boost::fusion is very cool! If we have a namespace with just structs as keys
 * and the fusion::map from these keys to the true types, then we may do ...
 * \code
 *   MPQCDataMap_t MPQCDataMap;
 *   using namespace MPQCDataFused;
 *   double MPQCData_energy_total = boost::fusion::at_key<MPQCDataFused::energy_total>(MPQCDataMap);
 * \endcode
 * i.e. at_key knows the correct type!
 *
 */
template <typename MapType, typename MapKey>
struct Summator {
  /** We retrieve the type of the MPQCData member variable from the
   * boost::fusion::map and stored it in this typedef. Note that for
   * internal types (e.g. MapValue::const_iterator we need to place
   * \b typename before).
   */
  typedef typename boost::fusion::result_of::value_at_key<MapType, MapKey>::type MapValue;

  /** Constructor for class Summator.
   *
   * \param _subsetmap map with hierarchy of IndexSet's
   * \param _data MPQCData converted to MPQCDataMap_t type
   * \param _jobids job ids to sum data in correct order
   * \param _container container of IndexSet's such that each set has correct order
   *        to job id and hence to _data.
   * \param _MatrixNrLookup lookup from job id to ordering in above vectors
   */
  Summator(
      SubsetMap::ptr &_subsetmap,
      const std::vector<MapType> &_data,
      const std::vector<JobId_t> &_jobids,
      const IndexSetContainer::Container_t &_container,
      std::map< JobId_t, size_t > &_MatrixNrLookup) : /* cannot make this const due to operator[] */
    indices(getSubsets(_container)),
    values(createValues(_data, _jobids, _container, _MatrixNrLookup)),
    OS(indices, values, _subsetmap)
  {
    ASSERT( _data.size() == _jobids.size(),
        "Summator() - data and ids don't have same size.");
    ASSERT( _jobids.size() == _MatrixNrLookup.size(),
        "Summator() - ids and MatrixNrLookup don't have same size.");
  }

  /** Summation operator.
   *
   * Initialises instantiated Summation of the respective type via
   * \a Summator::data, uses Summation::operator() to sum and returns
   * the result.
   *
   * \param level up to which level to sum up
   * \return result of Summation for given type from MPQCDataMap_t.
   */
  MapValue operator()(const size_t level)
  {
    // evaluate
    const MapValue result = OS(level);
    return result;
  }

private:
  /** Tiny helper to create the indices from a given IndexSetContainer.
   *
   * Basically, we just have to make sure we miss the last one but only
   * if the \a container has more than one set.
   *
   * @param container container with IndexSet
   * @return all subsets contained in \a container
   */
  typename Summation<MapValue>::InputSets_t getSubsets(
      const IndexSetContainer::Container_t &container)
  {
    // if we just have one indexset, we don't need to get rid of the "union index set"
    typename Summation<MapValue>::InputSets_t indices(
        container.begin(),
        container.end()-1 == container.begin() ? container.end() : container.end()-1);
    return indices;
  }

  /** Tiny helper to create the values for the summation in the correct order.
   *
   * @param data
   * @param jobids
   * @param container
   * @param MatrixNrLookup
   * @return
   */
  typename Summation<MapValue>::InputValues_t createValues(
      const std::vector<MapType> &data,
      const std::vector<JobId_t> &jobids,
      const IndexSetContainer::Container_t &container,
      std::map< JobId_t, size_t > &MatrixNrLookup)
  {
    // if we just have one indexset, we don't need to get rid of the "union index set"
    typename Summation<MapValue>::InputValues_t values(
        container.size() == 1 ? (size_t)1 : container.size()-1);
    typename std::vector<MapType>::const_iterator dataiter = data.begin();
    std::vector<size_t>::const_iterator iditer = jobids.begin();
    for (; dataiter != data.end(); ++dataiter, ++iditer) {
      const MapType &Data = *dataiter;
      const MapValue &value = boost::fusion::at_key<MapKey>(Data);
      values[ MatrixNrLookup[*iditer] ] = value;
    }
    return values;
  }
private:
  //!> created indices for OS such that we may hand over refs
  typename Summation<MapValue>::InputSets_t indices;
  //!> created values for OS such that we may hand over refs
  typename Summation<MapValue>::InputValues_t values;
  //!> Summation instance to use in operator()(level)
  Summation<MapValue> OS;
};


#endif /* SUMMATOR_HPP_ */
