/** \file molecule.hpp
 *
 * Class definitions of atom and molecule, element and periodentafel
 */

#ifndef MOLECULES_HPP_
#define MOLECULES_HPP_

using namespace std;

/*********************************************** includes ***********************************/

// GSL headers
#include <gsl/gsl_eigen.h>
#include <gsl/gsl_heapsort.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_multimin.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_randist.h>

//// STL headers
#include <map>
#include <set>
#include <deque>
#include <list>
#include <vector>

#include "graph.hpp"
#include "stackclass.hpp"
#include "tesselation.hpp"

/****************************************** forward declarations *****************************/

class atom;
class bond;
class element;
class ForceMatrix;
class LinkedCell;
class molecule;
class MoleculeLeafClass;
class MoleculeListClass;
class periodentafel;
class Vector;

/******************************** Some definitions for easier reading **********************************/

#define MoleculeList list <molecule *>
#define MoleculeListTest pair <MoleculeList::iterator, bool>

#define DistancePair pair < double, atom* >
#define DistanceMap multimap < double, atom* >
#define DistanceTestPair pair < DistanceMap::iterator, bool>


/************************************* Class definitions ****************************************/

/** Structure to contain parameters needed for evaluation of constraint potential.
 */
struct EvaluatePotential
{
  int startstep;              //!< start configuration (MDStep in atom::trajectory)
  int endstep;                //!< end configuration (MDStep in atom::trajectory)
  atom **PermutationMap;      //!< gives target ptr for each atom, array of size molecule::AtomCount (this is "x" in \f$ V^{con}(x) \f$ )
  DistanceMap **DistanceList; //!< distance list of each atom to each atom
  DistanceMap::iterator *StepList; //!< iterator to ascend through NearestNeighbours \a **DistanceList
  int *DoubleList;            //!< count of which sources want to move to this target, basically the injective measure (>1 -> not injective)
  DistanceMap::iterator *DistanceIterators; //!< marks which was the last picked target as injective candidate with smallest distance
  bool IsAngstroem;           //!< whether coordinates are in angstroem (true) or bohrradius (false)
  double *PenaltyConstants;   //!<  penalty constant in front of each term
};

#define MaxThermostats 6      //!< maximum number of thermostat entries in Ions#ThermostatNames and Ions#ThermostatImplemented 
enum thermostats { None, Woodcock, Gaussian, Langevin, Berendsen, NoseHoover };   //!< Thermostat names for output


/** The complete molecule.
 * Class incorporates number of types
 */
class molecule : public PointCloud {
  public:
    double cell_size[6];//!< cell size
    periodentafel *elemente; //!< periodic table with each element
    atom *start;        //!< start of atom list
    atom *end;          //!< end of atom list
    bond *first;        //!< start of bond list
    bond *last;         //!< end of bond list
    bond ***ListOfBondsPerAtom; //!< pointer list for each atom and each bond it has
    int MDSteps;        //!< The number of MD steps in Trajectories
    int *NumberOfBondsPerAtom;  //!< Number of Bonds each atom has
    int AtomCount;          //!< number of atoms, brought up-to-date by CountAtoms()
    int BondCount;          //!< number of atoms, brought up-to-date by CountBonds()
    int ElementCount;       //!< how many unique elements are therein
    int ElementsInMolecule[MAX_ELEMENTS]; //!< list whether element (sorted by atomic number) is alread present or not
    int NoNonHydrogen;  //!< number of non-hydrogen atoms in molecule
    int NoNonBonds;     //!< number of non-hydrogen bonds in molecule
    int NoCyclicBonds;  //!< number of cyclic bonds in molecule, by DepthFirstSearchAnalysis()
    double BondDistance;  //!< typical bond distance used in CreateAdjacencyList() and furtheron
    bool ActiveFlag;    //!< in a MoleculeListClass used to discern active from inactive molecules
    Vector Center;      //!< Center of molecule in a global box
    char name[MAXSTRINGSIZE];         //!< arbitrary name
    int IndexNr;        //!< index of molecule in a MoleculeListClass
    class Tesselation *TesselStruct;

  molecule(periodentafel *teil);
  virtual ~molecule();

  // re-definition of virtual functions from PointCloud
  Vector *GetCenter(ofstream *out);
  TesselPoint *GetPoint();
  TesselPoint *GetTerminalPoint();
  void GoToNext();
  void GoToPrevious();
  void GoToFirst();
  void GoToLast();
  bool IsEmpty();
  bool IsEnd();

  // templates for allowing global manipulation of all vectors
  template <typename res> void ActOnAllVectors( res (Vector::*f)() );
  template <typename res> void ActOnAllVectors( res (Vector::*f)() const);
  template <typename res> void ActOnAllVectors( res (Vector::*f)() ) const;
  template <typename res> void ActOnAllVectors( res (Vector::*f)() const) const;
  template <typename res, typename T> void ActOnAllVectors( res (Vector::*f)(T), T t );
  template <typename res, typename T> void ActOnAllVectors( res (Vector::*f)(T) const, T t );
  template <typename res, typename T> void ActOnAllVectors( res (Vector::*f)(T), T t ) const;
  template <typename res, typename T> void ActOnAllVectors( res (Vector::*f)(T) const, T t ) const;
  template <typename res, typename T, typename U> void ActOnAllVectors( res (Vector::*f)(T, U), T t, U u );
  template <typename res, typename T, typename U> void ActOnAllVectors( res (Vector::*f)(T, U) const, T t, U u );
  template <typename res, typename T, typename U> void ActOnAllVectors( res (Vector::*f)(T, U), T t, U u ) const;
  template <typename res, typename T, typename U> void ActOnAllVectors( res (Vector::*f)(T, U) const, T t, U u ) const;
  template <typename res, typename T, typename U, typename V> void ActOnAllVectors( res (Vector::*f)(T, U, V), T t, U u, V v);
  template <typename res, typename T, typename U, typename V> void ActOnAllVectors( res (Vector::*f)(T, U, V) const, T t, U u, V v);
  template <typename res, typename T, typename U, typename V> void ActOnAllVectors( res (Vector::*f)(T, U, V), T t, U u, V v) const;
  template <typename res, typename T, typename U, typename V> void ActOnAllVectors( res (Vector::*f)(T, U, V) const, T t, U u, V v) const;

  // templates for allowing global manipulation of molecule with each atom as single argument
  template <typename res> void ActWithEachAtom( res (molecule::*f)(atom *) );
  template <typename res> void ActWithEachAtom( res (molecule::*f)(atom *) const);
  template <typename res> void ActWithEachAtom( res (molecule::*f)(atom *) ) const;
  template <typename res> void ActWithEachAtom( res (molecule::*f)(atom *) const) const;

  // templates for allowing global copying of molecule with each atom as single argument
  template <typename res> void ActOnCopyWithEachAtom( res (molecule::*f)(atom *) , molecule *copy);
  template <typename res> void ActOnCopyWithEachAtom( res (molecule::*f)(atom *) const , molecule *copy);
  template <typename res> void ActOnCopyWithEachAtom( res (molecule::*f)(atom *) , molecule *copy) const;
  template <typename res> void ActOnCopyWithEachAtom( res (molecule::*f)(atom *) const, molecule *copy) const;

  // templates for allowing global manipulation of all atoms
  template <typename res> void ActOnAllAtoms( res (atom::*f)() );
  template <typename res> void ActOnAllAtoms( res (atom::*f)() const );
  template <typename res> void ActOnAllAtoms( res (atom::*f)() ) const;
  template <typename res> void ActOnAllAtoms( res (atom::*f)() const) const;
  template <typename res, typename T> void ActOnAllAtoms( res (atom::*f)(T), T t );
  template <typename res, typename T> void ActOnAllAtoms( res (atom::*f)(T) const, T t );
  template <typename res, typename T> void ActOnAllAtoms( res (atom::*f)(T), T t ) const;
  template <typename res, typename T> void ActOnAllAtoms( res (atom::*f)(T) const, T t ) const;
  template <typename res, typename T, typename U> void ActOnAllAtoms( res (atom::*f)(T, U), T t, U u );
  template <typename res, typename T, typename U> void ActOnAllAtoms( res (atom::*f)(T, U) const, T t, U u );
  template <typename res, typename T, typename U> void ActOnAllAtoms( res (atom::*f)(T, U), T t, U u ) const;
  template <typename res, typename T, typename U> void ActOnAllAtoms( res (atom::*f)(T, U) const, T t, U u ) const;
  template <typename res, typename T, typename U, typename V> void ActOnAllAtoms( res (atom::*f)(T, U, V), T t, U u, V v);
  template <typename res, typename T, typename U, typename V> void ActOnAllAtoms( res (atom::*f)(T, U, V) const, T t, U u, V v);
  template <typename res, typename T, typename U, typename V> void ActOnAllAtoms( res (atom::*f)(T, U, V), T t, U u, V v) const;
  template <typename res, typename T, typename U, typename V> void ActOnAllAtoms( res (atom::*f)(T, U, V) const, T t, U u, V v) const;
  template <typename res, typename T, typename U, typename V, typename W> void ActOnAllAtoms( res (atom::*f)(T, U, V, W), T t, U u, V v, W w);
  template <typename res, typename T, typename U, typename V, typename W> void ActOnAllAtoms( res (atom::*f)(T, U, V, W) const, T t, U u, V v, W w);
  template <typename res, typename T, typename U, typename V, typename W> void ActOnAllAtoms( res (atom::*f)(T, U, V, W), T t, U u, V v, W w) const;
  template <typename res, typename T, typename U, typename V, typename W> void ActOnAllAtoms( res (atom::*f)(T, U, V, W) const, T t, U u, V v, W w) const;

  // templates for allowing conditional global copying of molecule with each atom as single argument
  template <typename res> void ActOnCopyWithEachAtomIfTrue( res (molecule::*f)(atom *) , molecule *copy, bool (atom::*condition) () );
  template <typename res, typename T> void ActOnCopyWithEachAtomIfTrue( res (molecule::*f)(atom *) , molecule *copy, bool (atom::*condition) (T), T t );
  template <typename res, typename T, typename U> void ActOnCopyWithEachAtomIfTrue( res (molecule::*f)(atom *) , molecule *copy, bool (atom::*condition) (T, U), T t, U u );
  template <typename res, typename T, typename U, typename V> void ActOnCopyWithEachAtomIfTrue( res (molecule::*f)(atom *) , molecule *copy, bool (atom::*condition) (T, U, V), T t, U u, V v );

  // templates for allowing global manipulation of an array with one entry per atom
  void SetIndexedArrayForEachAtomTo ( atom **array, int TesselPoint::* index);
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int TesselPoint::* index, void (*Setor)(T *, T *));
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int TesselPoint::* index, void (*Setor)(T *, T *), T t);
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int TesselPoint::* index, void (*Setor)(T *, T *), T *t);
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int element::*index, void (*Setor)(T *, T *));
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int element::*index, void (*Setor)(T *, T *), T t);
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int element::*index, void (*Setor)(T *, T *), T *t);
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int TesselPoint::* index, T (atom::*Setor)(Vector &), Vector atom::*value);
  template <typename T> void SetIndexedArrayForEachAtomTo ( T *array, int TesselPoint::*index, T (atom::*Setor)(Vector &), Vector &vect );

  // templates for allowing global manipulation of each atom by entries in an array
  template <typename T> void SetAtomValueToIndexedArray ( T *array, int TesselPoint::*index, T atom::*value );
  template <typename T> void SetAtomValueToIndexedArray ( T *array, int element::*index, T atom::*value );

  /// remove atoms from molecule.
  bool AddAtom(atom *pointer);
  bool RemoveAtom(atom *pointer);
  bool UnlinkAtom(atom *pointer);
  bool CleanupMolecule();

  /// Add/remove atoms to/from molecule.
  atom * AddCopyAtom(atom *pointer);
  bool AddXYZFile(string filename);
  bool AddHydrogenReplacementAtom(ofstream *out, bond *Bond, atom *BottomOrigin, atom *TopOrigin, atom *TopReplacement, bond **BondList, int NumBond, bool IsAngstroem);
  bond * AddBond(atom *first, atom *second, int degree = 1);
  bool RemoveBond(bond *pointer);
  bool RemoveBonds(atom *BondPartner);

  /// Find atoms.
  atom * FindAtom(int Nr) const;
  atom * AskAtom(string text);

  /// Count and change present atoms' coordination.
  void CountAtoms(ofstream *out);
  void CountElements();
  void CalculateOrbitals(class config &configuration);
  bool CenterInBox(ofstream *out);
  bool BoundInBox(ofstream *out);
  void CenterEdge(ofstream *out, Vector *max);
  void CenterOrigin(ofstream *out);
  void CenterPeriodic(ofstream *out);
  void CenterAtVector(ofstream *out, Vector *newcenter);
  void Translate(const Vector *x);
  void TranslatePeriodically(const Vector *trans);
  void Mirror(const Vector *x);
  void Align(Vector *n);
  void Scale(double **factor);
  void DeterminePeriodicCenter(Vector &center);
  Vector * DetermineCenterOfGravity(ofstream *out);
  Vector * DetermineCenterOfAll(ofstream *out);
  void SetNameFromFilename(const char *filename);
  void SetBoxDimension(Vector *dim);
  void ScanForPeriodicCorrection(ofstream *out);
  bool VerletForceIntegration(ofstream *out, char *file, config &configuration);
  void Thermostats(config &configuration, double ActualTemp, int Thermostat);
  void PrincipalAxisSystem(ofstream *out, bool DoRotate);
  double VolumeOfConvexEnvelope(ofstream *out, bool IsAngstroem);
  Vector* FindEmbeddingHole(ofstream *out, molecule *srcmol);


  double ConstrainedPotential(ofstream *out, struct EvaluatePotential &Params);
  double MinimiseConstrainedPotential(ofstream *out, atom **&permutation, int startstep, int endstep, bool IsAngstroem);
  void EvaluateConstrainedForces(ofstream *out, int startstep, int endstep, atom **PermutationMap, ForceMatrix *Force);
  bool LinearInterpolationBetweenConfiguration(ofstream *out, int startstep, int endstep, const char *prefix, config &configuration, bool MapByIdentity);
	
  bool CheckBounds(const Vector *x) const;
  void GetAlignvector(struct lsq_params * par) const;

  /// Initialising routines in fragmentation
  void CreateAdjacencyList2(ofstream *out, ifstream *output);
  void CreateAdjacencyList(ofstream *out, double bonddistance, bool IsAngstroem);
  void CreateListOfBondsPerAtom(ofstream *out);

  // Graph analysis
  MoleculeLeafClass * DepthFirstSearchAnalysis(ofstream *out, class StackClass<bond *> *&BackEdgeStack);
  void CyclicStructureAnalysis(ofstream *out, class StackClass<bond *> *BackEdgeStack, int *&MinimumRingSize);
  bool PickLocalBackEdges(ofstream *out, atom **ListOfLocalAtoms, class StackClass<bond *> *&ReferenceStack, class StackClass<bond *> *&LocalStack);
  bond * FindNextUnused(atom *vertex);
  void SetNextComponentNumber(atom *vertex, int nr);
  void InitComponentNumbers();
  void OutputComponentNumber(ofstream *out, atom *vertex);
  void ResetAllBondsToUnused();
  void ResetAllAtomNumbers();
  int CountCyclicBonds(ofstream *out);
  bool CheckForConnectedSubgraph(ofstream *out, KeySet *Fragment);
  string GetColor(enum Shading color);

  molecule *CopyMolecule();
  molecule* CopyMoleculeFromSubRegion(Vector offset, double *parallelepiped);

  /// Fragment molecule by two different approaches:
  int FragmentMolecule(ofstream *out, int Order, config *configuration);
  bool CheckOrderAtSite(ofstream *out, bool *AtomMask, Graph *GlobalKeySetList, int Order, int *MinimumRingSize, char *path = NULL);
  bool StoreAdjacencyToFile(ofstream *out, char *path);
  bool CheckAdjacencyFileAgainstMolecule(ofstream *out, char *path, atom **ListOfAtoms);
  bool ParseOrderAtSiteFromFile(ofstream *out, char *path);
  bool StoreOrderAtSiteFile(ofstream *out, char *path);
  bool StoreForcesFile(ofstream *out, MoleculeListClass *BondFragments, char *path, int *SortIndex);
  bool CreateMappingLabelsToConfigSequence(ofstream *out, int *&SortIndex);
  void BreadthFirstSearchAdd(ofstream *out, molecule *Mol, atom **&AddedAtomList, bond **&AddedBondList, atom *Root, bond *Bond, int BondOrder, bool IsAngstroem);
  /// -# BOSSANOVA
  void FragmentBOSSANOVA(ofstream *out, Graph *&FragmentList, KeyStack &RootStack, int *MinimumRingSize);
  int PowerSetGenerator(ofstream *out, int Order, struct UniqueFragments &FragmentSearch, KeySet RestrictedKeySet);
  bool BuildInducedSubgraph(ofstream *out, const molecule *Father);
  molecule * StoreFragmentFromKeySet(ofstream *out, KeySet &Leaflet, bool IsAngstroem);
  void SPFragmentGenerator(ofstream *out, struct UniqueFragments *FragmentSearch, int RootDistance, bond **BondsSet, int SetDimension, int SubOrder);
  int LookForRemovalCandidate(ofstream *&out, KeySet *&Leaf, int *&ShortestPathList);
  int GuesstimateFragmentCount(ofstream *out, int order);

  // Recognize doubly appearing molecules in a list of them
  int * IsEqualToWithinThreshold(ofstream *out, molecule *OtherMolecule, double threshold);
  int * GetFatherSonAtomicMap(ofstream *out, molecule *OtherMolecule);

  // Output routines.
  bool Output(ofstream *out);
  bool OutputTrajectories(ofstream *out);
  void OutputListOfBonds(ofstream *out) const;
  bool OutputXYZ(ofstream *out) const;
  bool OutputTrajectoriesXYZ(ofstream *out);
  bool Checkout(ofstream *out) const;
  bool OutputTemperatureFromTrajectories(ofstream *out, int startstep, int endstep, ofstream *output);

  private:
  int last_atom;      //!< number given to last atom
  atom *InternalPointer;  //!< internal pointer for PointCloud
};

#include "molecule_template.hpp"

/** A list of \a molecule classes.
 */
class MoleculeListClass {
  public:
    MoleculeList ListOfMolecules; //!< List of the contained molecules
    int MaxIndex;

  MoleculeListClass();
  ~MoleculeListClass();

  bool AddHydrogenCorrection(ofstream *out, char *path);
  bool StoreForcesFile(ofstream *out, char *path, int *SortIndex);
  void insert(molecule *mol);
  molecule * ReturnIndex(int index);
  bool OutputConfigForListOfFragments(ofstream *out, config *configuration, int *SortIndex);
  int NumberOfActiveMolecules();
  void Enumerate(ofstream *out);
  void Output(ofstream *out);

  // merging of molecules
  bool SimpleMerge(molecule *mol, molecule *srcmol);
  bool SimpleAdd(molecule *mol, molecule *srcmol);
  bool SimpleMultiMerge(molecule *mol, int *src, int N);
  bool SimpleMultiAdd(molecule *mol, int *src, int N);
  bool ScatterMerge(molecule *mol, int *src, int N);
  bool EmbedMerge(molecule *mol, molecule *srcmol);

  private:
};


/** A leaf for a tree of \a molecule class
 * Wraps molecules in a tree structure
 */
class MoleculeLeafClass {
  public:
    molecule *Leaf;                   //!< molecule of this leaf
    //MoleculeLeafClass *UpLeaf;        //!< Leaf one level up
    //MoleculeLeafClass *DownLeaf;      //!< First leaf one level down
    MoleculeLeafClass *previous;  //!< Previous leaf on this level
    MoleculeLeafClass *next;      //!< Next leaf on this level

  //MoleculeLeafClass(MoleculeLeafClass *Up, MoleculeLeafClass *Previous);
  MoleculeLeafClass(MoleculeLeafClass *PreviousLeaf);
  ~MoleculeLeafClass();

  bool AddLeaf(molecule *ptr, MoleculeLeafClass *Previous);
  bool FillBondStructureFromReference(ofstream *out, molecule *reference, int &FragmentCounter, atom ***&ListOfLocalAtoms, bool FreeList = false);
  bool FillRootStackForSubgraphs(ofstream *out, KeyStack *&RootStack, bool *AtomMask, int &FragmentCounter);
  bool AssignKeySetsToFragment(ofstream *out, molecule *reference, Graph *KeySetList, atom ***&ListOfLocalAtoms, Graph **&FragmentList, int &FragmentCounter, bool FreeList = false);
  bool FillListOfLocalAtoms(ofstream *out, atom ***&ListOfLocalAtoms, const int FragmentCounter, const int GlobalAtomCount, bool &FreeList);
  void TranslateIndicesToGlobalIDs(ofstream *out, Graph **FragmentList, int &FragmentCounter, int &TotalNumberOfKeySets, Graph &TotalGraph);
  int Count() const;
};


#endif /*MOLECULES_HPP_*/

