/*
 * SetValueMap.hpp
 *
 *  Created on: Jul 4, 2012
 *      Author: heber
 */

#ifndef SETVALUEMAP_HPP_
#define SETVALUEMAP_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>

#include "IndexSet.hpp"
#include "SetValue.hpp"

#include "CodePatterns/Assert.hpp"

class SetValueMapTest;

/** This class represents a map from IndexSet to SetValue<T>.
 *
 * Each SetValue knows its IndexSet but we also need the inverse
 * association from IndexSet to SetValue.
 *
 */
template <typename T>
class SetValueMap
{
  //!> grant unit access
  friend class SetValueMapTest;
public:
  /** Either adds a new SetValue<T> to the map or changes the value the present
   * instance.
   *
   * @param ptr associated IndexSet
   * @param _value new value of the SetValue
   */
  void addValue(const IndexSet::ptr& ptr, const T &_value) {
    typename Lookup_t::iterator iter = Lookup.find(ptr);
    if (iter != Lookup.end()) {
      iter->second->setValue(_value);
    } else {
      Lookup.insert( std::make_pair( ptr, typename SetValue<T>::ptr(new SetValue<T>(ptr, _value)) ) );
    }
  }

  /** Getter for the SetValue to a specific IndexSet.
   *
   * @param ptr IndexSet
   * @return SetValue associated with this IndexSet
   */
  typename SetValue<T>::ptr & getValue(const IndexSet::ptr &ptr) {
    ASSERT( isIndexSetPresent(ptr),
        "SetValueMap<T>::getValue() - IndexSet "+toString(*ptr)+" is unknown.");
    typename Lookup_t::iterator iter = Lookup.find(ptr);
    ASSERT( *iter->second->getIndexSet() == *ptr,
        "SetValueMap<T>::getValue() - returned SetValue associated to set "+
        toString(*iter->second->getIndexSet())+"does not match with desired set "
        +toString(*ptr)+".");
    return iter->second;
  }

  /** Getter for the SetValue to a specific IndexSet.
   *
   * @param ptr IndexSet
   * @return SetValue associated with this IndexSet
   */
  const typename SetValue<T>::ptr & getConstValue(const IndexSet::ptr &ptr) const {
    ASSERT( isIndexSetPresent(ptr),
        "SetValueMap<T>::getValue() - IndexSet "+toString(*ptr)+" is unknown.");
    typename Lookup_t::const_iterator iter = Lookup.find(ptr);
    ASSERT( *iter->second->getIndexSet() == *ptr,
        "SetValueMap<T>::getValue() - returned SetValue associated to set "+
        toString(*iter->second->getIndexSet())+"does not match with desired set "
        +toString(*ptr)+".");
    return iter->second;
  }

  bool removeValue(const IndexSet::ptr &ptr) {
    if (isIndexSetPresent(ptr)) {
      Lookup.erase(ptr);
      return true;
    } else {
      return false;
    }
  }

  /** Checks whether a given IndexSet \a ptr is known.
   *
   * @param ptr IndexSet to check
   * @return true - set present, false - else
   */
  bool isIndexSetPresent(const IndexSet::ptr& ptr) const {
    typename Lookup_t::const_iterator iter = Lookup.find(ptr);
    return (iter != Lookup.end());
  }

private:
  //!> typedef for the internal lookup
  typedef std::map< IndexSet::ptr, typename SetValue<T>::ptr, IndexSetContainer::Comparator_t > Lookup_t;
  //!> internal map to provide the lookup
  Lookup_t Lookup;

public:
  //!> typedef to const iterator of internal map such that easy const traversal is possible
  typedef typename Lookup_t::const_iterator const_iterator;
  //!> typedef to const iterator of internal map such that easy const traversal is possible
  typedef typename Lookup_t::const_reverse_iterator const_reverse_iterator;

  /** Getter for first node of internal map.
   *
   * @return Lookup::begin()
   */
  const_iterator begin() const {
    return Lookup.begin();
  }

  /** Getter for last and one node of internal map.
   *
   * @return Lookup::end()
   */
  const_iterator end() const {
    return Lookup.end();
  }

  /** Getter for last node of internal map.
   *
   * @return Lookup::rbegin()
   */
  const_reverse_iterator rbegin() const {
    return Lookup.rbegin();
  }

  /** Getter for one before first node of internal map.
   *
   * @return Lookup::rend()
   */
  const_reverse_iterator rend() const {
    return Lookup.rend();
  }
};

#endif /* SETVALUEMAP_HPP_ */
