/*
 * QtObservedInstanceBoard.hpp
 *
 *  Created on: Oct 17, 2015
 *      Author: heber
 */


#ifndef QTOBSERVEDINSTANCEBOARD_HPP_
#define QTOBSERVEDINSTANCEBOARD_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui/QWidget>

#include "UIElements/Qt4/InstanceBoard/QtObservedAtom.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedBond.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedMolecule.hpp"

#include <map>
#include <boost/bimap.hpp>
#include <boost/function.hpp>

#include "CodePatterns/ObservedValue.hpp"
#include "CodePatterns/Observer/Observer.hpp"

#include "UIElements/Qt4/InstanceBoard/ObservedValue_types.hpp"
#include "UIElements/Qt4/InstanceBoard/ObservedValuesContainer.hpp"
#include "types.hpp"

class GLWorldScene;
class GLMoleculeObject_bond;
class GLMoleculeObject_molecule;

/** The QtObservedInstanceBoard is the central class for communicating instance
 * creation and destruction from MoleCuilder into the QtGui realm. It should be
 * thought of as an interface to allow for safe multi-threaded computing.
 *
 * The idea is as follows: As soon as a molecule is instantiated, all the QtGui
 * needs to access the instance (for displaying it) is wrapped up in a
 * ObservedValue. This ObservedValue separates the lifetime of the molecule object
 * from the information contained therein and thus makes the visual representation
 * independent of the life time. The Observer/Observable signal from the World,
 * i.e. World::MoleculeInserted, is caught (directly within the same thread) by
 * the QtObservedInstanceBoard. Here, we instantiate all ObservedValue's needed
 * for this specific molecule and store them in an internal map. Next, we emit
 * a Qt signal informing the QtGui part about the new molecule.
 * At some later stage, the QtGui will (probably in a different thread)
 * instantiate a GLMoleculeObject_molecule as a visual representation of the
 * molecule. It requests the ObservedValues from the QtObservedInstanceBoard
 * and uses these directly.
 * The QtObservedInstanceBoard also records all subjectKilled() signals from
 * each ObservedValue. Additionally, each class using the ObservedValues
 * additionally informs the QtObservedInstanceBoard when subjectKilled() was
 * called. If subjectKilled() for each ObservedValue of a molecule and from the
 * visual representation have been received, a removal Qt signal is emitted.
 *
 * The same holds for the atom
 */
class QtObservedInstanceBoard : public QWidget, public Observer
{
  Q_OBJECT

public:
  //!> copy bond id type from QtObservedBond
  typedef QtObservedBond::bondId_t bondId_t;

  /** Cstor of QtObservedInstanceBoard.
   *
   * \param _parent Qt parent to automatically destroy when parent is destroyed
   */
  QtObservedInstanceBoard(QWidget * _parent=0);

  /** Dstor of QtObservedInstanceBoard.
   *
   */
  ~QtObservedInstanceBoard();

  // Observer functions
  void update(Observable *publisher);
  void subjectKilled(Observable *publisher);
  void recieveNotification(Observable *publisher, Notification_ptr notification);

  const atomId_t getAtomIdToIndex(ObservedValue_Index_t _id) const;
  const bondId_t getBondIdToIndex(ObservedValue_Index_t _id) const;
  const moleculeId_t getMoleculeIdToIndex(ObservedValue_Index_t _id) const;

  QtObservedAtom::ptr getObservedAtom(const atomId_t _id);
  QtObservedAtom::ptr getObservedAtom(ObservedValue_Index_t _id);
  QtObservedBond::ptr getObservedBond(const bondId_t _id);
  QtObservedBond::ptr getObservedBond(ObservedValue_Index_t _id);
  QtObservedMolecule::ptr getObservedMolecule(const moleculeId_t _id);
  QtObservedMolecule::ptr getObservedMolecule(ObservedValue_Index_t _id);
  const ObservedValue<unsigned int>& getObservedWorldTime();
  const ObservedValue<int>& getObservedMaximumAtomTrajectorySize();
  void markObservedAtomAsConnected(ObservedValue_Index_t _id);
  void markObservedAtomAsDisconnected(ObservedValue_Index_t _id);
  void markObservedAtomForErase(ObservedValue_Index_t _id);
  void markObservedBondAsConnected(ObservedValue_Index_t _id);
  void markObservedBondAsDisconnected(ObservedValue_Index_t _id);
  void markObservedBondForErase(ObservedValue_Index_t _id);
  void markObservedMoleculeAsConnected(ObservedValue_Index_t _id);
  void markObservedMoleculeAsDisconnected(ObservedValue_Index_t _id);
  void markObservedMoleculeForErase(ObservedValue_Index_t _id);

signals:
  void atomInserted(QtObservedAtom::ptr _atom);
  void atomRemoved(ObservedValue_Index_t _atomid);
  void atomIndexChanged(const atomId_t _oldid, const atomId_t _newid);
  void bondInserted(QtObservedBond::ptr _bond);
  void bondRemoved(ObservedValue_Index_t _bondid);
  void bondIndexChanged(const bondId_t _oldid, const bondId_t _newid);
  void moleculeInserted(QtObservedMolecule::ptr _mol);
  void moleculeRemoved(ObservedValue_Index_t _molid);
  void moleculeIndexChanged(const moleculeId_t _oldid, const moleculeId_t _newid);
  void WorldTimeChanged();
  void MaximumAtomTrajectoryChanged();

private:
  friend class GLWorldScene;
  friend class GLMoleculeObject_bond;
  friend class QtObservedAtom;
  friend class QtObservedMolecule;

  //!> indicating whether we are still signedOn to World or not
  bool WorldSignedOn;

  //!> indicating whether we are still signedOn to World or not
  bool WorldTimeSignedOn;
  
  //!> indicating whether we are still signedOn to MaximumAtomTrajectorySize observable
  bool MaximumAtomTrajectorySizeSignedOn;

  typedef std::multiset<Observable *> SignedOn_t;
  //!> map indicating to which atom we are currently signed on
  SignedOn_t AtomSignedOn;
  //!> map indicating to which molecule we are currently signed on
  SignedOn_t MoleculeSignedOn;


  //!> "templated typedef" for an id to index map.
  template <class id>
  struct IdtoIndex_t : boost::bimap<id, ObservedValue_Index_t> {};
  IdtoIndex_t<atomId_t> atomids_lookup;
  IdtoIndex_t<bondId_t> bondids_lookup;
  IdtoIndex_t<moleculeId_t> moleculeids_lookup;

  /** Counts how many atom's ObservedValues got subjectKilled.
   *
   * This is used to give removal signal only when each and every
   * ObservedValue (and the instance itself) has been subjectKilled by the
   * monitored Observable. Only then can we safely remove the instance.
   *
   * \param _id observable that received the subjectKilled()
   */
  void atomcountsubjectKilled(ObservedValue_Index_t _id);

  /** Counts how many bond's ObservedValues got subjectKilled.
   *
   * This is used to give removal signal only when each and every
   * ObservedValue (and the instance itself) has been subjectKilled by the
   * monitored Observable. Only then can we safely remove the instance.
   *
   * \param _bondid id of the bond
   */
  void bondcountsubjectKilled(ObservedValue_Index_t _bondid);

  /** Counts how many molecule's ObservedValues got subjectKilled.
   *
   * This is used to give removal signal only when each and every
   * ObservedValue (and the instance itself) has been subjectKilled by the
   * monitored Observable. Only then can we safely remove the instance.
   *
   * \param _id observable that received the subjectKilled()
   */
  void moleculecountsubjectKilled(ObservedValue_Index_t _id);

  //!> container with all ObservedValues for each atom, associated by id
  ObservedValuesContainer<QtObservedAtom, ObservedValue_Index_t> atomObservedValues;
  //!> container with all ObservedValues for each bond, associated by id pairs
  ObservedValuesContainer<QtObservedBond, ObservedValue_Index_t> bondObservedValues;
  //!> container with all ObservedValues for each molecule, associated by id
  ObservedValuesContainer<QtObservedMolecule, ObservedValue_Index_t> moleculeObservedValues;
  //!> observed value for the world's time
  ObservedValue<unsigned int> worldTime;
  //!> observed value for the largest atom trajectory siye (i.e., current number of world time steps)
  ObservedValue<int> maximumAtomTrajectorySize;
};

#endif /* QTOBSERVEDINSTANCEBOARD_HPP_ */
