/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtHomologyList.cpp
 *
 *  Created on: Jun 24, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtHomologyList.hpp"

#include <iterator>
#include <iostream>

#include <QAbstractItemView>
#include <QtGui/QTreeWidget>
#include<Qt/qsplitter.h>

#include "CodePatterns/MemDebug.hpp"

#include "Fragmentation/Homology/HomologyContainer.hpp"
#include "Potentials/PotentialRegistry.hpp"
#include "World.hpp"

using namespace std;

const int QtHomologyList::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtHomologyList::COLUMNNAMES[QtHomologyList::COLUMNCOUNT]={"Number","Nodes","Edges","Occurrence"};

QtHomologyList::QtHomologyList(QWidget * _parent) :
    QWidget(_parent),
    Observer("QtHomologyList"),
    potentialregistry_enabled(false)
{
  QSplitter *splitter = new QSplitter (Qt::Horizontal, this );
  treewidget = new QTreeWidget (splitter);
  splitter->addWidget(treewidget);
  widget = new QWidget (splitter);
  splitter->addWidget(widget);
  splitter->setStretchFactor(10, 1);

  treewidget->setColumnCount(COLUMNCOUNT);
  treewidget->setSelectionMode(QAbstractItemView::MultiSelection);

  QStringList header;
	for(int i=0; i<COLUMNCOUNT;++i)
	  header << COLUMNNAMES[i];
	treewidget->setHeaderLabels(header);

	dirty = true;
  clearing = false;

  refill();

	HomologyContainer &homologies = World::getInstance().getHomologies();
	homologies.signOn(this);
	PotentialRegistry::getInstance().signOn(this);
	potentialregistry_enabled = true;

  connect(treewidget,SIGNAL(itemSelectionChanged()),this,SLOT(rowSelected()));
  connect(this,SIGNAL(changed()),this,SLOT(update()));
}

QtHomologyList::~QtHomologyList()
{
  HomologyContainer &homologies = World::getInstance().getHomologies();
  homologies.signOff(this);
  if (potentialregistry_enabled)
    PotentialRegistry::getInstance().signOff(this);
}

void QtHomologyList::update(Observable *publisher) {

  dirty = true;

  // force an update from Qt...
  clearing = true;
  treewidget->clear();
  clearing = false;
  //emit changed(); doesn't work!?!
}

void QtHomologyList::refill()
{
  clearing = true;
  const HomologyContainer &homologies = World::getInstance().getHomologies();

  // clear everything
  HomologySelection.clear();
  treewidget->clear();

  size_t count = 0;
  for (HomologyContainer::const_key_iterator iter = homologies.key_begin();
      iter != homologies.key_end(); iter = homologies.getNextKey(iter), ++count) {
    HomologyContainer::range_t occurences = homologies.getHomologousGraphs(*iter);
    const HomologyGraph &graph = occurences.first->first;
    const size_t times = std::distance(occurences.first, occurences.second);

    QTreeWidgetItem *treeItem = new QTreeWidgetItem(treewidget);
    treeItem->setText(NUMBER, QString::number(count));
    {
      std::stringstream output;
      graph.printNodes(output);
      treeItem->setText(NODES, QString(output.str().c_str()));
    }
    {
      std::stringstream output;
      graph.printEdges(output);
      treeItem->setText(EDGES, QString(output.str().c_str()));
    }
    if (times > 0) {
      treeItem->setText(OCCURRENCE, QString::number(times));
    } else {
      treeItem->setText(OCCURRENCE, "none");
      treeItem->setDisabled(true);
    }
    HomologySelection.push_back(treeItem->isSelected());
  }
  dirty = false;
  clearing = false;
}

void QtHomologyList::paintEvent(QPaintEvent * event)
{
  if (dirty)
    refill();
//  treewidget->paintEvent(event);
}

void QtHomologyList::subjectKilled(Observable *publisher)
{
  // as a new instance should always already be present ... just sign on
  if (static_cast<PotentialRegistry *>(publisher) == PotentialRegistry::getPointer()) {
    potentialregistry_enabled = false;
  } else {
    // its HomologyContainer
  }
}


void QtHomologyList::rowSelected()
{
  //std::cout << "rowSelected\n";
  for (int i=0;i<treewidget->topLevelItemCount();i++){
    QTreeWidgetItem *item = treewidget->topLevelItem(i);
    bool newSelection = item->isSelected();
    if (newSelection != HomologySelection[i]){
      HomologySelection[i] = newSelection;
    }
  }
}

