/*
 *    vmg - a versatile multigrid solver
 *    Copyright (C) 2012 Institute for Numerical Simulation, University of Bonn
 *
 *  vmg is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  vmg is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * dirichlet_fas_lr_mpi.cpp
 *
 *  Created on: 25.01.2011
 *      Author: Julian Iseringhausen
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/test/unit_test.hpp>
#include <boost/test/floating_point_comparison.hpp>

#ifdef HAVE_MPI

#include <mpi.h>
#ifdef HAVE_MARMOT
#include <enhancempicalls.h>
#include <sourceinfompicalls.h>
#endif

#include "base/factory.hpp"
#include "base/math.hpp"
#include "base/vector.hpp"
#include "comm/comm_mpi.hpp"
#include "comm/domain_decomposition_mpi.hpp"
#include "level/level_operator_fas.hpp"
#include "level/level_operator.hpp"
#include "samples/discretization_poisson_fv.hpp"
#include "samples/techniques.hpp"
#include "smoother/gsrb.hpp"
#ifdef HAVE_LAPACK
#include "solver/dgesv.hpp"
#endif
#include "solver/givens.hpp"
#include "solver/solver_regular.hpp"
#include "mg.hpp"

#include "interface_sinus.hpp"

using namespace VMG;

const vmg_float sine_factor = static_cast<vmg_float>(2.0 * Math::pi);

struct LibraryDirichletFASLRMPIFixture
{
  LibraryDirichletFASLRMPIFixture()
  {
    Factory& factory = MG::GetFactory();

    Comm *comm = new CommMPI(Boundary(Dirichlet, Dirichlet, Dirichlet), new DomainDecompositionMPI());
    comm->Register("COMM");

    Interface* interface = new VMGInterfaces::InterfaceSinus(sine_factor, comm->BoundaryConditions(), 2, 6, 0.0, 1.0, 2, 1.6);
    interface->Register("INTERFACE");

    Discretization* discretization = new DiscretizationPoissonFV(2);
    discretization->Register("DISCRETIZATION");

    LevelOperator* lop = new LevelOperatorFAS(Stencils::RestrictionFullWeight, Stencils::Injection, Stencils::InterpolationTrilinear);
    lop->Register("LEVEL_OPERATOR");

    Smoother* smoother = new GaussSeidelRB();
    smoother->Register("SMOOTHER");

#ifdef HAVE_LAPACK
    Solver* solver = new DGESV<SolverRegular>();
#else
    Solver* solver = new Givens<SolverRegular>();
#endif

    solver->Register("SOLVER");

    Techniques::SetFullApproximationSchemeDirichlet(interface->MinLevel(), interface->MaxLevel(), 2);

    factory.RegisterObjectStorage("PRESMOOTHSTEPS", 3);
    factory.RegisterObjectStorage("POSTSMOOTHSTEPS", 3);
    factory.RegisterObjectStorage("PRECISION", 1e-10);
    factory.RegisterObjectStorage("MAX_ITERATION", 8);

    MG::PostInit();

    MG::IsInitialized();
  }

  ~LibraryDirichletFASLRMPIFixture()
  {
    MG::Destroy();
  }
};

BOOST_FIXTURE_TEST_CASE(LibraryDirichletFASLRMPITest, LibraryDirichletFASLRMPIFixture)
{
  MG::Solve();

  double res_init = MG::GetFactory().Get("INITIAL_RESIDUAL")->Cast< ObjectStorage<double> >()->Val();
  double res = MG::GetComm()->ComputeResidualNorm(*MG::GetSol(), *MG::GetRhs());

  BOOST_CHECK_SMALL(res/res_init, 1e-10);
}

#endif /* HAVE_MPI */
