/*
 * MpqcParser_Parameters.hpp
 *
 *  Created on: Feb 3, 2011
 *      Author: heber
 */

#ifndef MPQCPARSER_PARAMETERS_HPP_
#define MPQCPARSER_PARAMETERS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <list>
#include <map>
#include <vector>

#include "CodePatterns/Clone.hpp"
#include "CodePatterns/Log.hpp"

#include "Parser/FormatParser_Parameters.hpp"

#include "Parser/Parameters/ContinuousParameter.hpp"

// specialization for bool (we want "yes/no" not "1/0")
template <> const std::string ContinuousValue<bool>::get() const;
template <> void ContinuousValue<bool>::set(const std::string _value);

class MpqcParser;

class MpqcParser_Parameters : public FormatParser_Parameters
{
  // MpqcParser should be friend to access params directly for types.
  friend class MpqcParser;
  // ParserMpqcUnitTest needs to be friend to check types contained in params.
  friend class ParserMpqcUnitTest;

public:
  /** Constructor of MpqcParser_Parameters.
   *
   */
  MpqcParser_Parameters();

  /** Destructor of MpqcParser_Parameters.
   *
   */
  virtual ~MpqcParser_Parameters();

  /** Enumeration of all known Parameters to allow placing them in vectors, maps.
   *
   */
  enum Parameters {
    hessianParam,    //!< HessianParam, whether hessian should be calculated or not
    savestateParam,  //!< savestateParam, whether intermediate/final states (wave function) should be stored
    do_gradientParam,//!< do_gradientParam, whether a gradient should be calculated
    maxiterParam,    //!< maxiterParam, number of maximum iterations for CG
    memoryParam,     //!< memoryParam, maximum amount of memory to use
    stdapproxParam,  //!< stdapproxParam, standard approximation in MBPT2 R12
    nfzcParam,       //!< nfzcParam, nfzc parameter in MBPT2 R12
    basisParam,      //!< basisParam, basis set to use
    aux_basisParam,  //!< aux_basisParam, auxiliary baseis set to use in MBPT2 R12
    integrationParam,//!< integrationParam, integration method to use in MBPT2 R12
    theoryParam,     //!< theoryParam, level of theory to use
    unknownParam};   //!< unknownParam, designates an unknown parameter

  /** Enumeration of all known theories.
   *
   */
  enum Theory {
    CLHF,       //!< Closed Shell Hartree-Fock equations
    CLKS,       //!< Closed Shell Kohn-Sham equations
    MBPT2,      //!< Moeller Plesset Perturbation Theory second order
    MBPT2_R12,  //!< Moeller Plesset Perturbation Theory second order with R12 integral
    unknownTheory //!< designates an unknown theory
  };

  /** Enumeration of all known integration methods
   *
   */
  enum IntegrationMethod {
    IntegralCints,  //!< Integration method Cints in MBPT2 R12?
    unknownIntegration //!< designates unknown integration method
  };

  // enum to string getters
  const std::string getParameter(const enum Parameters param) const;
  void setParameter(const enum Parameters param, const std::string &);
  const std::string &getParameterName(const enum Parameters param) const;
  const std::string &getTheoryName(const enum Theory theory) const;
  const std::string &getIntegrationMethodName(const enum IntegrationMethod integration) const;

private:


  //!> vector with all available theories in same order as enum Theory.
  std::vector<std::string> ValidTheories;

  //!> vector with all available integration methods in same order as enum IntegrationMethod.
  std::vector<std::string> ValidIntegrationMethods;

  bool checkWorldElementsAgainstCurrentBasis() const;

  /** Global initialization in cstor.
   *
   */
  void Init();

  /** Initializes BasisList.
   *
   */
  void initBasis();

  //!> vector with all parameter names in same order as enum Parameters
  std::vector<std::string> ParamNames;

  //!> typedef for the list of all available basis sets
  typedef std::map<std::string, std::list<std::string> > BasisMapType;

  //!> list of all basis along with their present element parametrization
  BasisMapType BasisList;
};

std::ostream & operator << (std::ostream& ost, const MpqcParser_Parameters &params);

std::istream & operator >> (std::istream& ist, MpqcParser_Parameters &params);

#endif /* MPQCPARSER_PARAMETERS_HPP_ */
