/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MpqcParser_Parameters.cpp
 *
 *  Created on: Feb 3, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <boost/tokenizer.hpp>
#include <string>

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"

#include "MpqcParser.hpp"

#include "MpqcParser_Parameters.hpp"

#include "Parser/Parameters/ContinuousParameter.hpp"
#include "Parser/Parameters/DiscreteParameter.hpp"
#include "Parser/Parameters/StringParameter.hpp"

template <>
const std::string ContinuousValue<bool>::get() const
{
  ASSERT(ValueSet,
      "ContinuousValue<bool>::get() - requesting unset value.");
  if (value)
    return std::string("yes");
  else
    return std::string("no");
}

template <>
void ContinuousValue<bool>::set(const std::string _value)
{
  if (_value == std::string("yes")) {
    setValue(true);
  } else if (_value == std::string("no")) {
    setValue(false);
  } else {
    ASSERT(0,
        "void ContinuousValue<bool>::set() - value "+_value+" is neither yes or no.");
  }
}


MpqcParser_Parameters::MpqcParser_Parameters()
{
  Init();
}

void MpqcParser_Parameters::Init()
{
  // add all known basis
  initBasis();

  // add all parameter names
  {
    ParamNames.clear();
    ParamNames.resize(unknownParam);
    ParamNames[hessianParam] = "Hessian";
    ParamNames[savestateParam] = "savestate";
    ParamNames[do_gradientParam] = "do_gradient";
    ParamNames[maxiterParam] = "maxiter";
    ParamNames[memoryParam] = "memory";
    ParamNames[stdapproxParam] = "stdapprox";
    ParamNames[nfzcParam] = "nfzc";
    ParamNames[basisParam] = "basis";
    ParamNames[aux_basisParam] = "aux_basis";
    ParamNames[integrationParam] = "integration";
    ParamNames[theoryParam] = "theory";
  }

  // create theory parameter
  {
    ValidTheories.clear();
    ValidTheories.resize(unknownTheory);
    ValidTheories[CLHF]="CLHF";
    ValidTheories[CLKS]="CLKS";
    ValidTheories[MBPT2]="MBPT2";
    ValidTheories[MBPT2_R12]="MBPT2_R12";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[theoryParam],
            ValidTheories,
            ValidTheories[MBPT2]));
  }
  //InvertMap<TheoryNamesType,TheoryLookupType>(TheoryNames,TheoryLookup);

  // create integration parameter
  {
    ValidIntegrationMethods.clear();
    ValidIntegrationMethods.resize(unknownIntegration);
    ValidIntegrationMethods[IntegralCints] = "IntegralCints";
    appendParameter(
        new DiscreteParameter<std::string>(
            ParamNames[integrationParam],
            ValidIntegrationMethods,
            ValidIntegrationMethods[IntegralCints]));
  }

  // add all continuous parameters
  {
    appendParameter(new ContinuousParameter<bool>(ParamNames[hessianParam], false));
    appendParameter(new ContinuousParameter<bool>(ParamNames[savestateParam], false));
    appendParameter(new ContinuousParameter<bool>(ParamNames[do_gradientParam], true));
    appendParameter(new ContinuousParameter<int>(ParamNames[maxiterParam], 1000));
    appendParameter(new ContinuousParameter<int>(ParamNames[memoryParam], 16000000));
    appendParameter(new StringParameter(ParamNames[stdapproxParam], "A'"));
    appendParameter(new ContinuousParameter<int>(ParamNames[nfzcParam], 1));
    appendParameter(new StringParameter(ParamNames[basisParam], "3-21G"));
    appendParameter(new StringParameter(ParamNames[aux_basisParam], "aug-cc-pVDZ"));
  }
}

MpqcParser_Parameters::~MpqcParser_Parameters()
{}

/** Getter for a specific Parameter.
 *
 * @param param index among enum Parameters
 * @return value of the desired Parameters
 */
const std::string MpqcParser_Parameters::getParameter(const enum Parameters param) const
{
  return FormatParser_Parameters::getParameter(ParamNames[param])->get();
}

/** Setter for a specific Parameter.
 *
 * @param param index among enum Parameters
 * @param _value value to set desired Parameter to
 */
void MpqcParser_Parameters::setParameter(const enum Parameters param, const std::string &_value)
{
  const std::string &name = getParameterName(param);
  FormatParser_Parameters::getParameter(name)->set(_value);
}

/** Getter for name of a specific Parameter.
 *
 * @param param index among enum Parameters
 * @return name of the desired Parameter
 */
const std::string &MpqcParser_Parameters::getParameterName(const enum Parameters param) const
{
  return ParamNames[param];
}

/** Getter for name of a specific Parameter.
 *
 * @param param index among enum Theory
 * @return name of the desired Theory
 */
const std::string &MpqcParser_Parameters::getTheoryName(const enum Theory theory) const
{
  return ValidTheories[theory];
}

/** Getter for the name of specific of IntegrationMethod.
 *
 * @param param index among enum IntegrationMethod
 * @return value of the desired IntegrationMethod
 */
const std::string &MpqcParser_Parameters::getIntegrationMethodName(const enum IntegrationMethod integration) const
{
  return ValidIntegrationMethods[integration];
}


/** Output operator for the contents of MpqcParser_Parameters::params.
 *
 * @param ost output stream
 * @param params reference to MpqcParser_Parameters containing params.
 * @return reference to output stream for concatenation
 */
std::ostream & operator << (std::ostream& ost, const MpqcParser_Parameters &params)
{
  // this is ugly, but with boost::any to safeguard const-ness is plain impossible
  std::ostringstream output;
  for (size_t param = (enum MpqcParser_Parameters::Parameters)0;
      param < (size_t)MpqcParser_Parameters::unknownParam; ++param)
    output << params.getParameterName((enum MpqcParser_Parameters::Parameters)param)
           <<  "=" << params.getParameter((enum MpqcParser_Parameters::Parameters)param) << ";";
  ost << output.str();
  return ost;
}

/** Input operator for a list of parameters to place into \a params.
 *
 * @param ist input stream
 * @param params parameters to parse into
 * @return input stream for concatenation
 */
std::istream & operator >> (std::istream& ist, MpqcParser_Parameters &params)
{
  typedef boost::tokenizer<boost::char_separator<char> >
      tokenizer;
  boost::char_separator<char> semicolonsep(";");
  boost::char_separator<char> equalitysep(" =");
  boost::char_separator<char> ticksep("\"");
  std::string line;
  std::getline( ist, line );
  //DoLog(0) && (Log() << Verbose(0) << "INFO: full line of parameters is '" << line << "'" << std::endl);
  tokenizer tokens(line, semicolonsep);
  ASSERT(tokens.begin() != tokens.end(),
      "operator<< on MpqcParser_Parameters - empty string, need at least ';' in line "+line+"!");
  for (tokenizer::iterator tok_iter = tokens.begin();
        tok_iter != tokens.end(); ++tok_iter) {
    tokenizer paramtokens(*tok_iter, equalitysep);
    if (paramtokens.begin() != paramtokens.end()) {
      tokenizer::iterator tok_paramiter = paramtokens.begin();
      tokenizer::iterator tok_valueiter = tok_paramiter;
      tokenizer::iterator tok_checkiter = ++tok_valueiter;
      ++tok_checkiter;
      // TODO: throw exception instead of ASSERT
      ASSERT(tok_paramiter != paramtokens.end(),
          "operator<< on MpqcParser_Parameters - missing value before ' =' in token "+*tok_iter+"!");
      ASSERT(tok_valueiter != paramtokens.end(),
          "operator<< on MpqcParser_Parameters - missing value after ' =' in token "+*tok_iter+"!");
      ASSERT(tok_checkiter == paramtokens.end(),
          "operator<< on MpqcParser_Parameters - still more tokens after ' =' in token "+*tok_iter+":"
          +*tok_checkiter+"!");
      std::stringstream keystream(*tok_paramiter);
      std::string key;
      keystream >> ws >> key;
      tokenizer ticklesstokens(*tok_valueiter, ticksep);
      ASSERT(ticklesstokens.begin() != ticklesstokens.end(),
          "operator<< on MpqcParser_Parameters - no tokens present after removing ticks in token "+*tok_valueiter+"!");
      std::stringstream valuestream(*(ticklesstokens.begin()));
      DoLog(2) && (Log() << Verbose(2)
          << "INFO: Token pair is " << key << "," << valuestream.str() << std::endl);

      // TODO: throw exception instead of DoeLog()
      ASSERT(params.haveParameter(key),
          "operator >> on MpqcParser_Parameters - unknown parameter name '"
          +key+"' with value "+valuestream.str()+"!");
      if (params.haveParameter(key)) {
        Parameter *instance = params.FormatParser_Parameters::getParameter(key);
        instance->set(valuestream.str());
      }
    } else {
      ist.setstate(std::ios::eofbit);
    }
  }
  return ist;
}

/** Checks whether all elements in the world also have parameters in the basis.
 *
 * @return true - all elements parametrized, false - at least one element is missing.
 */
bool MpqcParser_Parameters::checkWorldElementsAgainstCurrentBasis() const
{
  DoeLog(0) && (eLog() << Verbose(0)
      << "MpqcParser_Parameters::checkWorldElementsAgainstCurrentBasis() - not implemented yet."
      << std::endl);

  return false;
}

