/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FormatParser_Parameters.cpp
 *
 *  Created on: Sep 30, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <string>

#include "Parameters/Parameter.hpp"
#include "FormatParser_Parameters.hpp"

/** Constructor of class FormatParser_Parameters.
 *
 * \note we make sure that storage is always present
 *
 */
FormatParser_Parameters::FormatParser_Parameters() :
  storage(new ParameterStorage)
{}

/** Copy constructor of class FormatParser_Parameters.
 *
 * @param _parameters instance to copy from
 */
FormatParser_Parameters::FormatParser_Parameters(const FormatParser_Parameters &_parameters) :
  storage(new ParameterStorage(*_parameters.storage))
{}


/** Destructor of class FormatParser_Parameters.
 *
 */
FormatParser_Parameters::~FormatParser_Parameters()
{
  delete storage;
}

/** Implements the Parameter::clone() function.
 *
 * @return another instance with an identical copy of each parameter in \a storage
 */
FormatParser_Parameters* FormatParser_Parameters::clone() const
{
  return (new FormatParser_Parameters(*this));
}

/** This makes this instance a clone of \a _instance.
 *
 * \note This is basically the other way round to clone().
 *
 * @param _instance instance to clone from
 */
void FormatParser_Parameters::makeClone(const FormatParser_Parameters &_instance)
{
  // we simply remove storage
  delete storage;
  // and clone the one from _instance
  storage = new ParameterStorage(*_instance.storage);
}

/** Adds a parameter to \a storage.
 *
 * This just eases use, saves some typing and is more clear as to what is done.
 *
 * @param instance parameter to add
 */
void FormatParser_Parameters::appendParameter(Parameter *instance)
{
  storage->registerInstance(instance);
}

/** Checks for presence of a parameter in \a storage.
 *
 * This just eases use, saves some typing and is more clear as to what is done.
 *
 * @return true - parameter by this \a _name is present in storage, false - else
 */
bool FormatParser_Parameters::haveParameter(const std::string &_name) const
{
  return storage->isPresentByName(_name);
}

/** Gets a parameter from \a storage.
 *
 * This just eases use, saves some typing and is more clear as to what is done.
 *
 * @return pointer to instance with this \a _name
 */
Parameter *FormatParser_Parameters::getParameter(const std::string &_name) const
{
  return storage->getByName(_name);
}
