/*
 * Registry.hpp
 *
 *  Based on initial ActionRegistry code by Till Crueger.
 *
 *  The registry pattern is basically just a singleton map, wherein instantiations
 *  of a class can be registered, unregistered and retrieved.
 *
 *  Created on: Jul 28, 2010
 *      Author: heber
 */

#ifndef REGISTRY_HPP_
#define REGISTRY_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <map>
#include <string>

/**
 * This template produces a generic registry pattern.
 *
 * <h1> Registry Howto </h1>
 *
 * The Registry is a class where instances of other classes are stored and be retrieved
 * by a string token when desired. For this purpose a Registry should always be a singleton
 * (i.e. use both this Registry and the Singleton pattern to declare a Registry class). It
 * basically is simply a singleton container of a map, where the pointers to the class
 * instances are stored by a string key and can be retrieved thereby.
 *
 * The available functions are as follows if your class instances to be stored in Registry
 * are of type 'foo':
 *
 * - <code>foo* Registry<foo>::getByName()</code> : returns the instance of a specific
 *    class foo instance as a pointer associated with the given name
 * - <code>bool Registry<foo>::isPresentByName()</code> : returns whether an instance
 *    of class foo is present under the given name.
 * - <code>map<string,foo*>::iterator Registry<foo>::getBeginIter()</code> : returns an
 *    iterator to the beginning of the storage map (STL).
 * - <code>map<string,foo*>::const_iterator Registry<foo>::getBeginIter()</code> : returns a
 *    constant iterator to the beginning of the storage map (STL).
 * - <code>map<string,foo*>::const_iterator Registry<foo>::getEndIter()</code> : returns an
 *    iterator to the one step past the last element of the storage map (STL).
 * - <code>map<string,foo*>::const_iterator Registry<foo>::getEndIter()</code> : returns a
 *    constant iterator to the one step past the last element of the storage map (STL).
 *
 * In order to use this pattern, additionally to the requirements of the Singleton pattern,
 * do this:
 *   -# in the declaration derive your class from Registry<foo>, where foo is the class to be
 *      stored
 *   -# in the definition add CONSTRUCT_REGISTRY(foo) to the code such that the templated
 *      functions get instantiated there (otherwise you'll get undefined reference errors).
 *   -# The type stored in the registry needs to have a std::string getName() function that
 *      returns the name under which the instance is stored.
 *
 */

template <class T>
class Registry
{
public:
  Registry();
  ~Registry();

  typedef typename std::map<const std::string,T*> instance_map;
  typedef typename std::map<const std::string,T*>::iterator                  iterator;
  typedef typename std::map<const std::string,T*>::const_iterator            const_iterator;

  T* getByName(const std::string name) const;
  bool isPresentByName(const std::string name) const;
  virtual void registerInstance(T*);
  virtual void unregisterInstance(T*);
  void cleanup();

  iterator getBeginIter();
  const_iterator getBeginIter() const;
  iterator getEndIter();
  const_iterator getEndIter() const;

private:
  instance_map InstanceMap;
};

template <class T> std::ostream& operator<<(std::ostream& ost, const Registry<T>& m);

#endif /* REGISTRY_HPP_ */
