/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AnalyseFragmentationResultsAction.cpp
 *
 *  Created on: Mar 8, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// and before MemDebug due to placement new
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/foreach.hpp>
#include <boost/mpl/remove.hpp>

#include <vector>
#include <fstream>
#include <iostream>
#include <string>
#include <vector>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#ifdef HAVE_JOBMARKET
#include "JobMarket/types.hpp"
#else
typedef size_t JobId_t;
#endif

#include "Descriptors/AtomIdDescriptor.hpp"
#include "Fragmentation/Summation/Containers/FragmentationChargeDensity.hpp"
#include "Fragmentation/Summation/Containers/FragmentationLongRangeResults.hpp"
#include "Fragmentation/Summation/Containers/FragmentationResultContainer.hpp"
#include "Fragmentation/Summation/Containers/FragmentationShortRangeResults.hpp"
#include "Fragmentation/Summation/Containers/MPQCData.hpp"
#include "Fragmentation/Summation/Containers/MPQCData_printKeyNames.hpp"
#include "Fragmentation/Homology/HomologyContainer.hpp"
#include "Fragmentation/Homology/HomologyGraph.hpp"
#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/Summation/SetValues/Eigenvalues.hpp"
#include "Fragmentation/Summation/SetValues/Fragment.hpp"
#include "Fragmentation/Summation/SetValues/Histogram.hpp"
#include "Fragmentation/Summation/SetValues/IndexedVectors.hpp"
#include "Fragmentation/Summation/IndexSetContainer.hpp"
#include "Fragmentation/Summation/writeTable.hpp"
#ifdef HAVE_VMG
#include "Fragmentation/Summation/Containers/VMGData.hpp"
#include "Fragmentation/Summation/Containers/VMGDataFused.hpp"
#include "Fragmentation/Summation/Containers/VMGDataMap.hpp"
#include "Fragmentation/Summation/Containers/VMGData_printKeyNames.hpp"
#endif
#include "Helpers/defs.hpp"
#include "World.hpp"

#include "Actions/FragmentationAction/AnalyseFragmentationResultsAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "AnalyseFragmentationResultsAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */

void writeToFile(const std::string &filename, const std::string contents)
{
  std::ofstream tablefile(filename.c_str());
  tablefile << contents;
  tablefile.close();
}

/** Print (short range) energy, forces, and timings from received results.
 *
 * @param results summed up results container
 */
void printReceivedShortResults(
    const FragmentationShortRangeResults &results)
{
  // print tables (without eigenvalues, they go extra)
  {
    typedef boost::mpl::remove<
          boost::mpl::remove<MPQCDataEnergyVector_t, MPQCDataFused::energy_eigenvalues>::type,
        MPQCDataFused::energy_eigenhistogram>::type
      MPQCDataEnergyVector_noeigenvalues_t;
    const std::string energyresult =
        writeTable<MPQCDataEnergyMap_t, MPQCDataEnergyVector_noeigenvalues_t >()(
            results.Result_Energy_fused, results.getMaxLevel());
    LOG(2, "DEBUG: Energy table is \n" << energyresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_Energy.dat");
    writeToFile(filename, energyresult);
  }

  {
    typedef boost::mpl::list<
        MPQCDataFused::energy_eigenvalues
    > MPQCDataEigenvalues_t;
    const std::string eigenvalueresult =
        writeTable<MPQCDataEnergyMap_t, MPQCDataEigenvalues_t >()(
            results.Result_Energy_fused, results.getMaxLevel());
    LOG(2, "DEBUG: Eigenvalue table is \n" << eigenvalueresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_Eigenvalues.dat");
    writeToFile(filename, eigenvalueresult);
  }

  {
    typedef boost::mpl::list<
        MPQCDataFused::energy_eigenhistogram
    > MPQCDataEigenhistogram_t;
    const std::string eigenhistogramresult =
        writeTable<MPQCDataEnergyMap_t, MPQCDataEigenhistogram_t >()(
            results.Result_Energy_fused, results.getMaxLevel());
    LOG(2, "DEBUG: Eigenhistogram table is \n" << eigenhistogramresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_Eigenhistogram.dat");
    writeToFile(filename, eigenhistogramresult);
  }

  {
    typedef boost::mpl::list<
        MPQCDataFused::energy_eigenvalues
    > MPQCDataEigenvalues_t;
    const std::string eigenvalueresult =
        writeTable<MPQCDataEnergyMap_t, MPQCDataEigenvalues_t >()(
            results.Result_Energy_fused, results.getMaxLevel());
    LOG(2, "DEBUG: Eigenvalue table is \n" << eigenvalueresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_Eigenvalues.dat");
    writeToFile(filename, eigenvalueresult);
  }

  {
    const std::string forceresult =
        writeTable<MPQCDataForceMap_t, MPQCDataForceVector_t>()(
            results.Result_Force_fused, results.getMaxLevel());
    LOG(2, "DEBUG: Force table is \n" << forceresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_Forces.dat");
    writeToFile(filename, forceresult);
  }
  // we don't want to print grid to a table
  {
    // print times (without flops for now)
    typedef boost::mpl::remove<
        boost::mpl::remove<MPQCDataTimeVector_t, MPQCDataFused::times_total_flops>::type,
        MPQCDataFused::times_gather_flops>::type
        MPQCDataTimeVector_noflops_t;
    const std::string timesresult =
        writeTable<MPQCDataTimeMap_t, MPQCDataTimeVector_noflops_t >()(
            results.Result_Time_fused, results.getMaxLevel());
    LOG(2, "DEBUG: Times table is \n" << timesresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_Times.dat");
    writeToFile(filename, timesresult);
  }
}


/** Print long range energy from received results.
 *
 * @param results summed up results container
 */
void printReceivedFullResults(
    const FragmentationLongRangeResults &results)
{
  // print tables (without eigenvalues, they go extra)

  if (results.Result_LongRange_fused.size() >= (results.getMaxLevel()-2))
  {
    const std::string gridresult =
        writeTable<VMGDataMap_t, VMGDataVector_t >()(
            results.Result_LongRange_fused, results.getMaxLevel(), 2);
    LOG(2, "DEBUG: VMG table is \n" << gridresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_VMGEnergy.dat");
    writeToFile(filename, gridresult);
  }

  if (results.Result_LongRangeIntegrated_fused.size() >= (results.getMaxLevel()-2))
  {
    const std::string gridresult =
        writeTable<VMGDataLongRangeMap_t, VMGDataLongRangeVector_t >()(
            results.Result_LongRangeIntegrated_fused, results.getMaxLevel(), 2);
    LOG(2, "DEBUG: LongRange table is \n" << gridresult);
    std::string filename;
    filename += FRAGMENTPREFIX + std::string("_LongRangeEnergy.dat");
    writeToFile(filename, gridresult);
  }
}

bool appendToHomologyFile(
    const boost::filesystem::path &homology_file,
    const FragmentationShortRangeResults &shortrangeresults,
    const FragmentationLongRangeResults &longrangeresults
    )
{
  /// read homology container (if present)
  HomologyContainer homology_container;
  if (boost::filesystem::exists(homology_file)) {
    std::ifstream returnstream(homology_file.string().c_str());
    if (returnstream.good()) {
      boost::archive::text_iarchive ia(returnstream);
      ia >> homology_container;
    } else {
      ELOG(2, "Failed to parse from " << homology_file.string() << ".");
    }
    returnstream.close();
  } else {
    LOG(2, "Could not open " << homology_file.string()
        << ", creating empty container.");
  }

  /// append all fragments to a HomologyContainer
  HomologyContainer::container_t values;

  // convert KeySetContainer to IndexSetContainer
  IndexSetContainer::ptr ForceContainer(new IndexSetContainer(shortrangeresults.getForceKeySet()));
  const IndexSetContainer::Container_t &Indices = shortrangeresults.getContainer();
  const IndexSetContainer::Container_t &ForceIndices = ForceContainer->getContainer();
  IndexSetContainer::Container_t::const_iterator iter = Indices.begin();
  IndexSetContainer::Container_t::const_iterator forceiter = ForceIndices.begin();

  /// go through all fragments
  for (;iter != Indices.end(); ++iter, ++forceiter) // go through each IndexSet
  {
    /// create new graph entry in HomologyContainer which is (key,value) type
    LOG(1, "INFO: Creating new graph with " << **forceiter << ".");
    HomologyGraph graph(**forceiter);
    LOG(2, "DEBUG: Created graph " << graph << ".");
    const IndexSet::ptr &index = *iter;
    HomologyContainer::value_t value;

    // obtain fragment as key
    std::map<IndexSet::ptr, std::pair< MPQCDataFragmentMap_t, MPQCDataFragmentMap_t> >::const_iterator fragmentiter
        = longrangeresults.Result_perIndexSet_Fragment.find(index);
    ASSERT( fragmentiter != longrangeresults.Result_perIndexSet_Fragment.end(),
        "appendToHomologyFile() - cannot find index "+toString(*index)
        +" in FragmentResults.");
    value.first = boost::fusion::at_key<MPQCDataFused::fragment>(fragmentiter->second.first);

    // obtain energy as value
    std::map<IndexSet::ptr, std::pair< MPQCDataEnergyMap_t, MPQCDataEnergyMap_t> >::const_iterator energyiter
        = shortrangeresults.Result_perIndexSet_Energy.find(index);
    ASSERT( energyiter != shortrangeresults.Result_perIndexSet_Energy.end(),
        "appendToHomologyFile() - cannot find index "+toString(*index)
        +" in FragmentResults.");
    // value.second = boost::fusion::at_key<MPQCDataFused::energy_total>(energyiter->second.first); // values
    value.second = boost::fusion::at_key<MPQCDataFused::energy_total>(energyiter->second.second); // contributions
    values.insert( std::make_pair( graph, value) );
  }
  homology_container.insert(values);

  if (DoLog(2)) {
    LOG(2, "DEBUG: Listing all present atomic ids ...");
    std::stringstream output;
    for (World::AtomIterator iter = World::getInstance().getAtomIter();
        iter != World::getInstance().atomEnd(); ++iter)
      output << (*iter)->getId() << " ";
    LOG(2, "DEBUG: { " << output.str() << "} .");
  }

  // for debugging: print container
  if (DoLog(2)) {
    LOG(2, "DEBUG: Listing all present homologies ...");
    for (HomologyContainer::container_t::const_iterator iter =
        homology_container.begin(); iter != homology_container.end(); ++iter) {
      LOG(2, "DEBUG: graph " << iter->first << " has Fragment "
          << iter->second.first << " and associated energy " << iter->second.second << ".");
    }
  }

  /// store homology container again
  std::ofstream outputstream(homology_file.string().c_str());
  if (outputstream.good()) { // check if opened
    boost::archive::text_oarchive oa(outputstream);
    oa << homology_container;
    if (outputstream.fail()) { // check if correctly written
      LOG(1, "Failed to write to file " << homology_file.string() << ".");
      return false;
    } else
      outputstream.close();
  } else {
    LOG(1, "Failed to open file " << homology_file.string()
        << " for writing.");
    return false;
  }
  return true;
}

Action::state_ptr FragmentationAnalyseFragmentationResultsAction::performCall() {

  // if file is given, parse from file into resultscontainer
  FragmentationResultContainer& container = FragmentationResultContainer::getInstance();
  if (!params.resultsfile.get().empty()) {
    boost::filesystem::path resultsfile = params.resultsfile.get();
    if (boost::filesystem::exists(resultsfile)) {
      LOG(1, "INFO: Parsing results from " << resultsfile.string() << ".");
      std::ifstream returnstream(resultsfile.string().c_str());
      if (returnstream.good()) {
        boost::archive::text_iarchive ia(returnstream);
        ia >> container;
      }
    } else {
      ELOG(1, "Given file" << resultsfile.string() << " does not exist.");
    }
  }

  // get data and keysets from ResultsContainer
  const std::map<JobId_t, MPQCData> &shortrangedata = container.getShortRangeResults();
  const KeySetsContainer &keysets = container.getKeySets();
  const KeySetsContainer &forcekeysets = container.getForceKeySets();
  const bool DoLongrange = container.areFullRangeResultsPresent();
  const bool IsAngstroem = true;

  if (keysets.KeySets.empty()) {
    ELOG(2, "There are no results in the container.");
    return Action::failure;
  }

  FragmentationShortRangeResults shortrangeresults(shortrangedata, keysets, forcekeysets);
  shortrangeresults(shortrangedata);
  printReceivedShortResults(shortrangeresults);

  // adding obtained forces
  if ( World::getInstance().getAllAtoms().size() != 0) {
    const IndexedVectors::indexedvectors_t forces =
        boost::fusion::at_key<MPQCDataFused::forces>(
            shortrangeresults.Result_Force_fused.back()
            ).getVectors();
        ;
    for(IndexedVectors::indexedvectors_t::const_iterator iter = forces.begin();
        iter != forces.end(); ++iter) {
      const IndexedVectors::index_t &index = iter->first;
      const IndexedVectors::vector_t &forcevector = iter->second;
      ASSERT( forcevector.size() == NDIM,
          "printReceivedShortResults() - obtained force vector has incorrect dimension.");
      // note that mpqc calculates a gradient, hence force pointing into opposite direction
      // we have to mind different units here: MPQC has a_o, while we may have angstroem
      Vector ForceVector(-forcevector[0], -forcevector[1], -forcevector[2]);
      if (IsAngstroem)
        for (size_t i=0;i<NDIM;++i)
          ForceVector[i] *= AtomicLengthToAngstroem;
      atom *_atom = World::getInstance().getAtom(AtomById(index));
      if(_atom != NULL)
        _atom->setAtomicForce(_atom->getAtomicForce() + ForceVector);
      else
        ELOG(2, "Could not find atom to id " << index << ".");
    }
  } else {
    LOG(1, "INFO: Full molecule not loaded, hence will not add forces to atoms.");
  }

#ifdef HAVE_VMG
  if (DoLongrange) {
    if ( World::getInstance().getAllAtoms().size() == 0) {
      ELOG(1, "Please load the full molecule into the world before starting this action.");
      return Action::failure;
    }

    FragmentationChargeDensity summedChargeDensity(shortrangedata,keysets);
    const std::vector<SamplingGrid> full_sample = summedChargeDensity.getFullSampledGrid();

    std::map<JobId_t, VMGData> longrangeData = container.getLongRangeResults();
    // remove full solution corresponding to full_sample from map (must be highest ids), has to be treated extra
    std::map<JobId_t, VMGData>::iterator iter = longrangeData.end();
    for (size_t i=0;i<full_sample.size();++i)
      --iter;
    std::map<JobId_t, VMGData>::iterator remove_iter = iter;
    std::vector<VMGData> fullsolutionData;
    for (; iter != longrangeData.end(); ++iter)
      fullsolutionData.push_back(iter->second);
    longrangeData.erase(remove_iter, longrangeData.end());

    // Final phase: sum up and print result
    FragmentationLongRangeResults longrangeresults(
        shortrangedata,longrangeData,keysets, forcekeysets);
    longrangeresults(
        shortrangedata,
        longrangeData,
        fullsolutionData,
        full_sample);
    printReceivedFullResults(longrangeresults);

    // append all keysets to homology file
    if (!params.homology_file.get().empty()) {
      const boost::filesystem::path &homology_file = params.homology_file.get();
      if (homology_file.string() != "") {
        LOG(1, "INFO: Appending HomologyGraphs to file " << homology_file.string() << ".");
        if (!appendToHomologyFile(homology_file, shortrangeresults, longrangeresults))
          return Action::failure;
      }
    }
  }
#else
  if (DoLongrange)
    ELOG(2, "File contains long-range information but long-range analysis capability not compiled in.");
#endif

  container.clear();

  return Action::success;
}

Action::state_ptr FragmentationAnalyseFragmentationResultsAction::performUndo(Action::state_ptr _state) {
  return Action::success;
}

Action::state_ptr FragmentationAnalyseFragmentationResultsAction::performRedo(Action::state_ptr _state){
  return Action::success;
}

bool FragmentationAnalyseFragmentationResultsAction::canUndo() {
  return false;
}

bool FragmentationAnalyseFragmentationResultsAction::shouldUndo() {
  return false;
}
/** =========== end of function ====================== */
