/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GLMoleculeObject_molecule.cpp
 *
 *  Created on: Mar 30, 2012
 *      Author: ankele
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GLMoleculeObject_molecule.hpp"

#include <Qt3D/qglscenenode.h>
#include <Qt3D/qglbuilder.h>

#include "UIElements/Views/Qt4/Qt3D/GLMoleculeObject_atom.hpp"

#include "CodePatterns/MemDebug.hpp"

#include <boost/assign.hpp>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/IteratorAdaptors.hpp"
#include "CodePatterns/Log.hpp"

#include "LinearAlgebra/Vector.hpp"
#include "LinkedCell/PointCloudAdaptor.hpp"
#include "LinkedCell/linkedcell.hpp"
#include "Tesselation/tesselation.hpp"
#include "Tesselation/BoundaryLineSet.hpp"
#include "Tesselation/BoundaryTriangleSet.hpp"
#include "Tesselation/CandidateForTesselation.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedInstanceBoard.hpp"
#include "Atom/TesselPoint.hpp"
#include "World.hpp"

using namespace boost::assign;

static Observable::channels_t getAllAtomicChangesChannels()
{
  Observable::channels_t channels;
  channels += molecule::AtomInserted, molecule::AtomRemoved, molecule::AtomMoved;
  return channels;
}

const Observable::channels_t GLMoleculeObject_molecule::HullChannels(getAllAtomicChangesChannels());

static QGLSceneNode *createMoleculeMesh(const QGeometryData &_geo)
{
  // Build a mesh from the geometry.
  QGLBuilder builder;
  builder.addTriangles(_geo);
  QGLSceneNode *mesh = builder.finalizedSceneNode();
  return mesh;
}

GLMoleculeObject_molecule::GLMoleculeObject_molecule(
    QObject *parent,
    QtObservedInstanceBoard &_board,
    QtObservedMolecule::ptr &_ObservedMolecule) :
  GLMoleculeObject((QGLSceneNode *)NULL, parent),
  owner(NULL),
  molref(QtObservedMolecule::getMolecule(_ObservedMolecule->getMolIndex())),
  hoverAtomId(-1),
  board(_board),
  ObservedMolecule(_ObservedMolecule)
{
  init(ObservedMolecule->getMolIndex());
}

GLMoleculeObject_molecule::GLMoleculeObject_molecule(
    QGLSceneNode *mesh[],
    QObject *parent,
    QtObservedInstanceBoard &_board,
    QtObservedMolecule::ptr &_ObservedMolecule) :
  GLMoleculeObject(mesh, parent),
  owner(NULL),
  molref(QtObservedMolecule::getMolecule(_ObservedMolecule->getMolIndex())),
  hoverAtomId(-1),
  board(_board),
  ObservedMolecule(_ObservedMolecule)
{
  init(ObservedMolecule->getMolIndex());
}

void GLMoleculeObject_molecule::init(const moleculeId_t _molid)
{
  setObjectId(_molid);
  setMaterial(getMaterial(1));

  m_selected = const_cast<const World &>(World::getInstance()).isMoleculeSelected(_molid);

  // initially, atoms and bonds should be visible
  m_visible = false;

  connect (this, SIGNAL(hoverChanged(GLMoleculeObject *)), this, SLOT(hoverChangedSignalled(GLMoleculeObject *)));
  connect (this, SIGNAL(hoverChanged(GLMoleculeObject *)), this, SIGNAL(changed()));
  connect (ObservedMolecule.get(), SIGNAL(tesselationhullChanged()), this, SLOT(resetTesselationHull()));
  connect (ObservedMolecule.get(), SIGNAL(boundingboxChanged()), this, SLOT(resetBoundingBox()));
  connect (ObservedMolecule.get(), SIGNAL(indexChanged(const moleculeId_t, const moleculeId_t)),
      this, SLOT(resetIndex(const moleculeId_t, const moleculeId_t)));
  /// these are channeled through GLWorldScene instead to ensure synchronicity
//  connect (ObservedMolecule.get(), SIGNAL(atomInserted(QtObservedAtom::ptr)),
//      this, SLOT(atomInserted(QtObservedAtom::ptr)) );
//  connect (ObservedMolecule.get(), SIGNAL(atomRemoved(const atomId_t)),
//      this, SLOT(atomRemoved(const atomId_t)) );
  connect (ObservedMolecule.get(), SIGNAL(selectedChanged()), this, SLOT(resetSelected()));

  connect( this, SIGNAL(clicked()), this, SLOT(wasClicked()));
}

GLMoleculeObject_molecule::~GLMoleculeObject_molecule()
{}

void GLMoleculeObject_molecule::addAtomBonds(
    const bond::ptr &_bond,
    const GLMoleculeObject_bond::SideOfBond _side
    )
{
  bool bond_present = false;
  const BondIds ids = getBondIds(_bond, _side);
  // check whether bond is not present already
  bond_present = BondsinSceneMap.count(ids);
  if (!bond_present)
    bondInserted(ids.first, ids.second, _side);
  else {
    BondsinSceneMap[ids]->resetPosition();
    BondsinSceneMap[ids]->resetWidth();
  }
}

QGeometryData GLMoleculeObject_molecule::updateTesselationHull() const
{
  QGeometryData geo;

  const molecule * const molref =
      QtObservedMolecule::getMolecule(ObservedMolecule->getMolIndex());
  if (molref == NULL) {
    ELOG(1, "Could not createMoleculeMesh, molecule with id "
        << ObservedMolecule->getMolIndex() << " already gone.");
    return geo;
  }
  double minradius = 2.; // TODO: set to maximum bond length value
  LOG(3, "DEBUG: Molecule fits into sphere of radius " << minradius);
  // check minimum bond radius in molecule
  double minlength = std::numeric_limits<double>::max();
  for (molecule::const_iterator iter = molref->begin();
      iter != molref->end(); ++iter) {
    const BondList &ListOfBonds = (*iter)->getListOfBonds();
    for (BondList::const_iterator bonditer = ListOfBonds.begin();
         bonditer != ListOfBonds.end(); ++bonditer) {
      const double bond_distance = (*bonditer)->GetDistance();
      minlength = std::min(bond_distance, minlength);
    }
  }
  minradius = std::max( std::max(minradius, minlength), 1.);

  // we need at least three points for tesselation
  if (getPresentAtoms().size() >= 3) {
    // Tesselate the points.
    Tesselation T;
    PointCloudAdaptor<molecule> cloud(
        const_cast<molecule *>(molref),
        ObservedMolecule->getMolName());
    T(cloud, minradius);

    // Fill the points into a Qt geometry.
    LinkedCell_deprecated LinkedList(cloud, minradius);
    std::map<int, int> indices;
    std::map<int, Vector> normals;
    int index = 0;
    for (PointMap::const_iterator piter = T.PointsOnBoundary.begin();
        piter != T.PointsOnBoundary.end(); ++piter) {
      const Vector &point = piter->second->getPosition();
      // add data to the primitive
      geo.appendVertex(QVector3D(point[0], point[1], point[2]));
      Vector normalvector;
      for (LineMap::const_iterator lineiter = piter->second->lines.begin();
          lineiter != piter->second->lines.end(); ++lineiter)
        for (TriangleMap::const_iterator triangleiter = lineiter->second->triangles.begin();
            triangleiter != lineiter->second->triangles.end(); ++triangleiter)
          normalvector +=
              triangleiter->second->NormalVector;
      normalvector.Normalize();
      geo.appendNormal(QVector3D(normalvector[0], normalvector[1], normalvector[2]));
      geo.appendColor(QColor(1, 1, 1, 1));
      geo.appendTexCoord(QVector2D(0, 0));
      indices.insert( std::make_pair( piter->second->getNr(), index++));
    }

    // Fill the tesselated triangles into the geometry.
    for (TriangleMap::const_iterator runner = T.TrianglesOnBoundary.begin();
        runner != T.TrianglesOnBoundary.end(); runner++) {
      int v[3];
      for (size_t i=0; i<3; ++i)
        v[i] = runner->second->endpoints[i]->getNr();

      // Sort the vertices so the triangle is clockwise (relative to the normal vector).
      Vector cross = T.PointsOnBoundary[v[1]]->getPosition() - T.PointsOnBoundary[v[0]]->getPosition();
      cross.VectorProduct(T.PointsOnBoundary[v[2]]->getPosition() - T.PointsOnBoundary[v[0]]->getPosition());
      if (cross.ScalarProduct(runner->second->NormalVector) > 0)
        geo.appendIndices(indices[v[0]], indices[v[1]], indices[v[2]]);
      else
        geo.appendIndices(indices[v[0]], indices[v[2]], indices[v[1]]);
    }
  }

  return geo;
}

void GLMoleculeObject_molecule::resetTesselationHull()
{
  if (owner != NULL) {
    TesselationHull = updateTesselationHull();
    updateMesh(createMoleculeMesh(TesselationHull));
  }
}

void GLMoleculeObject_molecule::resetBoundingBox()
{
  molecule::BoundingBoxInfo info = ObservedMolecule->getBoundingBox();
  setPosition(QVector3D(info.position[0], info.position[1], info.position[2]));
  setScale(info.radius + 0.3); // getBoundingSphere() only sees atoms as points, so make the box a bit bigger
}

void GLMoleculeObject_molecule::resetIndex(const moleculeId_t, const moleculeId_t)
{
  const atomId_t newId = ObservedMolecule->getMolIndex();
  const size_t oldId = objectId();
  ASSERT( newId != oldId,
      "GLMoleculeObject_molecule::resetIndex() - index "+toString(newId)+" did not change.");
  LOG(4, "INFO: GLMoleculeObject_molecule: new index is "+toString(newId)+".");
  setObjectId(newId);

  emit indexChanged(this, oldId, newId);
}

void GLMoleculeObject_molecule::resetSelected()
{
  const bool new_selected = ObservedMolecule->getMolSelected();
  m_selected = new_selected;

  emit changed();
}

void GLMoleculeObject_molecule::initialize(QGLView *view, QGLPainter *painter)
{
  // Initialize all of the mesh objects that we have as children.
  if (m_visible) {
    GLMoleculeObject::initialize(view, painter);
  } else {
   foreach (QObject *obj, children()) {
     GLMoleculeObject *meshobj = qobject_cast<GLMoleculeObject *>(obj);
       if (meshobj)
         meshobj->initialize(view, painter);
   }
  }
}

void GLMoleculeObject_molecule::draw(QGLPainter *painter, const QVector4D &cameraPlane)
{
  // draw either molecule's mesh or all atoms and bonds
  if (m_visible) {
    resetTesselationHull();

    painter->modelViewMatrix().push();

    // Apply the material and effect to the painter.
    QGLMaterial *material;
    if (m_hovering)
        material = m_hoverMaterial;
    else if (m_selected)
        material = m_selectionMaterial;
    else
        material = m_material;

    ASSERT(material, "GLMoleculeObject::draw: chosen material is NULL");

    painter->setColor(material->diffuseColor());
    painter->setFaceMaterial(QGL::AllFaces, material);
    if (m_effect)
        painter->setUserEffect(m_effect);
    else
        painter->setStandardEffect(QGL::LitMaterial);

    // Mark the object for object picking purposes.
    int prevObjectId = painter->objectPickId();
    if (m_objectId != -1)
        painter->setObjectPickId(m_objectId);

    m_mesh[0]->draw(painter);

    // Turn off the user effect, if present.
    if (m_effect)
        painter->setStandardEffect(QGL::LitMaterial);

    // Revert to the previous object identifier.
    painter->setObjectPickId(prevObjectId);

    // Restore the modelview matrix.
    painter->modelViewMatrix().pop();

    //    GLMoleculeObject::draw(painter, cameraPlane);
  } else {
    // Draw all of the mesh objects that we have as children.
    foreach (QObject *obj, children()) {
      GLMoleculeObject *meshobj = qobject_cast<GLMoleculeObject *>(obj);
      if (meshobj)
        meshobj->draw(painter, cameraPlane);
    }

    // update bounding box prior to selection
    resetBoundingBox();

    painter->modelViewMatrix().push();
    painter->modelViewMatrix().translate(m_position);
    if (m_rotationAngle != 0.0f)
      painter->modelViewMatrix().rotate(m_rotationAngle, m_rotationVector);
    if ((m_scaleX != 1.0f) || (m_scaleY != 1.0f) || (m_scaleZ != 1.0f))
      painter->modelViewMatrix().scale(m_scaleX, m_scaleY, m_scaleZ);

    // Draw a box around the mesh, if selected.
    if (m_selected)
      drawSelectionBox(painter);

    // Restore the modelview matrix.
    painter->modelViewMatrix().pop();
  }
}

/** Adds an atom of this molecule to the scene.
 *
 * @param _atom atom to add
 */
void GLMoleculeObject_molecule::atomInserted(QtObservedAtom::ptr _atom)
{
  const atomId_t atomid = _atom->getAtomIndex();
  LOG(3, "INFO: GLMoleculeObject_molecule: Received signal atomInserted for atom "+toString(atomid)+".");

  if (_atom) {
    GLMoleculeObject_atom *atomObject =
        new GLMoleculeObject_atom(
            GLMoleculeObject::meshSphere,
            this,
            _atom);
    ASSERT( atomObject != NULL,
        "GLMoleculeObject_molecule::atomInserted - could not create atom object for "+toString(atomid));
    AtomNodeMap::iterator iter = AtomsinSceneMap.find(atomid);
    ASSERT(iter == AtomsinSceneMap.end(),
        "GLMoleculeObject_molecule::atomInserted - same atom with id "+toString(atomid)+" added again.");
    AtomsinSceneMap.insert( make_pair(atomid, atomObject) );

    qRegisterMetaType<atomId_t>("atomId_t");
    qRegisterMetaType<GLMoleculeObject_bond::SideOfBond>("GLMoleculeObject_bond::SideOfBond");
    connect (atomObject, SIGNAL(clicked(atomId_t)), this, SIGNAL(atomClicked(atomId_t)));
    connect (atomObject, SIGNAL(changed()), this, SIGNAL(changed()));
    connect (atomObject, SIGNAL(hoverChanged(GLMoleculeObject *)), this, SIGNAL(changed()));
    connect (atomObject, SIGNAL(hoverChanged(GLMoleculeObject *)), this, SLOT(hoverChangedSignalled(GLMoleculeObject *)));
  //  connect (atomObject, SIGNAL(bondsChanged()), this, SLOT(bondInserted(const atomId_t, const atomId_t, const GLMoleculeObject_bond::SideOfBond)));
    connect (atomObject, SIGNAL(BondsAdded(const atomId_t, const atomId_t, const GLMoleculeObject_bond::SideOfBond)), this, SLOT(bondInserted(const atomId_t, const atomId_t, const GLMoleculeObject_bond::SideOfBond)));
    connect (atomObject, SIGNAL(BondsRemoved(const atomId_t, const atomId_t)), this, SLOT(bondRemoved(const atomId_t, const atomId_t)));
    connect (atomObject, SIGNAL(indexChanged(GLMoleculeObject_atom*, const atomId_t, const atomId_t)), this, SLOT(changeAtomId(GLMoleculeObject_atom*, const atomId_t, const atomId_t)));

    if (m_objectId  == -1)
      setObjectId(atomid);

    emit changed();
    emit changeOccured();
  }
}

/** Removes an atom of this molecule from the scene.
 *
 * We just the id as the atom might have already been destroyed.
 *
 * @param _id id of atom to remove
 */
void GLMoleculeObject_molecule::atomRemoved(const atomId_t _id)
{
  LOG(3, "INFO: GLMoleculeObject_molecule: Received signal atomRemoved for atom "+toString(_id)+".");
  // bonds are removed by signal coming from ~bond

  if ((unsigned int)m_objectId == _id)
    setObjectId(-1);

  // remove atoms
  AtomNodeMap::iterator iter = AtomsinSceneMap.find(_id);
  ASSERT(iter != AtomsinSceneMap.end(),
      "GLMoleculeObject_molecule::atomRemoved() - atom "+toString(_id)+" not on display.");
  GLMoleculeObject_atom *atomObject = iter->second;
  AtomsinSceneMap.erase(iter);
  atomObject->disconnect();
  delete atomObject;

  if (AtomsinSceneMap.empty())
    emit moleculeEmptied(ObservedMolecule);

  emit changed();
  emit changeOccured();
}

void GLMoleculeObject_molecule::hoverChangedSignalled(GLMoleculeObject *ob)
{
  // Find the atom, ob corresponds to.
  hoverAtomId = -1;
  GLMoleculeObject_atom *atomObject = dynamic_cast<GLMoleculeObject_atom *>(ob);
  if (atomObject){
    for (AtomNodeMap::iterator iter = AtomsinSceneMap.begin();iter != AtomsinSceneMap.end(); ++ iter){
      if (iter->second == atomObject)
        hoverAtomId = iter->first;
    }

    // Propagate signal.
    emit hoverChanged(hoverAtomId);
  } else {
    // Find the atom, ob corresponds to.
    GLMoleculeObject_molecule *moleculeObject = dynamic_cast<GLMoleculeObject_molecule *>(ob);
    if (moleculeObject == this){
      // Propagate signal.
      emit hoverChanged(ObservedMolecule->getMolIndex(), 0);
    }
  }
}


/** Helper function to get bond ids in the correct order for BondNodeMap.
 *
 * \return pair of ids in correct order.
 */
GLMoleculeObject_molecule::BondIds GLMoleculeObject_molecule::getBondIds(
    const bond::ptr _bond,
    const enum GLMoleculeObject_bond::SideOfBond _side)
{
  BondIds ids;
  switch (_side) {
    case GLMoleculeObject_bond::left:
      ids = std::make_pair(_bond->leftatom->getId(), _bond->rightatom->getId());
      break;
    case GLMoleculeObject_bond::right:
      ids = std::make_pair(_bond->rightatom->getId(), _bond->leftatom->getId());
      break;
  }
  return ids;
}

/** Adds a bond to the scene.
 *
 * @param _bond bond to add
 * @param side which side of the bond (left or right)
 */
void GLMoleculeObject_molecule::bondInserted(
    const atomId_t _left, const atomId_t _right,
    const enum GLMoleculeObject_bond::SideOfBond _side)
{
  LOG(3, "INFO: GLWorldScene::bondInserted() - Adding bond "+toString(_left)
      +toString(_right)+".");
  //LOG(4, "INFO: Currently present bonds " << BondsinSceneMap << ".");

  const BondIds ids( std::make_pair(_left, _right) );
  BondNodeMap::iterator iter = BondsinSceneMap.find(ids);
  if (iter == BondsinSceneMap.end()) {
    GLMoleculeObject_bond * bondObject =
        new GLMoleculeObject_bond(GLMoleculeObject::meshCylinder, this, ids, _side);
    connect (
        bondObject, SIGNAL(BondRemoved(const atomId_t, const atomId_t)),
        this, SLOT(bondRemoved(const atomId_t, const atomId_t)));
    connect (bondObject, SIGNAL(changed()), this, SIGNAL(changed()));
    BondsinSceneMap.insert( make_pair(ids, bondObject) );
  //    BondIdsinSceneMap.insert( Leftids );
  } else {
    iter->second->resetPosition();
    iter->second->resetWidth();
  }
  emit changed();
  emit changeOccured();
}

/** Removes a bond from the scene.
 *
 * @param _bond bond to remove
 */
void GLMoleculeObject_molecule::bondRemoved(const atomId_t leftnr, const atomId_t rightnr)
{
  LOG(3, "INFO: GLWorldScene::bondRemoved() - Removing bond between "+toString(leftnr)+" and "+toString(rightnr)+".");
  {
    // left bond
    const BondIds Leftids( make_pair(leftnr, rightnr) );
    BondNodeMap::iterator leftiter = BondsinSceneMap.find( Leftids );
    ASSERT(leftiter != BondsinSceneMap.end(),
        "GLWorldScene::bondRemoved() - bond "+toString(leftnr)+"-"
        +toString(rightnr)+" not on display.");
    GLMoleculeObject_bond *bondObject = leftiter->second;
    bondObject->disconnect();
    BondsinSceneMap.erase(leftiter);
    delete bondObject; // is done by signal from bond itself
    //LOG(4, "INFO: Still present bonds " << BondsinSceneMap << ".");
  }

  emit changed();
  emit changeOccured();
}

void GLMoleculeObject_molecule::setVisible(bool value)
{
  // first update the mesh if we are going to be visible now
  if (value)
    updateTesselationHull();
  // then emit onward
  GLMoleculeObject::setVisible(value);

  emit changed();
  emit changeOccured();
}

std::ostream &operator<<(std::ostream &ost, const GLMoleculeObject_molecule::BondIds &t)
{
  ost << t.first << "," << t.second;
  return ost;
}

void GLMoleculeObject_molecule::wasClicked()
{
  LOG(4, "INFO: GLMoleculeObject_molecule: atom "
      << ObservedMolecule->getMolIndex() << " has been clicked");
  emit moleculeClicked(ObservedMolecule->getMolIndex());
}

void GLMoleculeObject_molecule::changeAtomId(
    GLMoleculeObject_atom *ob,
    const atomId_t oldId,
    const atomId_t newId)
{
  LOG(3, "INFO: GLMoleculeObject_molecule - change atom id " << oldId << " to " << newId << ".");

  // Remove from map.
  AtomNodeMap::iterator iter = AtomsinSceneMap.find(oldId);
  ASSERT(iter != AtomsinSceneMap.end(),
      "GLMoleculeObject_molecule::changeAtomId() - atom with old id "+toString(oldId)+" not on display.");
  ASSERT(iter->second == ob,
      "GLMoleculeObject_molecule::changeAtomId() - atom with id "
      +toString(oldId)+" does not match with object in AtomsinSceneMap.");
  AtomsinSceneMap.erase(iter);

  // Reinsert with new id.
  {
    AtomNodeMap::iterator iter = AtomsinSceneMap.find(newId);
    ASSERT(iter == AtomsinSceneMap.end(),
          "GLMoleculeObject_molecule::changeAtomId() - atom with new id "+toString(newId)+" already known.");
  }
  AtomsinSceneMap.insert( make_pair(newId, ob) );
}


GLMoleculeObject_molecule::atoms_t GLMoleculeObject_molecule::getPresentAtoms() const
{
  atoms_t returnAtomIds;
  returnAtomIds.insert(
      MapKeyConstIterator<AtomNodeMap::const_iterator>(const_cast<const AtomNodeMap &>(AtomsinSceneMap).begin()),
      MapKeyConstIterator<AtomNodeMap::const_iterator>(const_cast<const AtomNodeMap &>(AtomsinSceneMap).end()));
  return returnAtomIds;
}
