/*
 * GLWorldScene.hpp
 *
 *  This is based on the Qt3D example "teaservice", specifically parts of teaservice.cpp.
 *
 *  Created on: Aug 17, 2011
 *      Author: heber
 */

#ifndef GLWORLDSCENE_HPP_
#define GLWORLDSCENE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <Qt/qobject.h>

#include <iosfwd>

#include <boost/thread/recursive_mutex.hpp>

#include "Bond/bond.hpp"
#include "GLMoleculeObject_bond.hpp"
#include "GLMoleculeObject_molecule.hpp"
#include "types.hpp"

#include "UIElements/Qt4/InstanceBoard/QtObservedAtom.hpp"
#include "UIElements/Qt4/InstanceBoard/QtObservedMolecule.hpp"

class Shape;

class QGLPainter;
class QGLSceneNode;
class QGLView;

class GLMoleculeObject;
class GLMoleculeObject_atom;
class GLMoleculeObject_molecule;
class GLMoleculeObject_shape;

class QtObservedInstanceBoard;

/** This class contains a list of all molecules in the world.
 *
 */
class GLWorldScene : public QObject
{
  Q_OBJECT
public:
  GLWorldScene(
      QtObservedInstanceBoard * _board,
      QObject *parent=0);
  virtual ~GLWorldScene();

//#if !defined(QT_OPENGL_ES_1)
//  PerPixelEffect *lighting;
//#endif

  void changeMaterials(bool perPixel);
  QGLSceneNode* getAtom(size_t);
  QGLSceneNode* getMolecule(size_t);
  QGLSceneNode* getBond(size_t, size_t);

  void initialize(QGLView *view, QGLPainter *painter) const;
  void draw(QGLPainter *painter, const QVector4D &cameraPlane) const;

  enum SelectionModeType{
    SelectAtom,
    SelectMolecule
  };
  void setSelectionMode(SelectionModeType mode);

signals:
  void changed();
//  void updated();
  void changeOccured();
  void pressed();
  void released();
  void clicked();
  void clicked(atomId_t);
  void doubleClicked();
  void hoverChanged(const atomId_t);
  void hoverChanged(const moleculeId_t, int);
  void insertMolecule(QtObservedMolecule::ptr);
  void removeMolecule(QtObservedMolecule*);

private slots:
  void atomClicked(atomId_t no);
  void moleculeClicked(moleculeId_t no);
  void moleculeRemoved(QtObservedMolecule* _mol);
  void moleculeSignOff(const moleculeId_t _id);
  void moleculeEmpty(QtObservedMolecule::ptr _mol);
  void moleculeInserted(QtObservedMolecule::ptr);
  void moleculeSignOn(QtObservedMolecule::ptr);
  void moleculeIndexChanged(const moleculeId_t _oldid, const moleculeId_t _newid);
  void moleculesAtomRemoved(const atomId_t _atomid, QtObservedMolecule * _mol);
  void moleculesAtomInserted(QtObservedAtom::ptr, QtObservedMolecule *_mol);
  void atomRemoved(const atomId_t _atomid);
  void atomInserted(QtObservedAtom::ptr);
  void atomIndexChanged(const atomId_t _oldid, const atomId_t _newid);
  void setSelectionModeAtom();
  void setSelectionModeMolecule();
  void addShape(const std::string &_name);
  void removeShape(const std::string &_name);
//  void update();
  void moleculesVisibilityChanged(const moleculeId_t _id, bool _visible);

public:
  void updateSelectedShapes();

private:

  typedef std::map< moleculeId_t , GLMoleculeObject_molecule* > MoleculeNodeMap;
  typedef std::map< std::string , GLMoleculeObject_shape* > ShapeNodeMap;
  MoleculeNodeMap MoleculesinSceneMap;
  ShapeNodeMap ShapesinSceneMap;
  //!> flag to indicate whether state map is currently worked on
  boost::recursive_mutex MoleculeinSceneMap_mutex;

  //!> enumeration of all possible changes molecule might have missed before instantiation
  enum StateChangeType {
    atomInsertedState,
    atomRemovedState,
    MAX_StateChangeType
  };
  typedef std::map< atomId_t, QtObservedAtom::ptr> QtObservedAtomMap_t;
  typedef std::map< moleculeId_t, QtObservedMolecule::ptr> QtObservedMoleculeMap_t;
  typedef std::multimap< atomId_t, StateChangeType> StateChangeMap_t;
  typedef std::map< moleculeId_t, StateChangeMap_t> MoleculeMissedStateMap_t;
  //!> map of all missed state changes
  MoleculeMissedStateMap_t MoleculeMissedStateMap;
  //!> map to contain all QtObservedAtom that have not been instantiated so far
  QtObservedAtomMap_t QtObservedAtomMap;
  //!> map to contain all QtObservedMolecule that have not been instantiated so far
  QtObservedMoleculeMap_t QtObservedMoleculeMap;

  //!> map to get from QtObservedMolecule to the visual representant
  typedef std::map< QtObservedMolecule* , GLMoleculeObject_molecule* > RemovedMoleculesMap_t;
  //!> set of all currently removed molecules
  RemovedMoleculesMap_t RemovedMolecules;

  //!> flag to indicate whether state map is currently worked on
  boost::recursive_mutex MoleculeMissedStateMap_mutex;

  SelectionModeType selectionMode;

  QtObservedInstanceBoard * board;
};

#endif /* GLWORLDSCENE_HPP_ */
