/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * VectorUnitTest.cpp
 *
 *  Created on: Aug 17, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <limits>

#include "CodePatterns/Log.hpp"
#include "defs.hpp"
#include "Plane.hpp"
#include "RealSpaceMatrix.hpp"
#include "Vector.hpp"
#include "vector_ops.hpp"

#include "VectorUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

#include <iostream>

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( VectorTest );


void VectorTest::setUp()
{
  zero  = Vector(0.,0.,0.);
  unit = Vector(1.,0.,0.);
  otherunit = Vector(0.,1.,0.);
  notunit = Vector(0.,1.,1.);
  two = Vector(2.,1.,0.);
  three = Vector(1,2,3);
};


void VectorTest::tearDown()
{
  logger::purgeInstance();
  errorLogger::purgeInstance();
};

/** UnitTest for Constructors and Vector::IsZero() and Vector::IsOne().
 */
void VectorTest::AssignmentTest()
{
  // test with zero
  zero.at(0) = 0;
  zero.at(1) = 0;
  zero.at(2) = 0;
  double zero_array[3] = {0., 0., 0.};

  CPPUNIT_ASSERT_EQUAL( zero, Vector(0,0,0));
  CPPUNIT_ASSERT_EQUAL( zero, Vector(0.,0.,0.));
  CPPUNIT_ASSERT_EQUAL( zero, Vector(zero_array[0], zero_array[1], zero_array[2]));
  CPPUNIT_ASSERT_EQUAL( zero, Vector(zero_array));

  // test with unit
  unit.at(0) = 1;
  unit.at(1) = 0;
  unit.at(2) = 0;
  double unit_array[3] = {1., 0., 0.};

  CPPUNIT_ASSERT_EQUAL( unit, Vector(1,0,0));
  CPPUNIT_ASSERT_EQUAL( unit, Vector(1.,0.,0.));
  CPPUNIT_ASSERT_EQUAL( unit, Vector(unit_array[0], unit_array[1], unit_array[2]));
  CPPUNIT_ASSERT_EQUAL( unit, Vector(unit_array));

  // test with two
  two.at(0) = 2;
  two.at(1) = 1;
  two.at(2) = 0;
  double two_array[3] = {2., 1., 0.};

  CPPUNIT_ASSERT_EQUAL( two, Vector(2,1,0));
  CPPUNIT_ASSERT_EQUAL( two, Vector(2.,1.,0.));
  CPPUNIT_ASSERT_EQUAL( two, Vector(two_array[0], two_array[1], two_array[2]));
  CPPUNIT_ASSERT_EQUAL( two, Vector(two_array));

  // test with three
  three.at(0) = 1;
  three.at(1) = 2;
  three.at(2) = 3;
  double three_array[3] = {1., 2., 3.};

  CPPUNIT_ASSERT_EQUAL( three, Vector(1,2,3));
  CPPUNIT_ASSERT_EQUAL( three, Vector(1.,2.,3.));
  CPPUNIT_ASSERT_EQUAL( three, Vector(three_array[0], three_array[1], three_array[2]));
  CPPUNIT_ASSERT_EQUAL( three, Vector(three_array));
}

/** UnitTest for Constructors and Vector::IsZero() and Vector::IsOne().
 */
void VectorTest::UnityTest()
{
  // unity and zero tests
  CPPUNIT_ASSERT_EQUAL( true, zero.IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, zero.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, unit.IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, unit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, notunit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, otherunit.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, otherunit.IsZero() );
};

/** UnitTest for Vector::CopyVector(), Vector::AddVector, Vector::SubtractVector() and Vector::Scale()/
 */
void VectorTest::SimpleAlgebraTest()
{
  double factor;
  // copy vector
  fixture = Vector(2.,3.,4.);
  CPPUNIT_ASSERT_EQUAL( Vector(2.,3.,4.), fixture );
  // summation and scaling
  fixture = zero + unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  fixture = zero - unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsZero() );
  fixture = zero + zero;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsZero() );
  fixture = notunit - otherunit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  fixture = unit - otherunit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  fixture = notunit - unit - otherunit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsZero() );
  fixture = 0.98 * unit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  fixture = 1. * unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
  factor = 0.98;
  fixture = factor * unit;
  CPPUNIT_ASSERT_EQUAL( false, fixture.IsOne() );
  factor = 1.;
  fixture = factor * unit;
  CPPUNIT_ASSERT_EQUAL( true, fixture.IsOne() );
};


/** UnitTest for operator versions of Vector::CopyVector(), Vector::AddVector, Vector::SubtractVector() and Vector::Scale().
 */
void VectorTest::OperatorAlgebraTest()
{
  // summation and scaling
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (zero-unit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (zero-unit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (zero+zero).IsZero() );
  CPPUNIT_ASSERT_EQUAL( true, (notunit-otherunit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (unit+otherunit).IsOne() );
  CPPUNIT_ASSERT_EQUAL( false, (notunit-unit-otherunit).IsZero() );
  CPPUNIT_ASSERT_EQUAL( false, (unit*0.98).IsOne() );
  CPPUNIT_ASSERT_EQUAL( true, (unit*1.).IsOne() );

  CPPUNIT_ASSERT_EQUAL( unit, (zero+unit) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.,0.,1.), (notunit-otherunit) );
  CPPUNIT_ASSERT_EQUAL( Vector(-1, 0., 1.), (notunit-unit-otherunit) );
};

/** UnitTest for scalar products.
 */
void VectorTest::EuclidianScalarProductTest()
{
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(zero) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(otherunit) );
  CPPUNIT_ASSERT_EQUAL( 0., zero.ScalarProduct(notunit) );
  CPPUNIT_ASSERT_EQUAL( 1., unit.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., otherunit.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., otherunit.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.ScalarProduct(notunit) );
  CPPUNIT_ASSERT_EQUAL( 2., two.ScalarProduct(unit) );
  CPPUNIT_ASSERT_EQUAL( 1., two.ScalarProduct(otherunit) );
  CPPUNIT_ASSERT_EQUAL( 1., two.ScalarProduct(notunit) );
}

/** UnitTest for norms.
 */
void VectorTest::EuclidianNormTest()
{
  CPPUNIT_ASSERT_EQUAL( 0., zero.Norm() );
  CPPUNIT_ASSERT_EQUAL( 0., zero.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 1., unit.Norm() );
  CPPUNIT_ASSERT_EQUAL( 1., unit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.Norm() );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( 2., notunit.NormSquared() );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), notunit.Norm() );
}

/** UnitTest for distances.
 */
void VectorTest::EuclidianDistancesTest()
{
  CPPUNIT_ASSERT_EQUAL( 1., zero.distance(unit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), otherunit.distance(unit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(2.), zero.distance(notunit) );
  CPPUNIT_ASSERT_EQUAL( 1., otherunit.distance(notunit) );
  CPPUNIT_ASSERT_EQUAL( sqrt(5.), two.distance(notunit) );

  // check operator
  CPPUNIT_ASSERT( !(zero < zero) );
  CPPUNIT_ASSERT( !(unit < otherunit) );
  CPPUNIT_ASSERT( zero < unit );
  CPPUNIT_ASSERT( unit < two );
}

/** UnitTest for angles.
 */
void VectorTest::EuclidianAnglesTest()
{
  CPPUNIT_ASSERT_EQUAL( M_PI, zero.Angle(unit) );
  CPPUNIT_ASSERT_EQUAL( 0., unit.Angle(unit) );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/2. - otherunit.Angle(unit)) <= LINALG_MYEPSILON() );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/2. - unit.Angle(notunit)) <= LINALG_MYEPSILON() );
  CPPUNIT_ASSERT_EQUAL( true, fabs(M_PI/4. - otherunit.Angle(notunit)) <= LINALG_MYEPSILON() );
};

/** UnitTest for projections.
 */
void VectorTest::ProjectionTest()
{
  CPPUNIT_ASSERT_EQUAL( zero, zero.Projection(unit) );
  CPPUNIT_ASSERT_EQUAL( zero, otherunit.Projection(unit) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.4,0.2,0.),  otherunit.Projection(two) );
  CPPUNIT_ASSERT_EQUAL( Vector(0.,1.,0.),  two.Projection(otherunit) );
};

/**
 * Unittest for operation with normals
 */
void VectorTest::NormalsTest(){
  Vector testVector;
  // the zero Vector should produce an error
  CPPUNIT_ASSERT(!testVector.GetOneNormalVector(zero));

  // first one-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(unit));
  CPPUNIT_ASSERT(testVector.ScalarProduct(unit) <= LINALG_MYEPSILON());

  // second one-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(otherunit));
  CPPUNIT_ASSERT(testVector.ScalarProduct(otherunit) <= LINALG_MYEPSILON());

  // first two-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(notunit));
  CPPUNIT_ASSERT(testVector.ScalarProduct(notunit) <= LINALG_MYEPSILON());

  // second two-component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(two));
  CPPUNIT_ASSERT(testVector.ScalarProduct(two) <= LINALG_MYEPSILON());

  // three component system
  CPPUNIT_ASSERT(testVector.GetOneNormalVector(three));
  CPPUNIT_ASSERT(testVector.ScalarProduct(three) <= LINALG_MYEPSILON());
}
