/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ShapeFactory.cpp
 *
 *  Created on: Sep 5, 2012
 *      Author: ankele
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "ShapeFactory.hpp"

#include "BaseShapes.hpp"
#include "ShapeOps.hpp"

#include "CodePatterns/Singleton_impl.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "LinearAlgebra/RealSpaceMatrix.hpp"

const char *ShapeFactory::shapeNames[MAX_ShapeType] = {"nowhere", "everywhere", "sphere", "cube", "polygon", "combined", "cylinder"};

ShapeFactory::ShapeFactory()
{
  // Create map (type -> name).
  for (int i=0; i<MAX_ShapeType; i++)
    shapeNameMap.insert(std::pair<ShapeType, std::string>((ShapeType)i, shapeNames[i]));

  // Create inverse map.
  for (int i=0; i<MAX_ShapeType; i++)
    nameShapeMap.insert(std::pair<std::string, ShapeType>(shapeNames[i], (ShapeType)i));

  // Create baseShapeName list.
  for (int i=0; i<MAX_ShapeType; i++)
    if (strcmp(shapeNames[i], "combined") != 0)
      baseShapeNames.push_back(shapeNames[i]);
}

ShapeType ShapeFactory::getShapeByName(const std::string& name)
{
  NameShapeMap::iterator iter = nameShapeMap.find(name);
  ASSERT(iter != nameShapeMap.end(),
      "ShapeFactory::getShapeByName() - unknown name: "+name+".");
  return iter->second;
}

std::string ShapeFactory::getShapeName(ShapeType type)
{
  ShapeNameMap::iterator iter = shapeNameMap.find(type);
  ASSERT(iter != shapeNameMap.end(),
      "ShapeFactory::getShapeName() - unknown type: "+toString(type)+".");
  return iter->second;
}

bool ShapeFactory::isValidShapeName(const std::string& name)
{
  NameShapeMap::iterator iter = nameShapeMap.find(name);
  return (iter != nameShapeMap.end());
}

const std::vector<std::string> &ShapeFactory::getBaseShapeNames()
{
  return baseShapeNames;
}

ShapeFactory::~ShapeFactory()
{
}

Shape ShapeFactory::produce(ShapeType type, const Vector &translation, const Vector &stretch, double angleX, double angleY, double angleZ) const
{
  for (int i=0; i<NDIM; i++){
    ASSERT(stretch[i] > 0,
        "ShapeFactory::setStretch() - non positive component.");
  }

  // Create the basic shape.
  Shape s = Nowhere();
  if (type == NowhereType){
    s = Nowhere();
  }else if (type == EverywhereType){
    s = Everywhere();
  }else if (type == CuboidType){
    s = Cuboid();
  }else if (type == SphereType){
    s = Sphere();
  }else if (type == CylinderType){
    s = Cylinder();
  }else{
    ASSERT(false,
        "ShapeFactory::produce - unhandled shape type: "+toString(type)+".");
  }

  // Transform (if necessary).
  if (stretch != Vector(1., 1., 1.))
    s = ::stretch(s, stretch);
  if ((angleX != 0) && (angleY != 0) && (angleZ != 0)){
    RealSpaceMatrix rotation;
    rotation.setRotation(angleX, angleY, angleZ);
    s = transform(s, rotation);
  }
  if (!translation.IsZero())
    s = translate(s, translation);

  return s;
}

CONSTRUCT_SINGLETON(ShapeFactory)
