/*
 * AnalysisCorrelationToPointUnitTest.cpp
 *
 *  Created on: Oct 13, 2009
 *      Author: heber
 */

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "analysis_correlation.hpp"
#include "AnalysisCorrelationToPointUnitTest.hpp"

#include "atom.hpp"
#include "boundary.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "linkedcell.hpp"
#include "periodentafel.hpp"
#include "tesselation.hpp"

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AnalysisCorrelationToPointUnitTest );

void AnalysisCorrelationToPointUnitTest::setUp()
{
  atom *Walker = NULL;

  // init private all pointers to zero
  TestMolecule = NULL;
  hydrogen = NULL;
  tafel = NULL;
  pointmap = NULL;
  binmap = NULL;
  point = NULL;

  // construct element
  hydrogen = new element;
  hydrogen->Z = 1;
  strcpy(hydrogen->name, "hydrogen");
  strcpy(hydrogen->symbol, "H");


  // construct periodentafel
  tafel = new periodentafel;
  tafel->AddElement(hydrogen);

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = new molecule(tafel);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 0., 1. );
  TestMolecule->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 1., 1. );
  TestMolecule->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 1., 0. );
  TestMolecule->AddAtom(Walker);
  Walker = new atom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 0., 0. );
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->AtomCount, 4 );

  // init point
  point = new Vector(1.,1.,1.);

  // init maps
  pointmap = CorrelationToPoint( (ofstream *)&cout, (const molecule * const)TestMolecule, (const element * const)hydrogen, (const Vector *)point );
  binmap = NULL;

};


void AnalysisCorrelationToPointUnitTest::tearDown()
{
  if (pointmap != NULL)
    delete(pointmap);
  if (binmap != NULL)
    delete(binmap);

  // remove
  delete(TestMolecule);
  // note that all the atoms are cleaned by TestMolecule
  delete(point);
  delete(tafel);
  // note that element is cleaned by periodentafel
};

void AnalysisCorrelationToPointUnitTest::CorrelationToPointTest()
{
  // do the pair correlation
  CPPUNIT_ASSERT( pointmap != NULL );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, pointmap->size() );

};

void AnalysisCorrelationToPointUnitTest::CorrelationToPointBinNoRangeTest()
{
  BinPairMap::iterator tester;
  // put pair correlation into bins and check with no range
  binmap = BinData( (ofstream *)&cout, pointmap, 0.5, 0., 0. );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, binmap->size() );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 1., tester->first );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );

};

void AnalysisCorrelationToPointUnitTest::CorrelationToPointBinRangeTest()
{
  BinPairMap::iterator tester;
  // ... and check with [0., 2.] range
  binmap = BinData( (ofstream *)&cout, pointmap, 0.5, 0., 2. );
  CPPUNIT_ASSERT_EQUAL( (size_t)5, binmap->size() );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  tester = binmap->find(1.);
  CPPUNIT_ASSERT_EQUAL( 1., tester->first );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );

};

/********************************************** Main routine **************************************/

int main(int argc, char **argv)
{
  // Get the top level suite from the registry
  CppUnit::Test *suite = CppUnit::TestFactoryRegistry::getRegistry().makeTest();

  // Adds the test to the list of test to run
  CppUnit::TextUi::TestRunner runner;
  runner.addTest( suite );

  // Change the default outputter to a compiler error format outputter
  runner.setOutputter( new CppUnit::CompilerOutputter( &runner.result(),
                                                       std::cerr ) );
  // Run the tests.
  bool wasSucessful = runner.run();

  // Return error code 1 if the one of test failed.
  return wasSucessful ? 0 : 1;
};
