/*
 * FragmentController.hpp
 *
 *  Created on: Nov 27, 2011
 *      Author: heber
 */

#ifndef FRAGMENTCONTROLLER_HPP_
#define FRAGMENTCONTROLLER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/asio.hpp>
#include <string>
#include <vector>

#include "Connection.hpp"
#include "FragmentJob.hpp"
#include "FragmentResult.hpp"

/** The FragmentController sends bunches of jobs to a FragmentScheduler,
 *  waits for their calculation and is called when they are done. Then,
 *  he loads the bunch of results from the Scheduler.
 *
 * While the FragmentScheduler and FragmentWorker rather act on their own
 * this is the piece to implant into the user software to allow for
 * communication with the Server/Worker duo to perform the calculation
 * of the fragments on distant computers.
 */
class FragmentController
{
public:
  FragmentController(boost::asio::io_service& io_service, const std::string& host, const std::string& service);
  ~FragmentController();

  /// Handle completion of a calculate operation.
  void handle_connect_calc(const boost::system::error_code& e,
      boost::asio::ip::tcp::resolver::iterator endpoint_iterator);

  /// Handle completion of a CheckResults operation.
  void handle_connect_check(const boost::system::error_code& e,
      boost::asio::ip::tcp::resolver::iterator endpoint_iterator);

  /// Handle completion of a GetResults operation.
  void handle_connect_get(const boost::system::error_code& e,
      boost::asio::ip::tcp::resolver::iterator endpoint_iterator);

  /// Handle completion of a Shutdown operation.
  void handle_connect_shutdown(const boost::system::error_code& e,
      boost::asio::ip::tcp::resolver::iterator endpoint_iterator);

  /// Handle completion of an operation.
  void handle_FinishOperation(const boost::system::error_code& e);

  /// Callback function when bunch of jobs have been sent.
  void handle_SendJobs(const boost::system::error_code& e);

  /// Callback function when doneJobs have been received.
  void handle_ReceiveDoneJobs(const boost::system::error_code& e);

  /// Callback function when results are about to be received.
  void handle_ReceivingResults(const boost::system::error_code& e);

  /// Callback function when results have been received.
  void handle_ReceivedResults(const boost::system::error_code& e);

  /// place number of jobs into this controller
  void addJobs(const std::vector<FragmentJob> &jobs);

  /// prepares the calculation of the results
  void calculateResults();
  
  enum Exitflag_t {
    OkFlag = 0,
    ErrorFlag = 255
  };

  // get the exit flag of the last operations
  size_t getExitflag() const
  {
    return Exitflag;
  }

  /// prepares the receival of the results
  void obtainResults();

  /// prepares the server's shutdown
  void shutdown();

  /// get the results for the current jobs
  std::vector<FragmentResult> getResults();

  /// get the number of finished results for the current jobs
  void checkResults();

  /// Getter for doneJobs
  size_t getDoneJobs() const;

  /// Getter for number of jobs in the queue
  size_t getPresentJobs() const;

private:
  /// internal function to resolve host name and ip address
  boost::asio::ip::tcp::resolver::iterator getEndpointIterator();

  /// internal function to connect to server and send jobs for calculation
  void connect_calc();

  /// internal function to connect to server and check done jobs
  void connect_check();

  /// internal function to connect to server and receive calculated results
  void connect_get();

  /// internal function to connect to server and receive calculated results
  void connect_shutdown();

  /// internal function to disconnect from server
  void disconnect();

private:
  /// The Connection to the server.
  Connection connection_;

  /// host name of server
  const std::string host;

  // service to connect to to
  const std::string service;

  /// bunch of jobs
  std::vector<FragmentJob> jobs;

  /// flag to give on program exit
  enum Exitflag_t Exitflag;

  /// bunch of results
  std::vector<FragmentResult> results;

  /// currently calculated results
  size_t doneJobs;
};

#endif /* FRAGMENTCONTROLLER_HPP_ */
