/*
 * manipulateAtomsTest.cpp
 *
 *  Created on: Feb 18, 2010
 *      Author: crueger
 */

#include "manipulateAtomsTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>
#include <boost/bind.hpp>

#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Actions/ManipulateAtomsProcess.hpp"
#include "Actions/ActionRegistry.hpp"

#include "World.hpp"
#include "atom.hpp"

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( manipulateAtomsTest );

// some stubs
class AtomStub : public atom {
public:
  AtomStub(int _id) :
  atom(),
  id(_id),
  manipulated(false)
  {}

  virtual atomId_t getId(){
    return id;
  }

  virtual void doSomething(){
    manipulated = true;
  }

  bool manipulated;
private:
  atomId_t id;
};

class countObserver : public Observer{
public:
  countObserver() :
    count(0)
    {}
  virtual ~countObserver(){}

  void update(Observable *){
    count++;
  }

  void subjectKilled(Observable *)
  {}

  int count;
};

// set up and tear down
void manipulateAtomsTest::setUp(){
  World::get();
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= new AtomStub(i);
    World::get()->registerAtom(atoms[i]);
  }
}
void manipulateAtomsTest::tearDown(){
  World::destroy();
  ActionRegistry::purgeRegistry();
}

// some helper functions
static bool hasAll(std::vector<atom*> atoms,int min, int max, std::set<int> excluded = std::set<int>()){
  for(int i=min;i<max;++i){
    if(!excluded.count(i)){
      std::vector<atom*>::iterator iter;
      bool res=false;
      for(iter=atoms.begin();iter!=atoms.end();++iter){
        res |= (*iter)->getId() == i;
      }
      if(!res) {
        cout << "Atom " << i << " missing in returned list" << endl;
        return false;
      }
    }
  }
  return true;
}

static bool hasNoDuplicates(std::vector<atom*> atoms){
  std::set<int> found;
  std::vector<atom*>::iterator iter;
  for(iter=atoms.begin();iter!=atoms.end();++iter){
    int id = (*iter)->getId();
    if(found.count(id))
      return false;
    found.insert(id);
  }
  return true;
}

static void operation(atom* _atom){
  AtomStub *atom = dynamic_cast<AtomStub*>(_atom);
  assert(atom);
  atom->doSomething();
}


void manipulateAtomsTest::testManipulateSimple(){
  ManipulateAtomsProcess *proc = World::get()->manipulateAtoms(boost::bind(operation,_1),"FOO",AllAtoms());
  proc->call();
  std::vector<atom*> allAtoms = World::get()->getAllAtoms(AllAtoms());
  std::vector<atom*>::iterator iter;
  for(iter=allAtoms.begin();iter!=allAtoms.end();++iter){
    AtomStub *atom;
    atom = dynamic_cast<AtomStub*>(*iter);
    assert(atom);
    CPPUNIT_ASSERT(atom->manipulated);
  }
}

void manipulateAtomsTest::testManipulateExcluded(){
  ManipulateAtomsProcess *proc = World::get()->manipulateAtoms(boost::bind(operation,_1),"FOO",AllAtoms() && !AtomById(ATOM_COUNT/2));
  proc->call();
  std::vector<atom*> allAtoms = World::get()->getAllAtoms(AllAtoms());
  std::vector<atom*>::iterator iter;
  for(iter=allAtoms.begin();iter!=allAtoms.end();++iter){
    AtomStub *atom;
    atom = dynamic_cast<AtomStub*>(*iter);
    assert(atom);
    if(atom->getId()!=(int)ATOM_COUNT/2)
      CPPUNIT_ASSERT(atom->manipulated);
    else
      CPPUNIT_ASSERT(!atom->manipulated);
  }
}

void manipulateAtomsTest::testObserver(){
  countObserver *obs = new countObserver();
  World::get()->signOn(obs);
  ManipulateAtomsProcess *proc = World::get()->manipulateAtoms(boost::bind(operation,_1),"FOO",AllAtoms() && !AtomById(ATOM_COUNT/2));
  proc->call();

  CPPUNIT_ASSERT_EQUAL(1,obs->count);
  World::get()->signOff(obs);
  delete obs;
}
