/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C) 2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SubsetMapUnitTest.cpp
 *
 *  Created on: Jul 3, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "Fragmentation/Summation/SubsetMap.hpp"

#include "SubsetMapUnitTest.hpp"

#include <boost/assign.hpp>
#include <boost/foreach.hpp>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( SubsetMapTest );


void SubsetMapTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  setVerbosity(4);

  allsets.reserve(5);
  IndexSet tempset;
  tempset += 1;
  allsets += tempset;
  tempset += 2;
  allsets += tempset;
  tempset.clear();
  tempset += 3;
  allsets += tempset;
  tempset += 2;
  allsets += tempset;
  tempset += 4;
  allsets += tempset;
  tempset += 1;
  allsets += tempset;
  CPPUNIT_ASSERT_EQUAL( (size_t)6, allsets.size() );

  SM = NULL;
}


void SubsetMapTest::tearDown()
{
  allsets.clear();
  delete SM;
}

/** UnitTest for getNoPowerSets()
 */
void SubsetMapTest::getNoPowerSetsTest()
{
  CPPUNIT_ASSERT_EQUAL( (size_t)2, SubsetMap::getNoPowerSets(1) );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, SubsetMap::getNoPowerSets(2) );
  CPPUNIT_ASSERT_EQUAL( (size_t)8, SubsetMap::getNoPowerSets(3) );
  CPPUNIT_ASSERT_EQUAL( (size_t)16, SubsetMap::getNoPowerSets(4) );
  CPPUNIT_ASSERT_EQUAL( (size_t)32, SubsetMap::getNoPowerSets(5) );
  CPPUNIT_ASSERT_EQUAL( (size_t)64, SubsetMap::getNoPowerSets(6) );
}


/** UnitTest for getPowerSetIndex()
 */
void SubsetMapTest::getPowerSetIndexTest()
{
  // we just check here all subsets of order 4 "manually"
  IndexSet_ptr ptr(new IndexSet(allsets.back()));
  IndexSet tempset;
  // subset order 4
  CPPUNIT_ASSERT_EQUAL( (size_t)(0), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 1;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 2;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 2), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 2 | 4), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 2 | 4 | 8), SubsetMap::getPowerSetIndex(ptr, tempset));

  // subset order 3
  tempset.clear();
  tempset += 2;
  CPPUNIT_ASSERT_EQUAL( (size_t)(2), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( (size_t)(2 | 4), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(2 | 4 | 8), SubsetMap::getPowerSetIndex(ptr, tempset));

  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( (size_t)(1), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 4), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 4 | 8), SubsetMap::getPowerSetIndex(ptr, tempset));

  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( (size_t)(1), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 2;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 2), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 2 | 8), SubsetMap::getPowerSetIndex(ptr, tempset));

  // subset order 2
  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( (size_t)(1), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 4), SubsetMap::getPowerSetIndex(ptr, tempset));

  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( (size_t)(1), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(1 | 8), SubsetMap::getPowerSetIndex(ptr, tempset));

  tempset.clear();
  tempset += 2;
//  CPPUNIT_ASSERT_EQUAL( (size_t)(2), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( (size_t)(2 | 4), SubsetMap::getPowerSetIndex(ptr, tempset));

  tempset.clear();
  tempset += 2;
//  CPPUNIT_ASSERT_EQUAL( (size_t)(2), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(2 | 8), SubsetMap::getPowerSetIndex(ptr, tempset));

  tempset.clear();
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( (size_t)(4), SubsetMap::getPowerSetIndex(ptr, tempset));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(4 | 8), SubsetMap::getPowerSetIndex(ptr, tempset));

  // subset order 1
  tempset.clear();
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( (size_t)(8), SubsetMap::getPowerSetIndex(ptr, tempset));
}

/** UnitTest for getSubset()
 */
void SubsetMapTest::getSubsetTest()
{
  // we just check here all subsets of order 4 "manually"
  IndexSet_ptr ptr(new IndexSet(allsets.back()));
  IndexSet tempset;
  // subset order 4
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(0) ));
  tempset += 1;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1) ));
  tempset += 2;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 2) ));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 2 | 4) ));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 2 | 4 | 8) ));

  // subset order 3
  tempset.clear();
  tempset += 2;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(2) ));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(2 | 4) ));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(2 | 4 | 8) ));

  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1) ));
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 4) ));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 4 | 8) ));

  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1) ));
  tempset += 2;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 2) ));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 2 | 8) ));

  // subset order 2
  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1) ));
  tempset += 3;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 4) ));

  tempset.clear();
  tempset += 1;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1) ));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(1 | 8) ));

  tempset.clear();
  tempset += 2;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(2) ));
  tempset += 3;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(2 | 4) ));

  tempset.clear();
  tempset += 2;
//  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(2) ));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(2 | 8) ));

  tempset.clear();
  tempset += 3;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(4) ));
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(4 | 8) ));

  // subset order 1
  tempset.clear();
  tempset += 4;
  CPPUNIT_ASSERT_EQUAL( tempset, SubsetMap::getSubset(*ptr, (size_t)(8) ));
}

/** UnitTest for gatherSubset()
 */
void SubsetMapTest::gatherSubsetTest()
{
  /// we create a SubsetMap from an empty container
  IndexSet tempset;
  std::vector<IndexSet> sets;
  sets.push_back(tempset);
  tempset += 1;
  sets.push_back(tempset);
  tempset += 2;
  sets.push_back(tempset);
  tempset.clear();
  tempset += 2;
  sets.push_back(tempset);
  IndexSetContainer container( sets );
  // now we have {}, {1}, {2}, {1,2} in container

  SM = new SubsetMap(container);
  CPPUNIT_ASSERT( SM != NULL );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, SM->Lookup.size() );
  const IndexSetContainer::Container_t &_container = container.getContainer();
  for (IndexSetContainer::Container_t::const_iterator iter = _container.begin();
      iter != _container.end(); ++iter) {
    LOG(1, "INFO: Current set is " << **iter << ".");
    SubsetMap::Lookup_t::const_iterator lookupiter = SM->Lookup.find(*iter);
    CPPUNIT_ASSERT( lookupiter != SM->Lookup.end() );
    CPPUNIT_ASSERT_EQUAL( lookupiter->second->getContainer().size(), SM->getSubsets(*iter)->getContainer().size() );
    CPPUNIT_ASSERT_EQUAL( (size_t)(( 1 << (*iter)->size()) - 1), lookupiter->second->getContainer().size() );
  }
}

/** UnitTest for getMaximumSubsetLevel()
 */
void SubsetMapTest::getMaximumSubsetLevelTest()
{
  // create small container
  {
    IndexSet tempset;
    std::vector<IndexSet> sets;
    sets.push_back(tempset);
    tempset += 1;
    sets.push_back(tempset);
    tempset += 2;
    sets.push_back(tempset);
    tempset.clear();
    tempset += 2;
    sets.push_back(tempset);
    IndexSetContainer container( sets );
    // now we have {}, {1}, {2}, {1,2} in container

    SM = new SubsetMap(container);
    CPPUNIT_ASSERT( SM != NULL );

    CPPUNIT_ASSERT_EQUAL( (size_t)1, SM->getMaximumSubsetLevel() );
    // we are strictly less than super set (last one)
    CPPUNIT_ASSERT( SM->getMaximumSubsetLevel() < (--SM->Lookup.end())->first->size() );
  }

  // create larger container
  {
    IndexSetContainer container( allsets );
    delete SM;
    SM = new SubsetMap(container);
    CPPUNIT_ASSERT( SM != NULL );

    CPPUNIT_ASSERT_EQUAL( (size_t)3, SM->getMaximumSubsetLevel() );
    CPPUNIT_ASSERT( SM->getMaximumSubsetLevel() < (--SM->Lookup.end())->first->size() );
  }
}
