/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * \file FragmentScheduler.cpp
 *
 * This file strongly follows the Serialization example from the boost::asio
 * library (see server.cpp)
 *
 *  Created on: Oct 19, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <iostream>
#include <vector>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include <boost/serialization/vector.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "FragmentJob.hpp"
#include "JobId.hpp"

#include "FragmentScheduler.hpp"

FragmentJob FragmentScheduler::NoJob(std::string("NoJob"), JobId::NoJob);

/** Constructor of class FragmentScheduler.
 *
 * We setup both acceptors to accept connections from workers and Controller.
 *
 * \param io_service io_service of the asynchronous communications
 * \param workerport port to listen for worker connections
 * \param controllerport port to listen for controller connections.
 */
FragmentScheduler::FragmentScheduler(boost::asio::io_service& io_service, unsigned short workerport, unsigned short controllerport) :
  worker_acceptor_(io_service,
      boost::asio::ip::tcp::endpoint(boost::asio::ip::tcp::v4(), workerport)
  ),
  controller_acceptor_(io_service,
      boost::asio::ip::tcp::endpoint(boost::asio::ip::tcp::v4(), controllerport)
  ),
  result(JobId::NoJob),
  Exitflag(OkFlag)
{
  Info info(__FUNCTION__);
  {
    // Start an accept operation for worker connections.
    connection_ptr new_conn(new Connection(worker_acceptor_.get_io_service()));
    worker_acceptor_.async_accept(new_conn->socket(),
      boost::bind(&FragmentScheduler::handle_AcceptWorker, this,
        boost::asio::placeholders::error, new_conn));
  }

  {
    // Start an accept operation for controller connection.
    connection_ptr new_conn(new Connection(controller_acceptor_.get_io_service()));
    controller_acceptor_.async_accept(new_conn->socket(),
      boost::bind(&FragmentScheduler::handle_AcceptController, this,
        boost::asio::placeholders::error, new_conn));
  }
}

/** Handle a new worker connection.
 *
 * We check whether jobs are in the JobsQueue. If present, job is sent.
 *
 * \sa handle_SendJobtoWorker()
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::handle_AcceptWorker(const boost::system::error_code& e, connection_ptr conn)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully accepted a new connection.
    // Check whether there are jobs in the queue
    if (JobsQueue.isJobPresent()) {
      // pop a job and send it to the client.
      FragmentJob job(JobsQueue.popJob());
      // The connection::async_write() function will automatically
      // serialize the data structure for us.
      LOG(1, "INFO: Sending job #" << job.getId() << ".");
      conn->async_write(job,
        boost::bind(&FragmentScheduler::handle_SendJobtoWorker, this,
        boost::asio::placeholders::error, conn));

      // Start an accept operation for a new Connection only when there
      // are still jobs present otherwise we quit.
      connection_ptr new_conn(new Connection(worker_acceptor_.get_io_service()));
      worker_acceptor_.async_accept(new_conn->socket(),
        boost::bind(&FragmentScheduler::handle_AcceptWorker, this,
        boost::asio::placeholders::error, new_conn));
    } else {
      // send the static NoJob
      conn->async_write(NoJob,
        boost::bind(&FragmentScheduler::handle_SendJobtoWorker, this,
        boost::asio::placeholders::error, conn));

      // then there must be no read necesary

      ELOG(2, "There is currently no job present in the queue.");
    }
  }
  else
  {
    // An error occurred. Log it and return. Since we are not starting a new
    // accept operation the io_service will run out of work to do and the
    // server will exit.
    ELOG(0, e.message());
  }
}

/** Callback function when job has been sent.
 *
 * After job has been sent we start async_read() for the result.
 *
 * \sa handle_ReceiveResultFromWorker()
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::handle_SendJobtoWorker(const boost::system::error_code& e, connection_ptr conn)
{
    Info info(__FUNCTION__);
    LOG(1, "INFO: Job sent.");
    // obtain result
    LOG(1, "INFO: Receiving result for a job ...");
    conn->async_read(result,
      boost::bind(&FragmentScheduler::handle_ReceiveResultFromWorker, this,
      boost::asio::placeholders::error, conn));
}

/** Callback function when result has been received.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::handle_ReceiveResultFromWorker(const boost::system::error_code& e, connection_ptr conn)
{
  Info info(__FUNCTION__);
  // nothing to do
  LOG(1, "INFO: Received result for job #" << result.getId() << " ...");
  // and push into queue
  ASSERT(result.getId() != (JobId_t)JobId::NoJob,
      "FragmentScheduler::handle_ReceiveResultFromWorker() - result received has NoJob id.");
  ASSERT(result.getId() != (JobId_t)JobId::IllegalJob,
      "FragmentScheduler::handle_ReceiveResultFromWorker() - result received has IllegalJob id.");
  if ((result.getId() != (JobId_t)JobId::NoJob) && (result.getId() != (JobId_t)JobId::IllegalJob))
    JobsQueue.pushResult(result);
  // erase result
  result = FragmentResult(JobId::NoJob);
}

/** Handle a new controller connection.
 *
 * \sa handle_ReceiveJobs()
 * \sa handle_CheckResultState()
 * \sa handle_SendResults()
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::handle_AcceptController(const boost::system::error_code& e, connection_ptr conn)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    if (jobs.empty()) {
      // The connection::async_write() function will automatically
      // serialize the data structure for us.
      LOG(1, "INFO: Receiving bunch of jobs from a controller ...");
      conn->async_read(jobs,
        boost::bind(&FragmentScheduler::handle_ReceiveJobs, this,
        boost::asio::placeholders::error, conn));
    }
  }
  else
  {
    // An error occurred. Log it and return. Since we are not starting a new
    // accept operation the io_service will run out of work to do and the
    // server will exit.
    Exitflag = ErrorFlag;
    ELOG(0, e.message());
  }
}

/** Controller callback function when job has been sent.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::handle_ReceiveJobs(const boost::system::error_code& e, connection_ptr conn)
{
  Info info(__FUNCTION__);
  // jobs are received, hence place in JobsQueue
  if (!jobs.empty()) {
    LOG(1, "INFO: Pushing " << jobs.size() << " jobs into queue.");
    JobsQueue.pushJobs(jobs);
  }
  // launch new acceptor of queue has been filled/is full
  if (JobsQueue.isJobPresent()) {
    // Start an accept operation for a new Connection.
    connection_ptr new_conn(new Connection(controller_acceptor_.get_io_service()));
    controller_acceptor_.async_accept(new_conn->socket(),
      boost::bind(&FragmentScheduler::handle_AcceptController, this,
      boost::asio::placeholders::error, new_conn));
  } else {
    LOG(1, "INFO: Shutting down controller socket.");
  }

  jobs.clear();
}

