/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * IndexedVectorsUnitTest.cpp
 *
 *  Created on: Jul 29, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "IndexedVectorsUnitTest.hpp"

#include <cmath>
#include <limits>

#include <boost/assign.hpp>

#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace boost::assign;

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( IndexedVectorsTest );


void IndexedVectorsTest::setUp()
{
  // failing asserts should be thrown
  ASSERT_DO(Assert::Throw);

  // create two vector_t
  ones(1.,1.,1.);
  twos(2.,2.,2.);
  threes(3.,3.,3.);
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::value_t::FixedSize, ones.size() );
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::value_t::FixedSize, twos.size() );
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::value_t::FixedSize, threes.size() );

  // create values

  IndexedVectors::values_t values;
  values.push_back(ones);
  values.push_back(twos);
  IndexedVectors::values_t othervalues;
  othervalues.push_back(threes);
  othervalues.push_back(threes);

  // create two indices
  IndexedVectors::indices_t indices;
  indices += 1,2;
  IndexedVectors::indices_t otherindices;
  otherindices += 1,3;

  // create indexed values
  ivectors = new IndexedVectors(indices, values);
  otherivectors = new IndexedVectors(otherindices, othervalues);
}


void IndexedVectorsTest::tearDown()
{
  delete ivectors;
  delete otherivectors;
}

static void checkValueInIndexedVectors(
    const IndexedVectors &_vectors,
    const IndexedVectors::index_t &_index,
    const IndexedVectors::value_t &_compareto
    )
{
  const IndexedVectors::indexedvalues_t &indexedvalues = _vectors.getValues();
  IndexedVectors::indexedvalues_t::const_iterator iter = indexedvalues.find(_index);
  CPPUNIT_ASSERT( iter != indexedvalues.end() );
  CPPUNIT_ASSERT( _compareto == iter->second );
}

/** UnitTest for cstor's
 */
void IndexedVectorsTest::Constructor_Test()
{
  // check whether -1 is dropped
  IndexedVectors::indices_t indices;
  indices += 1,-1,3;
  IndexedVectors::values_t values;
  values.push_back(ones);
  values.push_back(twos);
  values.push_back(threes);
  IndexedVectors testivectors(indices, values);

  CPPUNIT_ASSERT_EQUAL( (size_t)2, testivectors.getValues().size() );
  checkValueInIndexedVectors(testivectors, 1, ones);
  checkValueInIndexedVectors(testivectors, 3, threes);
  CPPUNIT_ASSERT( testivectors.getValues().find(-1) == testivectors.getValues().end() );
}

/** UnitTest for operator+=()
 */
void IndexedVectorsTest::operatorPlusEqual_Test()
{
  // safeguard initial sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)2, ivectors->getValues().size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->getValues().size() );

  // perform operation
  *ivectors += *otherivectors;

  // check new and ole sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)3, ivectors->getValues().size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->getValues().size() );

  // then check result
  IndexedVectors::value_t result;
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::value_t::FixedSize, result.size() );
  for (size_t i=0; i<IndexedVectors::value_t::FixedSize; ++i)
    result[i] = ones[i] + threes[i];
  for (IndexedVectors::indexedvalues_t::const_iterator iter = ivectors->getValues().begin();
      iter != ivectors->getValues().end(); ++iter) {
    CPPUNIT_ASSERT_EQUAL( IndexedVectors::value_t::FixedSize, iter->second.size() );
  }
  checkValueInIndexedVectors(*ivectors, 1, result);
  checkValueInIndexedVectors(*ivectors, 2, twos);
  checkValueInIndexedVectors(*ivectors, 3, threes);
}

/** UnitTest for operator-=()
 */
void IndexedVectorsTest::operatorMinusEqual_Test()
{
  // safeguard initial sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)2, ivectors->getValues().size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->getValues().size() );

  // perform operation
  *ivectors -= *otherivectors;

  // check new and ole sizes
  CPPUNIT_ASSERT_EQUAL( (size_t)3, ivectors->getValues().size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, otherivectors->getValues().size() );

  // then check result
  IndexedVectors::value_t result;
  IndexedVectors::value_t thirdresult;
  CPPUNIT_ASSERT_EQUAL( IndexedVectors::value_t::FixedSize, result.size() );
  for (size_t i=0; i<IndexedVectors::value_t::FixedSize; ++i) {
    result[i] = ones[i] - threes[i];
    thirdresult[i] = -threes[i];
  }
  for (IndexedVectors::indexedvalues_t::const_iterator iter = ivectors->getValues().begin();
      iter != ivectors->getValues().end(); ++iter) {
    CPPUNIT_ASSERT_EQUAL( IndexedVectors::value_t::FixedSize, iter->second.size() );
  }
  checkValueInIndexedVectors(*ivectors, 1, result);
  checkValueInIndexedVectors(*ivectors, 2, twos);
  checkValueInIndexedVectors(*ivectors, 3, thirdresult);
}


/** UnitTest for operator==()
 */
void IndexedVectorsTest::equality_Test()
{
  CPPUNIT_ASSERT( !(*ivectors == *otherivectors) );
  CPPUNIT_ASSERT( *ivectors != *otherivectors );

  // test against empty ivectors
  IndexedVectors emptyivectors;
  CPPUNIT_ASSERT( !(*ivectors == emptyivectors) );
  CPPUNIT_ASSERT( *ivectors != emptyivectors );

  // tests against themselves
  CPPUNIT_ASSERT( *ivectors == *ivectors );
  CPPUNIT_ASSERT( *otherivectors == *otherivectors );
  CPPUNIT_ASSERT( emptyivectors == emptyivectors );

  // check against ZeroInstance
  CPPUNIT_ASSERT( *ivectors != ZeroInstance<IndexedVectors>() );
  CPPUNIT_ASSERT( *otherivectors != ZeroInstance<IndexedVectors>() );
}
