/** \file builder.cpp
 *
 * By stating absolute positions or binding angles and distances atomic positions of a molecule can be constructed.
 * The output is the complete configuration file for PCP for direct use.
 * Features:
 * -# Atomic data is retrieved from a file, if not found requested and stored there for later re-use
 * -# step-by-step construction of the molecule beginning either at a centre of with a certain atom
 *
 */

/*! \mainpage Molecuilder - a molecular set builder
 *
 * This introductory shall briefly make aquainted with the program, helping in installing and a first run.
 *
 * \section about About the Program
 *
 *  Molecuilder is a short program, written in C++, that enables the construction of a coordinate set for the
 *  atoms making up an molecule by the successive statement of binding angles and distances and referencing to
 *  already constructed atoms.
 *
 *  A configuration file may be written that is compatible to the format used by PCP - a parallel Car-Parrinello
 *  molecular dynamics implementation.
 *
 * \section install Installation
 *
 *  Installation should without problems succeed as follows:
 *  -# ./configure (or: mkdir build;mkdir run;cd build; ../configure --bindir=../run)
 *  -# make
 *  -# make install
 *
 *  Further useful commands are
 *  -# make clean uninstall: deletes .o-files and removes executable from the given binary directory\n
 *  -# make doxygen-doc: Creates these html pages out of the documented source
 *
 * \section run Running
 *
 *  The program can be executed by running: ./molecuilder
 *
 *  Note, that it uses a database, called "elements.db", in the executable's directory. If the file is not found,
 *  it is created and any given data on elements of the periodic table will be stored therein and re-used on
 *  later re-execution.
 *
 * \section ref References
 *
 *  For the special configuration file format, see the documentation of pcp.
 *
 */

#include "Helpers/MemDebug.hpp"

#include <boost/bind.hpp>

using namespace std;

#include <cstring>
#include <cstdlib>

#include "analysis_bonds.hpp"
#include "analysis_correlation.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "boundary.hpp"
#include "CommandLineParser.hpp"
#include "config.hpp"
#include "element.hpp"
#include "ellipsoid.hpp"
#include "helpers.hpp"
#include "leastsquaremin.hpp"
#include "linkedcell.hpp"
#include "log.hpp"
#include "memoryusageobserver.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "UIElements/UIFactory.hpp"
#include "UIElements/TextUI/TextUIFactory.hpp"
#include "UIElements/CommandLineUI/CommandLineUIFactory.hpp"
#include "UIElements/MainWindow.hpp"
#include "UIElements/Dialog.hpp"
#include "Menu/ActionMenuItem.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Actions/ActionHistory.hpp"
#include "Actions/MapOfActions.hpp"
#include "Actions/MethodAction.hpp"
#include "Actions/MoleculeAction/ChangeNameAction.hpp"
#include "World.hpp"
#include "version.h"
#include "World.hpp"


/********************************************* Subsubmenu routine ************************************/
#if 0
/** Submenu for adding atoms to the molecule.
 * \param *periode periodentafel
 * \param *molecule molecules with atoms
 */
static void AddAtoms(periodentafel *periode, molecule *mol)
{
  atom *first, *second, *third, *fourth;
  Vector **atoms;
  Vector x,y,z,n;  // coordinates for absolute point in cell volume
  double a,b,c;
  char choice;  // menu choice char
  bool valid;

  cout << Verbose(0) << "===========ADD ATOM============================" << endl;
  cout << Verbose(0) << " a - state absolute coordinates of atom" << endl;
  cout << Verbose(0) << " b - state relative coordinates of atom wrt to reference point" << endl;
  cout << Verbose(0) << " c - state relative coordinates of atom wrt to already placed atom" << endl;
  cout << Verbose(0) << " d - state two atoms, two angles and a distance" << endl;
  cout << Verbose(0) << " e - least square distance position to a set of atoms" << endl;
  cout << Verbose(0) << "all else - go back" << endl;
  cout << Verbose(0) << "===============================================" << endl;
  cout << Verbose(0) << "Note: Specifiy angles in degrees not multiples of Pi!" << endl;
  cout << Verbose(0) << "INPUT: ";
  cin >> choice;

  switch (choice) {
    default:
      DoeLog(2) && (eLog()<< Verbose(2) << "Not a valid choice." << endl);
      break;
      case 'a': // absolute coordinates of atom
        cout << Verbose(0) << "Enter absolute coordinates." << endl;
        first = new atom;
        first->x.AskPosition(World::getInstance().getDomain(), false);
        first->type = periode->AskElement();  // give type
        mol->AddAtom(first);  // add to molecule
        break;

      case 'b': // relative coordinates of atom wrt to reference point
        first = new atom;
        valid = true;
        do {
          if (!valid) DoeLog(2) && (eLog()<< Verbose(2) << "Resulting position out of cell." << endl);
          cout << Verbose(0) << "Enter reference coordinates." << endl;
          x.AskPosition(World::getInstance().getDomain(), true);
          cout << Verbose(0) << "Enter relative coordinates." << endl;
          first->x.AskPosition(World::getInstance().getDomain(), false);
          first->x.AddVector((const Vector *)&x);
          cout << Verbose(0) << "\n";
        } while (!(valid = mol->CheckBounds((const Vector *)&first->x)));
        first->type = periode->AskElement();  // give type
        mol->AddAtom(first);  // add to molecule
        break;

      case 'c': // relative coordinates of atom wrt to already placed atom
        first = new atom;
        valid = true;
        do {
          if (!valid) DoeLog(2) && (eLog()<< Verbose(2) << "Resulting position out of cell." << endl);
          second = mol->AskAtom("Enter atom number: ");
          DoLog(0) && (Log() << Verbose(0) << "Enter relative coordinates." << endl);
          first->x.AskPosition(World::getInstance().getDomain(), false);
          for (int i=NDIM;i--;) {
            first->x.x[i] += second->x.x[i];
          }
        } while (!(valid = mol->CheckBounds((const Vector *)&first->x)));
        first->type = periode->AskElement();  // give type
        mol->AddAtom(first);  // add to molecule
        break;

    case 'd': // two atoms, two angles and a distance
        first = new atom;
        valid = true;
        do {
          if (!valid) {
            DoeLog(2) && (eLog()<< Verbose(2) << "Resulting coordinates out of cell - " << first->x << endl);
          }
          cout << Verbose(0) << "First, we need two atoms, the first atom is the central, while the second is the outer one." << endl;
          second = mol->AskAtom("Enter central atom: ");
          third = mol->AskAtom("Enter second atom (specifying the axis for first angle): ");
          fourth = mol->AskAtom("Enter third atom (specifying a plane for second angle): ");
          a = ask_value("Enter distance between central (first) and new atom: ");
          b = ask_value("Enter angle between new, first and second atom (degrees): ");
          b *= M_PI/180.;
          bound(&b, 0., 2.*M_PI);
          c = ask_value("Enter second angle between new and normal vector of plane defined by first, second and third atom (degrees): ");
          c *= M_PI/180.;
          bound(&c, -M_PI, M_PI);
          cout << Verbose(0) << "radius: " << a << "\t phi: " << b*180./M_PI << "\t theta: " << c*180./M_PI << endl;
/*
          second->Output(1,1,(ofstream *)&cout);
          third->Output(1,2,(ofstream *)&cout);
          fourth->Output(1,3,(ofstream *)&cout);
          n.MakeNormalvector((const vector *)&second->x, (const vector *)&third->x, (const vector *)&fourth->x);
          x.Copyvector(&second->x);
          x.SubtractVector(&third->x);
          x.Copyvector(&fourth->x);
          x.SubtractVector(&third->x);

          if (!z.SolveSystem(&x,&y,&n, b, c, a)) {
         coutg() << Verbose(0) << "Failure solving self-dependent linear system!" << endl;
            continue;
          }
          DoLog(0) && (Log() << Verbose(0) << "resulting relative coordinates: ");
          z.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
          */
          // calc axis vector
          x.CopyVector(&second->x);
          x.SubtractVector(&third->x);
          x.Normalize();
          Log() << Verbose(0) << "x: ",
          x.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
          z.MakeNormalVector(&second->x,&third->x,&fourth->x);
          Log() << Verbose(0) << "z: ",
          z.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
          y.MakeNormalVector(&x,&z);
          Log() << Verbose(0) << "y: ",
          y.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);

          // rotate vector around first angle
          first->x.CopyVector(&x);
          first->x.RotateVector(&z,b - M_PI);
          Log() << Verbose(0) << "Rotated vector: ",
          first->x.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
          // remove the projection onto the rotation plane of the second angle
          n.CopyVector(&y);
          n.Scale(first->x.ScalarProduct(&y));
          Log() << Verbose(0) << "N1: ",
          n.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
          first->x.SubtractVector(&n);
          Log() << Verbose(0) << "Subtracted vector: ",
          first->x.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
          n.CopyVector(&z);
          n.Scale(first->x.ScalarProduct(&z));
          Log() << Verbose(0) << "N2: ",
          n.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
          first->x.SubtractVector(&n);
          Log() << Verbose(0) << "2nd subtracted vector: ",
          first->x.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);

          // rotate another vector around second angle
          n.CopyVector(&y);
          n.RotateVector(&x,c - M_PI);
          Log() << Verbose(0) << "2nd Rotated vector: ",
          n.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);

          // add the two linear independent vectors
          first->x.AddVector(&n);
          first->x.Normalize();
          first->x.Scale(a);
          first->x.AddVector(&second->x);

          DoLog(0) && (Log() << Verbose(0) << "resulting coordinates: ");
          first->x.Output();
          DoLog(0) && (Log() << Verbose(0) << endl);
        } while (!(valid = mol->CheckBounds((const Vector *)&first->x)));
        first->type = periode->AskElement();  // give type
        mol->AddAtom(first);  // add to molecule
        break;

      case 'e': // least square distance position to a set of atoms
        first = new atom;
        atoms = new (Vector*[128]);
        valid = true;
        for(int i=128;i--;)
          atoms[i] = NULL;
        int i=0, j=0;
        cout << Verbose(0) << "Now we need at least three molecules.\n";
        do {
          cout << Verbose(0) << "Enter " << i+1 << "th atom: ";
          cin >> j;
          if (j != -1) {
            second = mol->FindAtom(j);
            atoms[i++] = &(second->x);
          }
        } while ((j != -1) && (i<128));
        if (i >= 2) {
          first->x.LSQdistance((const Vector **)atoms, i);
          first->x.Output();
          first->type = periode->AskElement();  // give type
          mol->AddAtom(first);  // add to molecule
        } else {
          delete first;
          cout << Verbose(0) << "Please enter at least two vectors!\n";
        }
        break;
  };
};

/** Submenu for centering the atoms in the molecule.
 * \param *mol molecule with all the atoms
 */
static void CenterAtoms(molecule *mol)
{
  Vector x, y, helper;
  char choice;  // menu choice char

  cout << Verbose(0) << "===========CENTER ATOMS=========================" << endl;
  cout << Verbose(0) << " a - on origin" << endl;
  cout << Verbose(0) << " b - on center of gravity" << endl;
  cout << Verbose(0) << " c - within box with additional boundary" << endl;
  cout << Verbose(0) << " d - within given simulation box" << endl;
  cout << Verbose(0) << "all else - go back" << endl;
  cout << Verbose(0) << "===============================================" << endl;
  cout << Verbose(0) << "INPUT: ";
  cin >> choice;

  switch (choice) {
    default:
      cout << Verbose(0) << "Not a valid choice." << endl;
      break;
    case 'a':
      cout << Verbose(0) << "Centering atoms in config file on origin." << endl;
      mol->CenterOrigin();
      break;
    case 'b':
      cout << Verbose(0) << "Centering atoms in config file on center of gravity." << endl;
      mol->CenterPeriodic();
      break;
    case 'c':
      cout << Verbose(0) << "Centering atoms in config file within given additional boundary." << endl;
      for (int i=0;i<NDIM;i++) {
        cout << Verbose(0) << "Enter axis " << i << " boundary: ";
        cin >> y.x[i];
      }
      mol->CenterEdge(&x);  // make every coordinate positive
      mol->Center.AddVector(&y); // translate by boundary
      helper.CopyVector(&y);
      helper.Scale(2.);
      helper.AddVector(&x);
      mol->SetBoxDimension(&helper);  // update Box of atoms by boundary
      break;
    case 'd':
      cout << Verbose(1) << "Centering atoms in config file within given simulation box." << endl;
      for (int i=0;i<NDIM;i++) {
        cout << Verbose(0) << "Enter axis " << i << " boundary: ";
        cin >> x.x[i];
      }
      // update Box of atoms by boundary
      mol->SetBoxDimension(&x);
      // center
      mol->CenterInBox();
      break;
  }
};

/** Submenu for aligning the atoms in the molecule.
 * \param *periode periodentafel
 * \param *mol molecule with all the atoms
 */
static void AlignAtoms(periodentafel *periode, molecule *mol)
{
  atom *first, *second, *third;
  Vector x,n;
  char choice;  // menu choice char

  cout << Verbose(0) << "===========ALIGN ATOMS=========================" << endl;
  cout << Verbose(0) << " a - state three atoms defining align plane" << endl;
  cout << Verbose(0) << " b - state alignment vector" << endl;
  cout << Verbose(0) << " c - state two atoms in alignment direction" << endl;
  cout << Verbose(0) << " d - align automatically by least square fit" << endl;
  cout << Verbose(0) << "all else - go back" << endl;
  cout << Verbose(0) << "===============================================" << endl;
  cout << Verbose(0) << "INPUT: ";
  cin >> choice;

  switch (choice) {
    default:
    case 'a': // three atoms defining mirror plane
      first = mol->AskAtom("Enter first atom: ");
      second = mol->AskAtom("Enter second atom: ");
      third = mol->AskAtom("Enter third atom: ");

      n.MakeNormalVector((const Vector *)&first->x,(const Vector *)&second->x,(const Vector *)&third->x);
      break;
    case 'b': // normal vector of mirror plane
      cout << Verbose(0) << "Enter normal vector of mirror plane." << endl;
      n.AskPosition(World::getInstance().getDomain(),0);
      n.Normalize();
      break;
    case 'c': // three atoms defining mirror plane
      first = mol->AskAtom("Enter first atom: ");
      second = mol->AskAtom("Enter second atom: ");

      n.CopyVector((const Vector *)&first->x);
      n.SubtractVector((const Vector *)&second->x);
      n.Normalize();
      break;
    case 'd':
      char shorthand[4];
      Vector a;
      struct lsq_params param;
      do {
        fprintf(stdout, "Enter the element of atoms to be chosen: ");
        fscanf(stdin, "%3s", shorthand);
      } while ((param.type = periode->FindElement(shorthand)) == NULL);
      cout << Verbose(0) << "Element is " << param.type->name << endl;
      mol->GetAlignvector(&param);
      for (int i=NDIM;i--;) {
        x.x[i] = gsl_vector_get(param.x,i);
        n.x[i] = gsl_vector_get(param.x,i+NDIM);
      }
      gsl_vector_free(param.x);
      cout << Verbose(0) << "Offset vector: ";
      x.Output();
      DoLog(0) && (Log() << Verbose(0) << endl);
      n.Normalize();
      break;
  };
  DoLog(0) && (Log() << Verbose(0) << "Alignment vector: ");
  n.Output();
  DoLog(0) && (Log() << Verbose(0) << endl);
  mol->Align(&n);
};

/** Submenu for mirroring the atoms in the molecule.
 * \param *mol molecule with all the atoms
 */
static void MirrorAtoms(molecule *mol)
{
  atom *first, *second, *third;
  Vector n;
  char choice;  // menu choice char

  DoLog(0) && (Log() << Verbose(0) << "===========MIRROR ATOMS=========================" << endl);
  DoLog(0) && (Log() << Verbose(0) << " a - state three atoms defining mirror plane" << endl);
  DoLog(0) && (Log() << Verbose(0) << " b - state normal vector of mirror plane" << endl);
  DoLog(0) && (Log() << Verbose(0) << " c - state two atoms in normal direction" << endl);
  DoLog(0) && (Log() << Verbose(0) << "all else - go back" << endl);
  DoLog(0) && (Log() << Verbose(0) << "===============================================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
  cin >> choice;

  switch (choice) {
    default:
    case 'a': // three atoms defining mirror plane
      first = mol->AskAtom("Enter first atom: ");
      second = mol->AskAtom("Enter second atom: ");
      third = mol->AskAtom("Enter third atom: ");

      n.MakeNormalVector((const Vector *)&first->x,(const Vector *)&second->x,(const Vector *)&third->x);
      break;
    case 'b': // normal vector of mirror plane
      DoLog(0) && (Log() << Verbose(0) << "Enter normal vector of mirror plane." << endl);
      n.AskPosition(World::getInstance().getDomain(),0);
      n.Normalize();
      break;
    case 'c': // three atoms defining mirror plane
      first = mol->AskAtom("Enter first atom: ");
      second = mol->AskAtom("Enter second atom: ");

      n.CopyVector((const Vector *)&first->x);
      n.SubtractVector((const Vector *)&second->x);
      n.Normalize();
      break;
  };
  DoLog(0) && (Log() << Verbose(0) << "Normal vector: ");
  n.Output();
  DoLog(0) && (Log() << Verbose(0) << endl);
  mol->Mirror((const Vector *)&n);
};

/** Submenu for removing the atoms from the molecule.
 * \param *mol molecule with all the atoms
 */
static void RemoveAtoms(molecule *mol)
{
  atom *first, *second;
  int axis;
  double tmp1, tmp2;
  char choice;  // menu choice char

  DoLog(0) && (Log() << Verbose(0) << "===========REMOVE ATOMS=========================" << endl);
  DoLog(0) && (Log() << Verbose(0) << " a - state atom for removal by number" << endl);
  DoLog(0) && (Log() << Verbose(0) << " b - keep only in radius around atom" << endl);
  DoLog(0) && (Log() << Verbose(0) << " c - remove this with one axis greater value" << endl);
  DoLog(0) && (Log() << Verbose(0) << "all else - go back" << endl);
  DoLog(0) && (Log() << Verbose(0) << "===============================================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
  cin >> choice;

  switch (choice) {
    default:
    case 'a':
      if (mol->RemoveAtom(mol->AskAtom("Enter number of atom within molecule: ")))
        DoLog(1) && (Log() << Verbose(1) << "Atom removed." << endl);
      else
        DoLog(1) && (Log() << Verbose(1) << "Atom not found." << endl);
      break;
    case 'b':
      second = mol->AskAtom("Enter number of atom as reference point: ");
      DoLog(0) && (Log() << Verbose(0) << "Enter radius: ");
      cin >> tmp1;
      first = mol->start;
      second = first->next;
      while(second != mol->end) {
        first = second;
        second = first->next;
        if (first->x.DistanceSquared((const Vector *)&second->x) > tmp1*tmp1) // distance to first above radius ...
          mol->RemoveAtom(first);
      }
      break;
    case 'c':
      DoLog(0) && (Log() << Verbose(0) << "Which axis is it: ");
      cin >> axis;
      DoLog(0) && (Log() << Verbose(0) << "Lower boundary: ");
      cin >> tmp1;
      DoLog(0) && (Log() << Verbose(0) << "Upper boundary: ");
      cin >> tmp2;
      first = mol->start;
      second = first->next;
      while(second != mol->end) {
        first = second;
        second = first->next;
        if ((first->x.x[axis] < tmp1) || (first->x.x[axis] > tmp2)) {// out of boundary ...
          //Log() << Verbose(0) << "Atom " << *first << " with " << first->x.x[axis] << " on axis " << axis << " is out of bounds [" << tmp1 << "," << tmp2 << "]." << endl;
          mol->RemoveAtom(first);
        }
      }
      break;
  };
  //mol->Output();
  choice = 'r';
};

/** Submenu for measuring out the atoms in the molecule.
 * \param *periode periodentafel
 * \param *mol molecule with all the atoms
 */
static void MeasureAtoms(periodentafel *periode, molecule *mol, config *configuration)
{
  atom *first, *second, *third;
  Vector x,y;
  double min[256], tmp1, tmp2, tmp3;
  int Z;
  char choice;  // menu choice char

  DoLog(0) && (Log() << Verbose(0) << "===========MEASURE ATOMS=========================" << endl);
  DoLog(0) && (Log() << Verbose(0) << " a - calculate bond length between one atom and all others" << endl);
  DoLog(0) && (Log() << Verbose(0) << " b - calculate bond length between two atoms" << endl);
  DoLog(0) && (Log() << Verbose(0) << " c - calculate bond angle" << endl);
  DoLog(0) && (Log() << Verbose(0) << " d - calculate principal axis of the system" << endl);
  DoLog(0) && (Log() << Verbose(0) << " e - calculate volume of the convex envelope" << endl);
  DoLog(0) && (Log() << Verbose(0) << " f - calculate temperature from current velocity" << endl);
  DoLog(0) && (Log() << Verbose(0) << " g - output all temperatures per step from velocities" << endl);
  DoLog(0) && (Log() << Verbose(0) << "all else - go back" << endl);
  DoLog(0) && (Log() << Verbose(0) << "===============================================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
  cin >> choice;

  switch(choice) {
    default:
      DoLog(1) && (Log() << Verbose(1) << "Not a valid choice." << endl);
      break;
    case 'a':
      first = mol->AskAtom("Enter first atom: ");
      for (int i=MAX_ELEMENTS;i--;)
        min[i] = 0.;

      second = mol->start;
      while ((second->next != mol->end)) {
        second = second->next; // advance
        Z = second->type->Z;
        tmp1 = 0.;
        if (first != second) {
          x.CopyVector((const Vector *)&first->x);
          x.SubtractVector((const Vector *)&second->x);
          tmp1 = x.Norm();
        }
        if ((tmp1 != 0.) && ((min[Z] == 0.) || (tmp1 < min[Z]))) min[Z] = tmp1;
        //Log() << Verbose(0) << "Bond length between Atom " << first->nr << " and " << second->nr << ": " << tmp1 << " a.u." << endl;
      }
      for (int i=MAX_ELEMENTS;i--;)
        if (min[i] != 0.) Log() << Verbose(0) << "Minimum Bond length between " << first->type->name << " Atom " << first->nr << " and next Ion of type " << (periode->FindElement(i))->name << ": " << min[i] << " a.u." << endl;
      break;

    case 'b':
      first = mol->AskAtom("Enter first atom: ");
      second = mol->AskAtom("Enter second atom: ");
      for (int i=NDIM;i--;)
        min[i] = 0.;
      x.CopyVector((const Vector *)&first->x);
      x.SubtractVector((const Vector *)&second->x);
      tmp1 = x.Norm();
      DoLog(1) && (Log() << Verbose(1) << "Distance vector is ");
      x.Output();
      DoLog(0) && (Log() << Verbose(0) << "." << endl << "Norm of distance is " << tmp1 << "." << endl);
      break;

    case 'c':
      DoLog(0) && (Log() << Verbose(0) << "Evaluating bond angle between three - first, central, last - atoms." << endl);
      first = mol->AskAtom("Enter first atom: ");
      second = mol->AskAtom("Enter central atom: ");
      third  = mol->AskAtom("Enter last atom: ");
      tmp1 = tmp2 = tmp3 = 0.;
      x.CopyVector((const Vector *)&first->x);
      x.SubtractVector((const Vector *)&second->x);
      y.CopyVector((const Vector *)&third->x);
      y.SubtractVector((const Vector *)&second->x);
      DoLog(0) && (Log() << Verbose(0) << "Bond angle between first atom Nr." << first->nr << ", central atom Nr." << second->nr << " and last atom Nr." << third->nr << ": ");
      DoLog(0) && (Log() << Verbose(0) << (acos(x.ScalarProduct((const Vector *)&y)/(y.Norm()*x.Norm()))/M_PI*180.) << " degrees" << endl);
      break;
    case 'd':
      DoLog(0) && (Log() << Verbose(0) << "Evaluating prinicipal axis." << endl);
      DoLog(0) && (Log() << Verbose(0) << "Shall we rotate? [0/1]: ");
      cin >> Z;
      if ((Z >=0) && (Z <=1))
        mol->PrincipalAxisSystem((bool)Z);
      else
        mol->PrincipalAxisSystem(false);
      break;
    case 'e':
      {
        DoLog(0) && (Log() << Verbose(0) << "Evaluating volume of the convex envelope.");
        class Tesselation *TesselStruct = NULL;
        const LinkedCell *LCList = NULL;
        LCList = new LinkedCell(mol, 10.);
        FindConvexBorder(mol, TesselStruct, LCList, NULL);
        double clustervolume = VolumeOfConvexEnvelope(TesselStruct, configuration);
        DoLog(0) && (Log() << Verbose(0) << "The tesselated surface area is " << clustervolume << "." << endl);\
        delete(LCList);
        delete(TesselStruct);
      }
      break;
    case 'f':
      mol->OutputTemperatureFromTrajectories((ofstream *)&cout, mol->MDSteps-1, mol->MDSteps);
      break;
    case 'g':
      {
        char filename[255];
        DoLog(0) && (Log() << Verbose(0) << "Please enter filename: " << endl);
        cin >> filename;
        DoLog(1) && (Log() << Verbose(1) << "Storing temperatures in " << filename << "." << endl);
        ofstream *output = new ofstream(filename, ios::trunc);
        if (!mol->OutputTemperatureFromTrajectories(output, 0, mol->MDSteps))
          DoLog(2) && (Log() << Verbose(2) << "File could not be written." << endl);
        else
          DoLog(2) && (Log() << Verbose(2) << "File stored." << endl);
        output->close();
        delete(output);
      }
      break;
  }
};

/** Submenu for measuring out the atoms in the molecule.
 * \param *mol molecule with all the atoms
 * \param *configuration configuration structure for the to be written config files of all fragments
 */
static void FragmentAtoms(molecule *mol, config *configuration)
{
  int Order1;
  clock_t start, end;

  DoLog(0) && (Log() << Verbose(0) << "Fragmenting molecule with current connection matrix ..." << endl);
  DoLog(0) && (Log() << Verbose(0) << "What's the desired bond order: ");
  cin >> Order1;
  if (mol->first->next != mol->last) {  // there are bonds
    start = clock();
    mol->FragmentMolecule(Order1, configuration);
    end = clock();
    DoLog(0) && (Log() << Verbose(0) << "Clocks for this operation: " << (end-start) << ", time: " << ((double)(end-start)/CLOCKS_PER_SEC) << "s." << endl);
  } else
    DoLog(0) && (Log() << Verbose(0) << "Connection matrix has not yet been generated!" << endl);
};

/********************************************** Submenu routine **************************************/

/** Submenu for manipulating atoms.
 * \param *periode periodentafel
 * \param *molecules list of molecules whose atoms are to be manipulated
 */
static void ManipulateAtoms(periodentafel *periode, MoleculeListClass *molecules, config *configuration)
{
  atom *first, *second, *third;
  molecule *mol = NULL;
  Vector x,y,z,n; // coordinates for absolute point in cell volume
  double *factor; // unit factor if desired
  double bond, minBond;
  char choice;  // menu choice char
  bool valid;

  DoLog(0) && (Log() << Verbose(0) << "=========MANIPULATE ATOMS======================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "a - add an atom" << endl);
  DoLog(0) && (Log() << Verbose(0) << "r - remove an atom" << endl);
  DoLog(0) && (Log() << Verbose(0) << "b - scale a bond between atoms" << endl);
  DoLog(0) && (Log() << Verbose(0) << "t - turn an atom round another bond" << endl);
  DoLog(0) && (Log() << Verbose(0) << "u - change an atoms element" << endl);
  DoLog(0) && (Log() << Verbose(0) << "l - measure lengths, angles, ... for an atom" << endl);
  DoLog(0) && (Log() << Verbose(0) << "all else - go back" << endl);
  DoLog(0) && (Log() << Verbose(0) << "===============================================" << endl);
  if (molecules->NumberOfActiveMolecules() > 1)
    DoeLog(2) && (eLog()<< Verbose(2) << "There is more than one molecule active! Atoms will be added to each." << endl);
  DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
  cin >> choice;

  switch (choice) {
    default:
      DoLog(0) && (Log() << Verbose(0) << "Not a valid choice." << endl);
      break;

    case 'a': // add atom
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        AddAtoms(periode, mol);
      }
      break;

    case 'b': // scale a bond
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        DoLog(0) && (Log() << Verbose(0) << "Scaling bond length between two atoms." << endl);
        first = mol->AskAtom("Enter first (fixed) atom: ");
        second = mol->AskAtom("Enter second (shifting) atom: ");
        minBond = 0.;
        for (int i=NDIM;i--;)
          minBond += (first->x.x[i]-second->x.x[i])*(first->x.x[i] - second->x.x[i]);
        minBond = sqrt(minBond);
        DoLog(0) && (Log() << Verbose(0) << "Current Bond length between " << first->type->name << " Atom " << first->nr << " and " << second->type->name << " Atom " << second->nr << ": " << minBond << " a.u." << endl);
        DoLog(0) && (Log() << Verbose(0) << "Enter new bond length [a.u.]: ");
        cin >> bond;
        for (int i=NDIM;i--;) {
          second->x.x[i] -= (second->x.x[i]-first->x.x[i])/minBond*(minBond-bond);
        }
        //Log() << Verbose(0) << "New coordinates of Atom " << second->nr << " are: ";
        //second->Output(second->type->No, 1);
      }
      break;

    case 'c': // unit scaling of the metric
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
       DoLog(0) && (Log() << Verbose(0) << "Angstroem -> Bohrradius: 1.8897261\t\tBohrradius -> Angstroem: 0.52917721" << endl);
       DoLog(0) && (Log() << Verbose(0) << "Enter three factors: ");
       factor = new double[NDIM];
       cin >> factor[0];
       cin >> factor[1];
       cin >> factor[2];
       valid = true;
       mol->Scale((const double ** const)&factor);
       delete[](factor);
      }
     break;

    case 'l': // measure distances or angles
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        MeasureAtoms(periode, mol, configuration);
      }
      break;

    case 'r': // remove atom
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        RemoveAtoms(mol);
      }
      break;

    case 't': // turn/rotate atom
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
          mol = *ListRunner;
          DoLog(0) && (Log() << Verbose(0) << "Turning atom around another bond - first is atom to turn, second (central) and third specify bond" << endl);
          first = mol->AskAtom("Enter turning atom: ");
          second = mol->AskAtom("Enter central atom: ");
          third  = mol->AskAtom("Enter bond atom: ");
          cout << Verbose(0) << "Enter new angle in degrees: ";
          double tmp = 0.;
          cin >> tmp;
          // calculate old angle
          x.CopyVector((const Vector *)&first->x);
          x.SubtractVector((const Vector *)&second->x);
          y.CopyVector((const Vector *)&third->x);
          y.SubtractVector((const Vector *)&second->x);
          double alpha = (acos(x.ScalarProduct((const Vector *)&y)/(y.Norm()*x.Norm()))/M_PI*180.);
          cout << Verbose(0) << "Bond angle between first atom Nr." << first->nr << ", central atom Nr." << second->nr << " and last atom Nr." << third->nr << ": ";
          cout << Verbose(0) << alpha << " degrees" << endl;
          // rotate
          z.MakeNormalVector(&x,&y);
          x.RotateVector(&z,(alpha-tmp)*M_PI/180.);
          x.AddVector(&second->x);
          first->x.CopyVector(&x);
          // check new angle
          x.CopyVector((const Vector *)&first->x);
          x.SubtractVector((const Vector *)&second->x);
          alpha = (acos(x.ScalarProduct((const Vector *)&y)/(y.Norm()*x.Norm()))/M_PI*180.);
          cout << Verbose(0) << "new Bond angle between first atom Nr." << first->nr << ", central atom Nr." << second->nr << " and last atom Nr." << third->nr << ": ";
          cout << Verbose(0) << alpha << " degrees" << endl;
        }
      break;

    case 'u': // change an atom's element
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        int Z;
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        first = NULL;
        do {
          DoLog(0) && (Log() << Verbose(0) << "Change the element of which atom: ");
          cin >> Z;
        } while ((first = mol->FindAtom(Z)) == NULL);
        DoLog(0) && (Log() << Verbose(0) << "New element by atomic number Z: ");
        cin >> Z;
        first->type = periode->FindElement(Z);
        DoLog(0) && (Log() << Verbose(0) << "Atom " << first->nr << "'s element is " << first->type->name << "." << endl);
      }
      break;
  }
};

/** Submenu for manipulating molecules.
 * \param *periode periodentafel
 * \param *molecules list of molecule to manipulate
 */
static void ManipulateMolecules(periodentafel *periode, MoleculeListClass *molecules, config *configuration)
{
  atom *first = NULL;
  Vector x,y,z,n; // coordinates for absolute point in cell volume
  int j, axis, count, faktor;
  char choice;  // menu choice char
  molecule *mol = NULL;
  element **Elements;
  Vector **vectors;
  MoleculeLeafClass *Subgraphs = NULL;

  DoLog(0) && (Log() << Verbose(0) << "=========MANIPULATE GLOBALLY===================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "c - scale by unit transformation" << endl);
  DoLog(0) && (Log() << Verbose(0) << "d - duplicate molecule/periodic cell" << endl);
  DoLog(0) && (Log() << Verbose(0) << "f - fragment molecule many-body bond order style" << endl);
  DoLog(0) && (Log() << Verbose(0) << "g - center atoms in box" << endl);
  DoLog(0) && (Log() << Verbose(0) << "i - realign molecule" << endl);
  DoLog(0) && (Log() << Verbose(0) << "m - mirror all molecules" << endl);
  DoLog(0) && (Log() << Verbose(0) << "o - create connection matrix" << endl);
  DoLog(0) && (Log() << Verbose(0) << "t - translate molecule by vector" << endl);
  DoLog(0) && (Log() << Verbose(0) << "all else - go back" << endl);
  DoLog(0) && (Log() << Verbose(0) << "===============================================" << endl);
  if (molecules->NumberOfActiveMolecules() > 1)
    DoeLog(2) && (eLog()<< Verbose(2) << "There is more than one molecule active! Atoms will be added to each." << endl);
  DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
  cin >> choice;

  switch (choice) {
    default:
      DoLog(0) && (Log() << Verbose(0) << "Not a valid choice." << endl);
      break;

    case 'd': // duplicate the periodic cell along a given axis, given times
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++) 
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        DoLog(0) && (Log() << Verbose(0) << "State the axis [(+-)123]: ");
        cin >> axis;
        DoLog(0) && (Log() << Verbose(0) << "State the factor: ");
        cin >> faktor;

        mol->CountAtoms(); // recount atoms
        if (mol->getAtomCount() != 0) {  // if there is more than none
          count = mol->getAtomCount();  // is changed becausing of adding, thus has to be stored away beforehand
          Elements = new element *[count];
          vectors = new Vector *[count];
          j = 0;
          first = mol->start;
          while (first->next != mol->end) { // make a list of all atoms with coordinates and element
            first = first->next;
            Elements[j] = first->type;
            vectors[j] = &first->x;
            j++;
          }
          if (count != j)
            DoeLog(1) && (eLog()<< Verbose(1) << "AtomCount " << count << " is not equal to number of atoms in molecule " << j << "!" << endl);
          x.Zero();
          y.Zero();
          y.x[abs(axis)-1] = World::getInstance().getDomain()[(abs(axis) == 2) ? 2 : ((abs(axis) == 3) ? 5 : 0)] * abs(axis)/axis; // last term is for sign, first is for magnitude
          for (int i=1;i<faktor;i++) {  // then add this list with respective translation factor times
            x.AddVector(&y); // per factor one cell width further
            for (int k=count;k--;) { // go through every atom of the original cell
              first = new atom(); // create a new body
              first->x.CopyVector(vectors[k]);  // use coordinate of original atom
              first->x.AddVector(&x);     // translate the coordinates
              first->type = Elements[k];  // insert original element
              mol->AddAtom(first);        // and add to the molecule (which increments ElementsInMolecule, AtomCount, ...)
            }
          }
          if (mol->first->next != mol->last) // if connect matrix is present already, redo it
            mol->CreateAdjacencyList(mol->BondDistance, configuration->GetIsAngstroem(), &BondGraph::CovalentMinMaxDistance, NULL);
          // free memory
          delete[](Elements);
          delete[](vectors);
          // correct cell size
          if (axis < 0) { // if sign was negative, we have to translate everything
            x.Zero();
            x.AddVector(&y);
            x.Scale(-(faktor-1));
            mol->Translate(&x);
          }
          World::getInstance().getDomain()[(abs(axis) == 2) ? 2 : ((abs(axis) == 3) ? 5 : 0)] *= faktor;
        }
      }
      break;

    case 'f':
      FragmentAtoms(mol, configuration);
      break;

    case 'g': // center the atoms
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        CenterAtoms(mol);
      }
      break;

    case 'i': // align all atoms
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        AlignAtoms(periode, mol);
      }
      break;

    case 'm': // mirror atoms along a given axis
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        MirrorAtoms(mol);
      }
      break;

    case 'o': // create the connection matrix
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
          mol = *ListRunner;
          double bonddistance;
          clock_t start,end;
          DoLog(0) && (Log() << Verbose(0) << "What's the maximum bond distance: ");
          cin >> bonddistance;
          start = clock();
          mol->CreateAdjacencyList(bonddistance, configuration->GetIsAngstroem(), &BondGraph::CovalentMinMaxDistance, NULL);
          end = clock();
          DoLog(0) && (Log() << Verbose(0) << "Clocks for this operation: " << (end-start) << ", time: " << ((double)(end-start)/CLOCKS_PER_SEC) << "s." << endl);
        }
      break;

    case 't': // translate all atoms
      for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if ((*ListRunner)->ActiveFlag) {
        mol = *ListRunner;
        DoLog(0) && (Log() << Verbose(0) << "Current molecule is: " << mol->IndexNr << "\t" << mol->name << endl);
        DoLog(0) && (Log() << Verbose(0) << "Enter translation vector." << endl);
        x.AskPosition(World::getInstance().getDomain(),0);
        mol->Center.AddVector((const Vector *)&x);
     }
     break;
  }
  // Free all
  if (Subgraphs != NULL) {  // free disconnected subgraph list of DFS analysis was performed
    while (Subgraphs->next != NULL) {
      Subgraphs = Subgraphs->next;
      delete(Subgraphs->previous);
    }
    delete(Subgraphs);
  }
};


/** Submenu for creating new molecules.
 * \param *periode periodentafel
 * \param *molecules list of molecules to add to
 */
static void EditMolecules(periodentafel *periode, MoleculeListClass *molecules)
{
  char choice;  // menu choice char
  Vector center;
  int nr, count;
  molecule *mol = NULL;

  DoLog(0) && (Log() << Verbose(0) << "==========EDIT MOLECULES=====================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "c - create new molecule" << endl);
  DoLog(0) && (Log() << Verbose(0) << "l - load molecule from xyz file" << endl);
  DoLog(0) && (Log() << Verbose(0) << "n - change molecule's name" << endl);
  DoLog(0) && (Log() << Verbose(0) << "N - give molecules filename" << endl);
  DoLog(0) && (Log() << Verbose(0) << "p - parse atoms in xyz file into molecule" << endl);
  DoLog(0) && (Log() << Verbose(0) << "r - remove a molecule" << endl);
  DoLog(0) && (Log() << Verbose(0) << "all else - go back" << endl);
  DoLog(0) && (Log() << Verbose(0) << "===============================================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
  cin >> choice;

  switch (choice) {
    default:
      DoLog(0) && (Log() << Verbose(0) << "Not a valid choice." << endl);
      break;
    case 'c':
      mol = World::getInstance().createMolecule();
      molecules->insert(mol);
      break;

    case 'l': // load from XYZ file
      {
        char filename[MAXSTRINGSIZE];
        DoLog(0) && (Log() << Verbose(0) << "Format should be XYZ with: ShorthandOfElement\tX\tY\tZ" << endl);
        mol = World::getInstance().createMolecule();
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter file name: ");
          cin >> filename;
        } while (!mol->AddXYZFile(filename));
        mol->SetNameFromFilename(filename);
        // center at set box dimensions
        mol->CenterEdge(&center);
        double * const cell_size = World::getInstance().getDomain();
        cell_size[0] = center.x[0];
        cell_size[1] = 0;
        cell_size[2] = center.x[1];
        cell_size[3] = 0;
        cell_size[4] = 0;
        cell_size[5] = center.x[2];
        molecules->insert(mol);
      }
      break;

    case 'n':
      {
        char filename[MAXSTRINGSIZE];
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter index of molecule: ");
          cin >> nr;
          mol = molecules->ReturnIndex(nr);
        } while (mol == NULL);
        DoLog(0) && (Log() << Verbose(0) << "Enter name: ");
        cin >> filename;
        strcpy(mol->name, filename);
      }
      break;

    case 'N':
      {
        char filename[MAXSTRINGSIZE];
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter index of molecule: ");
          cin >> nr;
          mol = molecules->ReturnIndex(nr);
        } while (mol == NULL);
        DoLog(0) && (Log() << Verbose(0) << "Enter name: ");
        cin >> filename;
        mol->SetNameFromFilename(filename);
      }
      break;

    case 'p': // parse XYZ file
      {
        char filename[MAXSTRINGSIZE];
        mol = NULL;
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter index of molecule: ");
          cin >> nr;
          mol = molecules->ReturnIndex(nr);
        } while (mol == NULL);
        DoLog(0) && (Log() << Verbose(0) << "Format should be XYZ with: ShorthandOfElement\tX\tY\tZ" << endl);
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter file name: ");
          cin >> filename;
        } while (!mol->AddXYZFile(filename));
        mol->SetNameFromFilename(filename);
      }
      break;

    case 'r':
      DoLog(0) && (Log() << Verbose(0) << "Enter index of molecule: ");
      cin >> nr;
      count = 1;
      for(MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
        if (nr == (*ListRunner)->IndexNr) {
          mol = *ListRunner;
          molecules->ListOfMolecules.erase(ListRunner);
          delete(mol);
          break;
        }
      break;
  }
};


/** Submenu for merging molecules.
 * \param *periode periodentafel
 * \param *molecules list of molecules to add to
 */
static void MergeMolecules(periodentafel *periode, MoleculeListClass *molecules)
{
  char choice;  // menu choice char

  DoLog(0) && (Log() << Verbose(0) << "===========MERGE MOLECULES=====================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "a - simple add of one molecule to another" << endl);
  DoLog(0) && (Log() << Verbose(0) << "b - count the number of bonds of two elements" << endl);
  DoLog(0) && (Log() << Verbose(0) << "B - count the number of bonds of three elements " << endl);
  DoLog(0) && (Log() << Verbose(0) << "e - embedding merge of two molecules" << endl);
  DoLog(0) && (Log() << Verbose(0) << "h - count the number of hydrogen bonds" << endl);
  DoLog(0) && (Log() << Verbose(0) << "b - count the number of hydrogen bonds" << endl);
  DoLog(0) && (Log() << Verbose(0) << "m - multi-merge of all molecules" << endl);
  DoLog(0) && (Log() << Verbose(0) << "s - scatter merge of two molecules" << endl);
  DoLog(0) && (Log() << Verbose(0) << "t - simple merge of two molecules" << endl);
  DoLog(0) && (Log() << Verbose(0) << "all else - go back" << endl);
  DoLog(0) && (Log() << Verbose(0) << "===============================================" << endl);
  DoLog(0) && (Log() << Verbose(0) << "INPUT: ");
  cin >> choice;

  switch (choice) {
    default:
      DoLog(0) && (Log() << Verbose(0) << "Not a valid choice." << endl);
      break;

    case 'a':
      {
        int src, dest;
        molecule *srcmol = NULL, *destmol = NULL;
        {
          do {
            DoLog(0) && (Log() << Verbose(0) << "Enter index of destination molecule: ");
            cin >> dest;
            destmol = molecules->ReturnIndex(dest);
          } while ((destmol == NULL) && (dest != -1));
          do {
            DoLog(0) && (Log() << Verbose(0) << "Enter index of source molecule to add from: ");
            cin >> src;
            srcmol = molecules->ReturnIndex(src);
          } while ((srcmol == NULL) && (src != -1));
          if ((src != -1) && (dest != -1))
            molecules->SimpleAdd(srcmol, destmol);
        }
      }
      break;

    case 'b':
      {
        const int nr = 2;
        char *names[nr] = {"first", "second"};
        int Z[nr];
        element *elements[nr];
        for (int i=0;i<nr;i++) {
          Z[i] = 0;
          do {
            cout << "Enter " << names[i] << " element: ";
            cin >> Z[i];
          } while ((Z[i] <= 0) && (Z[i] > MAX_ELEMENTS));
          elements[i] = periode->FindElement(Z[i]);
        }
        const int count = CountBondsOfTwo(molecules, elements[0], elements[1]);
        cout << endl << "There are " << count << " ";
        for (int i=0;i<nr;i++) {
          if (i==0)
            cout << elements[i]->symbol;
          else
            cout << "-" << elements[i]->symbol;
        }
        cout << " bonds." << endl;
      }
    break;

    case 'B':
      {
        const int nr = 3;
        char *names[nr] = {"first", "second", "third"};
        int Z[nr];
        element *elements[nr];
        for (int i=0;i<nr;i++) {
          Z[i] = 0;
          do {
            cout << "Enter " << names[i] << " element: ";
            cin >> Z[i];
          } while ((Z[i] <= 0) && (Z[i] > MAX_ELEMENTS));
          elements[i] = periode->FindElement(Z[i]);
        }
        const int count = CountBondsOfThree(molecules, elements[0], elements[1], elements[2]);
        cout << endl << "There are " << count << " ";
        for (int i=0;i<nr;i++) {
          if (i==0)
            cout << elements[i]->symbol;
          else
            cout << "-" << elements[i]->symbol;
        }
        cout << " bonds." << endl;
      }
    break;

    case 'e':
      {
        int src, dest;
        molecule *srcmol = NULL, *destmol = NULL;
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter index of matrix molecule (the variable one): ");
          cin >> src;
          srcmol = molecules->ReturnIndex(src);
        } while ((srcmol == NULL) && (src != -1));
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter index of molecule to merge into (the fixed one): ");
          cin >> dest;
          destmol = molecules->ReturnIndex(dest);
        } while ((destmol == NULL) && (dest != -1));
        if ((src != -1) && (dest != -1))
          molecules->EmbedMerge(destmol, srcmol);
      }
      break;

    case 'h':
      {
        int Z;
        cout << "Please enter interface element: ";
        cin >> Z;
        element * const InterfaceElement = periode->FindElement(Z);
        cout << endl << "There are " << CountHydrogenBridgeBonds(molecules, InterfaceElement) << " hydrogen bridges with connections to " << (InterfaceElement != 0 ? InterfaceElement->name : "None") << "." << endl;
      }
      break;

    case 'm':
      {
        int nr;
        molecule *mol = NULL;
        do {
          DoLog(0) && (Log() << Verbose(0) << "Enter index of molecule to merge into: ");
          cin >> nr;
          mol = molecules->ReturnIndex(nr);
        } while ((mol == NULL) && (nr != -1));
        if (nr != -1) {
          int N = molecules->ListOfMolecules.size()-1;
          int *src = new int(N);
          for(MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
            if ((*ListRunner)->IndexNr != nr)
              src[N++] = (*ListRunner)->IndexNr;        
          molecules->SimpleMultiMerge(mol, src, N);
          delete[](src);
        }
      }
      break;

    case 's':
      DoLog(0) && (Log() << Verbose(0) << "Not implemented yet." << endl);
      break;

    case 't':
      {
        int src, dest;
        molecule *srcmol = NULL, *destmol = NULL;
        {
          do {
            DoLog(0) && (Log() << Verbose(0) << "Enter index of destination molecule: ");
            cin >> dest;
            destmol = molecules->ReturnIndex(dest);
          } while ((destmol == NULL) && (dest != -1));
          do {
            DoLog(0) && (Log() << Verbose(0) << "Enter index of source molecule to merge into: ");
            cin >> src;
            srcmol = molecules->ReturnIndex(src);
          } while ((srcmol == NULL) && (src != -1));
          if ((src != -1) && (dest != -1))
            molecules->SimpleMerge(srcmol, destmol);
        }
      }
      break;
  }
};

/********************************************** Test routine **************************************/

/** Is called always as option 'T' in the menu.
 * \param *molecules list of molecules
 */
static void testroutine(MoleculeListClass *molecules)
{
  // the current test routine checks the functionality of the KeySet&Graph concept:
  // We want to have a multiindex (the KeySet) describing a unique subgraph
  int i, comp, counter=0;

  // create a clone
  molecule *mol = NULL;
  if (molecules->ListOfMolecules.size() != 0) // clone
    mol = (molecules->ListOfMolecules.front())->CopyMolecule();
  else {
    DoeLog(0) && (eLog()<< Verbose(0) << "I don't have anything to test on ... ");
    performCriticalExit();
    return;
  }
  atom *Walker = mol->start;

  // generate some KeySets
  DoLog(0) && (Log() << Verbose(0) << "Generating KeySets." << endl);
  KeySet TestSets[mol->getAtomCount()+1];
  i=1;
  while (Walker->next != mol->end) {
    Walker = Walker->next;
    for (int j=0;j<i;j++) {
      TestSets[j].insert(Walker->nr);
    }
    i++;
  }
  DoLog(0) && (Log() << Verbose(0) << "Testing insertion of already present item in KeySets." << endl);
  KeySetTestPair test;
  test = TestSets[mol->getAtomCount()-1].insert(Walker->nr);
  if (test.second) {
    DoLog(1) && (Log() << Verbose(1) << "Insertion worked?!" << endl);
  } else {
    DoLog(1) && (Log() << Verbose(1) << "Insertion rejected: Present object is " << (*test.first) << "." << endl);
  }
  TestSets[mol->getAtomCount()].insert(mol->end->previous->nr);
  TestSets[mol->getAtomCount()].insert(mol->end->previous->previous->previous->nr);

  // constructing Graph structure
  DoLog(0) && (Log() << Verbose(0) << "Generating Subgraph class." << endl);
  Graph Subgraphs;

  // insert KeySets into Subgraphs
  DoLog(0) && (Log() << Verbose(0) << "Inserting KeySets into Subgraph class." << endl);
  for (int j=0;j<mol->getAtomCount();j++) {
    Subgraphs.insert(GraphPair (TestSets[j],pair<int, double>(counter++, 1.)));
  }
  DoLog(0) && (Log() << Verbose(0) << "Testing insertion of already present item in Subgraph." << endl);
  GraphTestPair test2;
  test2 = Subgraphs.insert(GraphPair (TestSets[mol->getAtomCount()],pair<int, double>(counter++, 1.)));
  if (test2.second) {
    DoLog(1) && (Log() << Verbose(1) << "Insertion worked?!" << endl);
  } else {
    DoLog(1) && (Log() << Verbose(1) << "Insertion rejected: Present object is " << (*(test2.first)).second.first << "." << endl);
  }

  // show graphs
  DoLog(0) && (Log() << Verbose(0) << "Showing Subgraph's contents, checking that it's sorted." << endl);
  Graph::iterator A = Subgraphs.begin();
  while (A !=  Subgraphs.end()) {
    DoLog(0) && (Log() << Verbose(0) << (*A).second.first << ": ");
    KeySet::iterator key = (*A).first.begin();
    comp = -1;
    while (key != (*A).first.end()) {
      if ((*key) > comp)
        DoLog(0) && (Log() << Verbose(0) << (*key) << " ");
      else
        DoLog(0) && (Log() << Verbose(0) << (*key) << "! ");
      comp = (*key);
      key++;
    }
    DoLog(0) && (Log() << Verbose(0) << endl);
    A++;
  }
  delete(mol);
};


/** Tries given filename or standard on saving the config file.
 * \param *ConfigFileName name of file
 * \param *configuration pointer to configuration structure with all the values
 * \param *periode pointer to periodentafel structure with all the elements
 * \param *molecules list of molecules structure with all the atoms and coordinates
 */
static void SaveConfig(char *ConfigFileName, config *configuration, periodentafel *periode, MoleculeListClass *molecules)
{
  char filename[MAXSTRINGSIZE];
  ofstream output;
  molecule *mol = World::getInstance().createMolecule();
  mol->SetNameFromFilename(ConfigFileName);

  if (!strcmp(configuration->configpath, configuration->GetDefaultPath())) {
    DoeLog(2) && (eLog()<< Verbose(2) << "config is found under different path then stated in config file::defaultpath!" << endl);
  }


  // first save as PDB data
  if (ConfigFileName != NULL)
    strcpy(filename, ConfigFileName);
  if (output == NULL)
    strcpy(filename,"main_pcp_linux");
  DoLog(0) && (Log() << Verbose(0) << "Saving as pdb input ");
  if (configuration->SavePDB(filename, molecules))
    DoLog(0) && (Log() << Verbose(0) << "done." << endl);
  else
    DoLog(0) && (Log() << Verbose(0) << "failed." << endl);

  // then save as tremolo data file
  if (ConfigFileName != NULL)
    strcpy(filename, ConfigFileName);
  if (output == NULL)
    strcpy(filename,"main_pcp_linux");
  DoLog(0) && (Log() << Verbose(0) << "Saving as tremolo data input ");
  if (configuration->SaveTREMOLO(filename, molecules))
    DoLog(0) && (Log() << Verbose(0) << "done." << endl);
  else
    DoLog(0) && (Log() << Verbose(0) << "failed." << endl);

  // translate each to its center and merge all molecules in MoleculeListClass into this molecule
  int N = molecules->ListOfMolecules.size();
  int *src = new int[N];
  N=0;
  for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++) {
    src[N++] = (*ListRunner)->IndexNr;
    (*ListRunner)->Translate(&(*ListRunner)->Center);
  }
  molecules->SimpleMultiAdd(mol, src, N);
  delete[](src);

  // ... and translate back
  for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++) {
    (*ListRunner)->Center.Scale(-1.);
    (*ListRunner)->Translate(&(*ListRunner)->Center);
    (*ListRunner)->Center.Scale(-1.);
  }

  DoLog(0) && (Log() << Verbose(0) << "Storing configuration ... " << endl);
  // get correct valence orbitals
  mol->CalculateOrbitals(*configuration);
  configuration->InitMaxMinStopStep = configuration->MaxMinStopStep = configuration->MaxPsiDouble;
  if (ConfigFileName != NULL) { // test the file name
    strcpy(filename, ConfigFileName);
    output.open(filename, ios::trunc);
  } else if (strlen(configuration->configname) != 0) {
    strcpy(filename, configuration->configname);
    output.open(configuration->configname, ios::trunc);
    } else {
      strcpy(filename, DEFAULTCONFIG);
      output.open(DEFAULTCONFIG, ios::trunc);
    }
  output.close();
  output.clear();
  DoLog(0) && (Log() << Verbose(0) << "Saving of config file ");
  if (configuration->Save(filename, periode, mol))
    DoLog(0) && (Log() << Verbose(0) << "successful." << endl);
  else
    DoLog(0) && (Log() << Verbose(0) << "failed." << endl);

  // and save to xyz file
  if (ConfigFileName != NULL) {
    strcpy(filename, ConfigFileName);
    strcat(filename, ".xyz");
    output.open(filename, ios::trunc);
  }
  if (output == NULL) {
    strcpy(filename,"main_pcp_linux");
    strcat(filename, ".xyz");
    output.open(filename, ios::trunc);
  }
  DoLog(0) && (Log() << Verbose(0) << "Saving of XYZ file ");
  if (mol->MDSteps <= 1) {
    if (mol->OutputXYZ(&output))
      DoLog(0) && (Log() << Verbose(0) << "successful." << endl);
    else
      DoLog(0) && (Log() << Verbose(0) << "failed." << endl);
  } else {
    if (mol->OutputTrajectoriesXYZ(&output))
      DoLog(0) && (Log() << Verbose(0) << "successful." << endl);
    else
      DoLog(0) && (Log() << Verbose(0) << "failed." << endl);
  }
  output.close();
  output.clear();

  // and save as MPQC configuration
  if (ConfigFileName != NULL)
    strcpy(filename, ConfigFileName);
  if (output == NULL)
    strcpy(filename,"main_pcp_linux");
  DoLog(0) && (Log() << Verbose(0) << "Saving as mpqc input ");
  if (configuration->SaveMPQC(filename, mol))
    DoLog(0) && (Log() << Verbose(0) << "done." << endl);
  else
    DoLog(0) && (Log() << Verbose(0) << "failed." << endl);

  if (!strcmp(configuration->configpath, configuration->GetDefaultPath())) {
    DoeLog(2) && (eLog()<< Verbose(2) << "config is found under different path then stated in config file::defaultpath!" << endl);
  }

  World::getInstance().destroyMolecule(mol);
};

#endif

/** Parses the command line options.
 * Note that this function is from now on transitional. All commands that are not passed
 * here are handled by CommandLineParser and the actions of CommandLineUIFactory.
 * \param argc argument count
 * \param **argv arguments array
 * \param *molecules list of molecules structure
 * \param *periode elements structure
 * \param configuration config file structure
 * \param *ConfigFileName pointer to config file name in **argv
 * \param *PathToDatabases pointer to db's path in **argv
 * \param &ArgcList list of arguments that we do not parse here
 * \return exit code (0 - successful, all else - something's wrong)
 */
static int ParseCommandLineOptions(int argc, char **argv, MoleculeListClass *&molecules, periodentafel *&periode,
                                   config& configuration, char **ConfigFileName, set<int> &ArgcList)
{
  Vector x,y,z,n;  // coordinates for absolute point in cell volume
  ifstream test;
  ofstream output;
  string line;
  bool SaveFlag = false;
  int ExitFlag = 0;
  int j;
  double volume = 0.;
  enum ConfigStatus configPresent = absent;
  int argptr;
  molecule *mol = NULL;
  string BondGraphFileName("\n");

  if (argc > 1) { // config file specified as option
    // 1. : Parse options that just set variables or print help
    argptr = 1;
    do {
      if (argv[argptr][0] == '-') {
        DoLog(0) && (Log() << Verbose(0) << "Recognized command line argument: " << argv[argptr][1] << ".\n");
        argptr++;
        switch(argv[argptr-1][1]) {
          case 'h':
          case 'H':
          case '?':
            ArgcList.insert(argptr-1);
            return(1);
            break;
          case 'v':
            if ((argptr >= argc) || (argv[argptr][0] == '-')) {
              DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments for specifying verbosity: -v <level>" << endl);
              performCriticalExit();
            } else {
              setVerbosity(atoi(argv[argptr]));
              ArgcList.insert(argptr-1);
              ArgcList.insert(argptr);
              argptr++;
            }
            break;
          case 'V':
            ArgcList.insert(argptr-1);
            return(1);
            break;
          case 'B':
            if (ExitFlag == 0) ExitFlag = 1;
            if ((argptr+5 >= argc)) {
              DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments for setting Box: -B <xx> <<xy> <<xz> <yy> <yz> <zz>" << endl);
              performCriticalExit();
            } else {
              ArgcList.insert(argptr-1);
              ArgcList.insert(argptr);
              ArgcList.insert(argptr+1);
              ArgcList.insert(argptr+2);
              ArgcList.insert(argptr+3);
              ArgcList.insert(argptr+4);
              ArgcList.insert(argptr+5);
              argptr+=6;
            }
            break;
          case 'e':
            if ((argptr >= argc) || (argv[argptr][0] == '-')) {
              DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments for specifying element db: -e <db file>" << endl);
              performCriticalExit();
            } else {
              ArgcList.insert(argptr-1);
              ArgcList.insert(argptr);
              argptr+=1;
            }
            break;
          case 'g':
            if ((argptr >= argc) || (argv[argptr][0] == '-')) {
              DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments for specifying bond length table: -g <table file>" << endl);
              performCriticalExit();
            } else {
              ArgcList.insert(argptr-1);
              ArgcList.insert(argptr);
              argptr+=1;
            }
            break;
          case 'M':
            if ((argptr >= argc) || (argv[argptr][0] == '-')) {
              ExitFlag = 255;
              DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for setting MPQC basis: -M <basis name>" << endl);
              performCriticalExit();
            } else {
              ArgcList.insert(argptr-1);
              ArgcList.insert(argptr);
              argptr+=1;
            }
            break;
          case 'n':
            if ((argptr >= argc) || (argv[argptr][0] == '-')) {
              ExitFlag = 255;
              DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for setting fast-parsing: -n <0/1>" << endl);
              performCriticalExit();
            } else {
              ArgcList.insert(argptr-1);
              ArgcList.insert(argptr);
              argptr+=1;
            }
            break;
          case 'X':
            if ((argptr >= argc) || (argv[argptr][0] == '-')) {
              ExitFlag = 255;
              DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for setting default molecule name: -X <name>" << endl);
              performCriticalExit();
            } else {
              ArgcList.insert(argptr-1);
              ArgcList.insert(argptr);
              argptr+=1;
            }
            break;
          default:   // no match? Step on
            argptr++;
            break;
        }
      } else
        argptr++;
    } while (argptr < argc);

    // 3b. Find config file name and parse if possible, also BondGraphFileName
    if (argv[1][0] != '-') {
      // simply create a new molecule, wherein the config file is loaded and the manipulation takes place
      DoLog(0) && (Log() << Verbose(0) << "Config file given." << endl);
      test.open(argv[1], ios::in);
      if (test == NULL) {
        //return (1);
        output.open(argv[1], ios::out);
        if (output == NULL) {
          DoLog(1) && (Log() << Verbose(1) << "Specified config file " << argv[1] << " not found." << endl);
          configPresent = absent;
        } else {
          DoLog(0) && (Log() << Verbose(0) << "Empty configuration file." << endl);
          strcpy(*ConfigFileName, argv[1]);
          configPresent = empty;
          output.close();
        }
      } else {
        test.close();
        strcpy(*ConfigFileName, argv[1]);
        DoLog(1) && (Log() << Verbose(1) << "Specified config file found, parsing ... ");
        switch (configuration.TestSyntax(*ConfigFileName, periode)) {
          case 1:
            DoLog(0) && (Log() << Verbose(0) << "new syntax." << endl);
            configuration.Load(*ConfigFileName, BondGraphFileName, periode, molecules);
            configPresent = present;
            break;
          case 0:
            DoLog(0) && (Log() << Verbose(0) << "old syntax." << endl);
            configuration.LoadOld(*ConfigFileName, BondGraphFileName, periode, molecules);
            configPresent = present;
            break;
          default:
            DoLog(0) && (Log() << Verbose(0) << "Unknown syntax or empty, yet present file." << endl);
            configPresent = empty;
       }
      }
    } else
      configPresent = absent;
     // set mol to first active molecule
     if (molecules->ListOfMolecules.size() != 0) {
       for (MoleculeList::iterator ListRunner = molecules->ListOfMolecules.begin(); ListRunner != molecules->ListOfMolecules.end(); ListRunner++)
         if ((*ListRunner)->ActiveFlag) {
           mol = *ListRunner;
           break;
         }
     }
     if (mol == NULL) {
       mol = World::getInstance().createMolecule();
       mol->ActiveFlag = true;
       if (*ConfigFileName != NULL)
         mol->SetNameFromFilename(*ConfigFileName);
       molecules->insert(mol);
     }

    // 4. parse again through options, now for those depending on elements db and config presence
    argptr = 1;
    do {
      DoLog(0) && (Log() << Verbose(0) << "Current Command line argument: " << argv[argptr] << "." << endl);
      if (argv[argptr][0] == '-') {
        argptr++;
        if ((configPresent == present) || (configPresent == empty)) {
          switch(argv[argptr-1][1]) {
            case 'p':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough arguments for parsing: -p <xyz file>" << endl);
                performCriticalExit();
              } else {
                SaveFlag = true;
                DoLog(1) && (Log() << Verbose(1) << "Parsing xyz file for new atoms." << endl);
                if (!mol->AddXYZFile(argv[argptr]))
                  DoLog(2) && (Log() << Verbose(2) << "File not found." << endl);
                else {
                  DoLog(2) && (Log() << Verbose(2) << "File found and parsed." << endl);
                  configPresent = present;
                }
              }
              break;
            case 'a':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+4 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough arguments for adding atom: -a <Z> --position <x> <y> <z>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                argptr+=5;
              }
              break;
            default:   // no match? Don't step on (this is done in next switch's default)
              break;
          }
        }
        if (configPresent == present) {
          switch(argv[argptr-1][1]) {
            case 'D':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for depth-first-search analysis: -D <max. bond distance>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                argptr+=1;
              }
              break;
            case 'I':
              DoLog(1) && (Log() << Verbose(1) << "Dissecting molecular system into a set of disconnected subgraphs ... " << endl);
              ArgcList.insert(argptr-1);
              argptr+=0;
              break;
            case 'C':
              {
                if (ExitFlag == 0) ExitFlag = 1;
                if ((argptr+11 >= argc)) {
                  ExitFlag = 255;
                  DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for pair correlation analysis: -C[p] <type: E/P/S> [more params] <output> <bin output> <BinStart> <BinEnd>" << endl);
                  performCriticalExit();
                } else {
                  switch(argv[argptr][0]) {
                    case 'E':
                      ArgcList.insert(argptr-1);
                      ArgcList.insert(argptr);
                      ArgcList.insert(argptr+1);
                      ArgcList.insert(argptr+2);
                      ArgcList.insert(argptr+3);
                      ArgcList.insert(argptr+4);
                      ArgcList.insert(argptr+5);
                      ArgcList.insert(argptr+6);
                      ArgcList.insert(argptr+7);
                      ArgcList.insert(argptr+8);
                      ArgcList.insert(argptr+9);
                      ArgcList.insert(argptr+10);
                      ArgcList.insert(argptr+11);
                      argptr+=12;
                      break;

                    case 'P':
                      ArgcList.insert(argptr-1);
                      ArgcList.insert(argptr);
                      ArgcList.insert(argptr+1);
                      ArgcList.insert(argptr+2);
                      ArgcList.insert(argptr+3);
                      ArgcList.insert(argptr+4);
                      ArgcList.insert(argptr+5);
                      ArgcList.insert(argptr+6);
                      ArgcList.insert(argptr+7);
                      ArgcList.insert(argptr+8);
                      ArgcList.insert(argptr+9);
                      ArgcList.insert(argptr+10);
                      ArgcList.insert(argptr+11);
                      ArgcList.insert(argptr+12);
                      ArgcList.insert(argptr+13);
                      ArgcList.insert(argptr+14);
                      argptr+=15;
                      break;

                    case 'S':
                      ArgcList.insert(argptr-1);
                      ArgcList.insert(argptr);
                      ArgcList.insert(argptr+1);
                      ArgcList.insert(argptr+2);
                      ArgcList.insert(argptr+3);
                      ArgcList.insert(argptr+4);
                      ArgcList.insert(argptr+5);
                      ArgcList.insert(argptr+6);
                      ArgcList.insert(argptr+7);
                      ArgcList.insert(argptr+8);
                      ArgcList.insert(argptr+9);
                      ArgcList.insert(argptr+10);
                      ArgcList.insert(argptr+11);
                      ArgcList.insert(argptr+12);
                      ArgcList.insert(argptr+13);
                      ArgcList.insert(argptr+14);
                      argptr+=15;
                      break;

                    default:
                      ExitFlag = 255;
                      DoeLog(0) && (eLog()<< Verbose(0) << "Invalid type given for pair correlation analysis: -C <type: E/P/S> [more params] <output> <bin output>" << endl);
                      performCriticalExit();
                      break;
                  }
                }
                break;
              }
            case 'E':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (!IsValidNumber(argv[argptr]))) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for changing element: -E <atom nr.> --element <Z>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;
            case 'F':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+12 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for filling with molecule: -F <filler xyz file> --MaxDistance <distance or -1> --distances <x> <y> <z>  --lengths <surface> <randatm> <randmol> --DoRotate <0/1>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                ArgcList.insert(argptr+5);
                ArgcList.insert(argptr+6);
                ArgcList.insert(argptr+7);
                ArgcList.insert(argptr+8);
                ArgcList.insert(argptr+9);
                ArgcList.insert(argptr+10);
                ArgcList.insert(argptr+11);
                ArgcList.insert(argptr+12);
                argptr+=13;
              }
              break;
            case 'A':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag =255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Missing source file for bonds in molecule: -A <bond sourcefile> --molecule-by-id <molecule_id>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;

            case 'J':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag =255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Missing path of adjacency file: -J <path> --molecule-by-id <molecule_id>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;

            case 'j':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag =255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Missing path of bonds file: -j <path> --molecule-by-id <molecule_id>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;

            case 'N':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+4 >= argc) || (argv[argptr][0] == '-')){
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for non-convex envelope: -N <molecule_id> --sphere-radius <radius> --nonconvex-file <output prefix>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                argptr+=5;
              }
              break;
            case 'S':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for storing tempature: -S <temperature file> --molecule-by-id 0" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;
            case 'L':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+8 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for linear interpolation: -L <prefix> --start-step <step0> --end-step <step1> --molecule-by-id 0 --id-mapping <0/1>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                ArgcList.insert(argptr+5);
                ArgcList.insert(argptr+6);
                ArgcList.insert(argptr+7);
                ArgcList.insert(argptr+8);
                argptr+=9;
              }
              break;
            case 'P':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for parsing and integrating forces: -P <forces file> --molecule-by-id <molecule_id>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;
            case 'R':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+4 >= argc) || (argv[argptr][0] == '-'))  {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for removing atoms: -R <distance> --position <x> <y> <z>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                argptr+=5;
              }
              break;
            case 't':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+4 >= argc) || (!IsValidNumber(argv[argptr])) || (!IsValidNumber(argv[argptr+1])) || (!IsValidNumber(argv[argptr+2])) ) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for translation: -t <x> <y> <z> --molecule-by-id <molecule_id> --periodic <0/1>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                ArgcList.insert(argptr+5);
                ArgcList.insert(argptr+6);
                argptr+=7;
              }
              break;
            case 's':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (!IsValidNumber(argv[argptr])) || (!IsValidNumber(argv[argptr+1])) || (!IsValidNumber(argv[argptr+2])) ) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for scaling: -s <factor_x> [factor_y] [factor_z]" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;
            case 'b':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+5 >= argc) || (argv[argptr][0] == '-') || (!IsValidNumber(argv[argptr])) || (!IsValidNumber(argv[argptr+1])) || (!IsValidNumber(argv[argptr+2])) || (!IsValidNumber(argv[argptr+3])) || (!IsValidNumber(argv[argptr+4])) || (!IsValidNumber(argv[argptr+5])) ) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for centering in box: -b <xx> <xy> <xz> <yy> <yz> <zz>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                ArgcList.insert(argptr+5);
                argptr+=6;
              }
              break;
            case 'B':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+5 >= argc) || (argv[argptr][0] == '-') || (!IsValidNumber(argv[argptr])) || (!IsValidNumber(argv[argptr+1])) || (!IsValidNumber(argv[argptr+2])) || (!IsValidNumber(argv[argptr+3])) || (!IsValidNumber(argv[argptr+4])) || (!IsValidNumber(argv[argptr+5])) ) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for bounding in box: -B <xx> <xy> <xz> <yy> <yz> <zz>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                ArgcList.insert(argptr+5);
                argptr+=6;
              }
              break;
            case 'c':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (argv[argptr][0] == '-') || (!IsValidNumber(argv[argptr])) || (!IsValidNumber(argv[argptr+1])) || (!IsValidNumber(argv[argptr+2])) ) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for centering with boundary: -c <boundary_x> <boundary_y> <boundary_z>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;
            case 'O':
              if (ExitFlag == 0) ExitFlag = 1;
              ArgcList.insert(argptr-1);
              argptr+=0;
              break;
            case 'r':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr >= argc) || (argv[argptr][0] == '-') || (!IsValidNumber(argv[argptr])))  {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for removing atoms: -r <id>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                argptr+=1;
              }
              break;
            case 'f':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+1 >= argc) || (argv[argptr][0] == '-')) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments for fragmentation: -f <max. bond distance> <bond order>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                argptr+=5;
              }
              break;
            case 'm':
              if (ExitFlag == 0) ExitFlag = 1;
              j = atoi(argv[argptr++]);
              if ((j<0) || (j>1)) {
                DoeLog(1) && (eLog()<< Verbose(1) << "Argument of '-m' should be either 0 for no-rotate or 1 for rotate." << endl);
                j = 0;
              }
              if (j) {
                SaveFlag = true;
                DoLog(0) && (Log() << Verbose(0) << "Converting to prinicipal axis system." << endl);
                mol->PrincipalAxisSystem((bool)j);
              } else
                ArgcList.insert(argptr-1);
                argptr+=0;
              break;
            case 'o':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+4 >= argc) || (argv[argptr][0] == '-')){
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for convex envelope: -o <molecule_id> --output-file <output file> --output-file <binned output file>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                ArgcList.insert(argptr+3);
                ArgcList.insert(argptr+4);
                argptr+=5;
              }
              break;
            case 'U':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+1 >= argc) || (argv[argptr][0] == '-') || (!IsValidNumber(argv[argptr])) || (!IsValidNumber(argv[argptr+1])) ) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for suspension with specified volume: -U <volume> <density>" << endl);
                performCriticalExit();
              } else {
                volume = atof(argv[argptr++]);
                DoLog(0) && (Log() << Verbose(0) << "Using " << volume << " angstrom^3 as the volume instead of convex envelope one's." << endl);
              }
            case 'u':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr >= argc) || (argv[argptr][0] == '-') || (!IsValidNumber(argv[argptr])) ) {
                if (volume != -1)
                  ExitFlag = 255;
                  DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for suspension: -u <density>" << endl);
                  performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                argptr+=1;
              }
              break;
            case 'd':
              if (ExitFlag == 0) ExitFlag = 1;
              if ((argptr+2 >= argc) || (argv[argptr][0] == '-') || (!IsValidNumber(argv[argptr])) || (!IsValidNumber(argv[argptr+1])) || (!IsValidNumber(argv[argptr+2])) ) {
                ExitFlag = 255;
                DoeLog(0) && (eLog()<< Verbose(0) << "Not enough or invalid arguments given for repeating cells: -d <repeat_x> <repeat_y> <repeat_z>" << endl);
                performCriticalExit();
              } else {
                ArgcList.insert(argptr-1);
                ArgcList.insert(argptr);
                ArgcList.insert(argptr+1);
                ArgcList.insert(argptr+2);
                argptr+=3;
              }
              break;
            default:   // no match? Step on
              if ((argptr < argc) && (argv[argptr][0] != '-')) // if it started with a '-' we've already made a step!
                argptr++;
              break;
          }
        }
      } else argptr++;
    } while (argptr < argc);
    if (SaveFlag)
      configuration.SaveAll(*ConfigFileName, periode, molecules);
  } else {  // no arguments, hence scan the elements db
    if (periode->LoadPeriodentafel(configuration.databasepath))
      DoLog(0) && (Log() << Verbose(0) << "Element list loaded successfully." << endl);
    else
      DoLog(0) && (Log() << Verbose(0) << "Element list loading failed." << endl);
    configuration.RetrieveConfigPathAndName("main_pcp_linux");
  }
  return(ExitFlag);
};

/********************************************** Main routine **************************************/

void cleanUp(){
  World::purgeInstance();
  logger::purgeInstance();
  errorLogger::purgeInstance();
  UIFactory::purgeInstance();
  MapOfActions::purgeInstance();
  CommandLineParser::purgeInstance();
  ActionRegistry::purgeInstance();
  ActionHistory::purgeInstance();
#ifdef LOG_OBSERVER
  cout << observerLog().getLog();
#endif
  Memory::getState();
}

int main(int argc, char **argv)
{
    // while we are non interactive, we want to abort from asserts
    //ASSERT_DO(Assert::Abort);
    Vector x, y, z, n;
    ifstream test;
    ofstream output;
    string line;
    char **Arguments = NULL;
    int ArgcSize = 0;
    int ExitFlag = 0;
    bool ArgumentsCopied = false;
    char *ConfigFileName = new char[MAXSTRINGSIZE];

    // print version check whether arguments are present at all
    cout << ESPACKVersion << endl;
    if (argc < 2) {
      cout << "Obtain help with " << argv[0] << " -h." << endl;
      cleanUp();
      Memory::getState();
      return(1);
    }


    setVerbosity(0);
    // need to init the history before any action is created
    ActionHistory::init();

    // In the interactive mode, we can leave the user the choice in case of error
    ASSERT_DO(Assert::Ask);

    // from this moment on, we need to be sure to deeinitialize in the correct order
    // this is handled by the cleanup function
    atexit(cleanUp);

    // Parse command line options and if present create respective UI
    {
      set<int> ArgcList;
      ArgcList.insert(0); // push back program!
      ArgcList.insert(1); // push back config file name
      // handle arguments by ParseCommandLineOptions()
      ExitFlag = ParseCommandLineOptions(argc,argv,World::getInstance().getMolecules(),World::getInstance().getPeriode(),*World::getInstance().getConfig(), &ConfigFileName, ArgcList);
      World::getInstance().setExitFlag(ExitFlag);
      // copy all remaining arguments to a new argv
      Arguments = new char *[ArgcList.size()];
      cout << "The following arguments are handled by CommandLineParser: ";
      for (set<int>::iterator ArgcRunner = ArgcList.begin(); ArgcRunner != ArgcList.end(); ++ArgcRunner) {
        Arguments[ArgcSize] = new char[strlen(argv[*ArgcRunner])+2];
        strcpy(Arguments[ArgcSize], argv[*ArgcRunner]);
        cout << " " << argv[*ArgcRunner];
        ArgcSize++;
      }
      cout << endl;
      ArgumentsCopied = true;
      // handle remaining arguments by CommandLineParser
      MapOfActions::getInstance().AddOptionsToParser();
      map <std::string, std::string> ShortFormToActionMap = MapOfActions::getInstance().getShortFormToActionMap();
      CommandLineParser::getInstance().Run(ArgcSize,Arguments, ShortFormToActionMap);
      if (!CommandLineParser::getInstance().isEmpty()) {
        DoLog(0) && (Log() << Verbose(0) << "Setting UI to CommandLine." << endl);
        UIFactory::registerFactory(new CommandLineUIFactory::description());
        UIFactory::makeUserInterface("CommandLine");
      } else {
        DoLog(0) && (Log() << Verbose(0) << "Setting UI to Text." << endl);
        UIFactory::registerFactory(new TextUIFactory::description());
        UIFactory::makeUserInterface("Text");
      }
    }

    {
      MainWindow *mainWindow = UIFactory::getInstance().makeMainWindow();
      mainWindow->display();
      delete mainWindow;
    }

    Log() << Verbose(0) << "Saving to " << ConfigFileName << "." << endl;
    World::getInstance().getConfig()->SaveAll(ConfigFileName, World::getInstance().getPeriode(), World::getInstance().getMolecules());

  // free the new argv
  if (ArgumentsCopied) {
    for (int i=0; i<ArgcSize;i++)
      delete[](Arguments[i]);
    delete[](Arguments);
  }
  delete[](ConfigFileName);

  ExitFlag = World::getInstance().getExitFlag();
  return (ExitFlag == 1 ? 0 : ExitFlag);
}

/********************************************** E N D **************************************************/
