/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ReceiveJobsOperation.cpp
 *
 *  Created on: Dec 11, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <iostream>
#include <vector>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include <boost/serialization/vector.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "ControllerChoices.hpp"
#include "Jobs/FragmentJob.hpp"

#include "Controller/Commands/ReceiveJobsOperation.hpp"

/** Handle connect operation to receive jobs from controller
 *
 * \param e error code if something went wrong
 * \param endpoint_iterator endpoint of the connection
 */
void ReceiveJobsOperation::handle_connect(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Give choice.
    enum ControllerChoices choice = ReceiveJobs;
    connection_.async_write(choice,
      boost::bind(&ReceiveJobsOperation::handle_SendJobs, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&ReceiveJobsOperation::handle_connect, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }
}

/** Callback function when jobs have been sent.
 *
 * \param e error code if something went wrong
 */
void ReceiveJobsOperation::handle_SendJobs(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Start operation to read the vector
    // of jobs. The connection::async_write() function will automatically
    // encode the data that is written to the underlying socket.
    LOG(1, "INFO: Sending "+toString(jobs.size())+" jobs ...");
    connection_.async_write(jobs,
      boost::bind(&ReceiveJobsOperation::handle_FinishOperation, this,
      boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }

  // Since we are not starting a new operation the io_service will run out of
  // work to do and the client will exit.
}

/** Callback function when an operation has been completed.
 *
 * \param e error code if something went wrong
 */
void ReceiveJobsOperation::handle_FinishOperation(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);

  LOG(1, "INFO: Jobs have been sent. Clearing.");
  jobs.clear();

  Operation::handle_FinishOperation(e);
}

/** Place number of jobs into this controller.
 *
 * \param _jobs jobs to add
 */
void ReceiveJobsOperation::addJobs(const std::vector<FragmentJob::ptr> &_jobs)
{
  jobs.reserve(jobs.size()+_jobs.size());
  BOOST_FOREACH(FragmentJob::ptr job, _jobs) {
    jobs.push_back(job);
  }
  LOG(1, "INFO: ReceiveJobsOperation now contains " << jobs.size() << " job(s).");
}

