/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtFragmentList.cpp
 *
 *  Created on: Sep 03, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtFragmentList.hpp"
#include "Views/Qt4/QtNumericalItem.hpp"

#include <QAbstractItemView>

#include <iterator>
#include <iostream>

#include <boost/lexical_cast.hpp>

//#include "CodePatterns/MemDebug.hpp"

#include <iterator>
#include <iostream>

#include <boost/assign.hpp>

#include "CodePatterns/Log.hpp"

#include "Descriptors/AtomIdDescriptor.hpp"
#include "Formula.hpp"
#include "Fragmentation/KeySetsContainer.hpp"
#include "Fragmentation/Summation/IndexSetContainer.hpp"
#include "Fragmentation/Summation/Containers/FragmentationResultContainer.hpp"
#include "Fragmentation/Summation/Containers/MPQCData.hpp"
#include "World.hpp"

using namespace std;

using namespace boost::assign;

const int QtFragmentList::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtFragmentList::COLUMNNAMES[QtFragmentList::COLUMNCOUNT]={"Number","KeySet","Formula","Energy value","Energy contribution"};

QtFragmentList::QtFragmentList(QWidget * _parent) :
    QTreeWidget (_parent),
    Observer("QtFragmentList"),
    dirty(true),
    fragmentresultcontainer_enabled(false)
{
  setColumnCount(COLUMNCOUNT);
  setSortingEnabled(true);
  QStringList header;
	for(int i=0; i<COLUMNCOUNT;++i)
	  header << COLUMNNAMES[i];
	setHeaderLabels(header);
	sortByColumn(0, Qt::AscendingOrder);

	setSelectionMode( MultiSelection );

  refill();

  FragmentationResultContainer &fragments =
      FragmentationResultContainer::getInstance();
	fragments.signOn(this);
	fragmentresultcontainer_enabled = true;

  connect(this,SIGNAL(itemSelectionChanged()),this,SLOT(rowSelected()));
  connect(this,SIGNAL(changed()),this,SLOT(update()));
}

QtFragmentList::~QtFragmentList()
{
  if (fragmentresultcontainer_enabled) {
    FragmentationResultContainer &fragments =
        FragmentationResultContainer::getInstance();
    fragments.signOff(this);
  }
}

void QtFragmentList::update(Observable *publisher) {
  boost::unique_lock<boost::mutex> lock(flag_mutex);
  dirty = true;
}

void QtFragmentList::refill()
{
  boost::unique_lock<boost::mutex> lock(flag_mutex);
  FragmentationResultContainer &fragments =
      FragmentationResultContainer::getInstance();

  // clear everything
  FragmentSelection.clear();
  indexsets.clear();
  clear();

  size_t count = 0;
  const IndexSetContainer keysets(fragments.getKeySets());
  const FragmentationResultContainer::shortrangedata_t &shortrange = fragments.getShortRangeResults();
  const FragmentationShortRangeResults::summedshortrange_t &summedshortrange = fragments.getShortRangeSummedResults();
  IndexSetContainer::Container_t::const_iterator keyiter = keysets.getContainer().begin();
  FragmentationResultContainer::shortrangedata_t::const_iterator resultiter = shortrange.begin();
  for (;(keyiter != keysets.getContainer().end()) && (resultiter != shortrange.end());
      ++keyiter,++resultiter,++count) {

    std::vector<int> numerical_columns;
    numerical_columns += ENERGYVALUE, ENERGYCONTRIBUTION;
    QTreeWidgetItem *treeItem = new QtNumericalItem(NUMBER, numerical_columns, this);
    treeItem->setText(NUMBER, QString::number(count));
    {
      std::stringstream output;
      output << **keyiter;
      treeItem->setText(KEYSET, QString(output.str().c_str()));
    }
    {
      std::stringstream output;
      Formula formula;
      for (IndexSet::const_iterator indexiter = (*keyiter)->begin();
          indexiter != (*keyiter)->end();
          ++indexiter) {
        const atom * const _atom = const_cast<const World &>(World::getInstance()).
            getAtom(AtomById(*indexiter));
        if (_atom != NULL)
          formula.addElements(_atom->getElementNo(), 1);
        else
          ELOG(2, "QtFragmentList::refill() - Could not find id " << *indexiter << " for formula.");
      }
      output << formula.toString();
      treeItem->setText(FORMULA, QString(output.str().c_str()));
    }
    if (summedshortrange.empty()) {
      {
        std::stringstream output;
        output << resultiter->second.energies.total;
        treeItem->setText(ENERGYVALUE, QString(output.str().c_str()));
      }
      {
        std::stringstream output;
        output << "";
        treeItem->setText(ENERGYCONTRIBUTION, QString(output.str().c_str()));
      }
    } else {
      FragmentationShortRangeResults::summedshortrange_t::const_iterator sumresiter =
          summedshortrange.find(*keyiter);
      ASSERT( sumresiter != summedshortrange.end(),
          "QtFragmentList::refill() - "+toString(*keyiter)+" not present in summed results.");
      {
        std::stringstream output;
        output << sumresiter->second.first.energies.total;
        treeItem->setText(ENERGYVALUE, QString(output.str().c_str()));
      }
      {
        std::stringstream output;
        output << sumresiter->second.second.energies.total;
        treeItem->setText(ENERGYCONTRIBUTION, QString(output.str().c_str()));
      }
    }
    FragmentSelection.push_back(treeItem->isSelected());
    indexsets.push_back( **keyiter );
  }
  dirty = false;
}

void QtFragmentList::paintEvent(QPaintEvent * event)
{
  if (dirty)
    refill();
  QTreeWidget::paintEvent(event);
}

void QtFragmentList::subjectKilled(Observable *publisher)
{
  // as a new instance should always already be present ... just sign on
  if (static_cast<FragmentationResultContainer *>(publisher) == FragmentationResultContainer::getPointer()) {
    fragmentresultcontainer_enabled = false;
  }
}

void QtFragmentList::rowSelected()
{
  //std::cout << "rowSelected\n";
  // unselect
  for (int i=0;i<topLevelItemCount();++i) {
    QTreeWidgetItem *item = topLevelItem(i);
    bool newSelection = item->isSelected();
    const int itemno = boost::lexical_cast<int>(item->text(NUMBER).toStdString());
    if (newSelection != FragmentSelection[itemno]){
//      LOG(1, "INFO: selection changed for item #" << itemno);
      // get the keyset and select the specific atoms
      if (!newSelection) {
//        LOG(2, "INFO: Seeing unselection for item #" << itemno);
        for (IndexSet::const_iterator iter = indexsets[itemno].begin();
            iter != indexsets[itemno].end();
            ++iter)
          World::getInstance().unselectAtom(*iter);
      }
    }
  }
  // select all
  for (int i=0;i<topLevelItemCount();++i) {
    QTreeWidgetItem *item = topLevelItem(i);
    bool newSelection = item->isSelected();
    const int itemno = boost::lexical_cast<int>(item->text(NUMBER).toStdString());
    if (newSelection != FragmentSelection[itemno]){
      FragmentSelection[itemno] = newSelection;
      // get the keyset and select the specific atoms
      if (newSelection) {
//        LOG(2, "INFO: Seeing selection for item #" << itemno);
        for (IndexSet::const_iterator iter = indexsets[itemno].begin();
            iter != indexsets[itemno].end();
            ++iter)
          World::getInstance().selectAtom(*iter);
      }
    }
  }
}

