#!/bin/bash

optimizations=("-O0"\
	       "-O1"\
               "-O2"\
               "-O3"\
               "-Os");

options=("" \
         "-DLOG_OBSERVER" \
         "-DNO_MEMDEBUG" \
         "-DNO_CACHING" \
         "-DNDEBUG" \
         "-DNO_MEMDEBUG -DLOG_OBSERVER" \
         "-DNO_CACHING -DLOG_OBSERVER" \
         "-DNO_CACHING -DNO_MEMDEBUG" \
         "-DNDEBUG -DNO_CACHING" \
	 "-DNO_CACHING -DNO_MEMDEBUG -DLOG_OBSERVER" \
        );

outfile="test.log";
logfile="full.log";
docheck=0;
docheck_mem=0;

function usage(){
  echo "usage $0 options";
  echo "";
  echo "This script runs a full test for molecuilder, using several compilation options";
  echo "";
  echo "OPTIONS:";
  echo "  -h			Show this message"
  echo "  -o <outfile>		Outfile to use for test results";
  echo "  -f <logfile>		File to use for output from commands";
  echo "  -s			Short tests (no memcheck)";
  echo "  -c            	Only configure and compile (implies -s)";
  echo "  -O <opt-level>	Only compile this optimization level";
}

while getopts “ho:f:scO:” OPTION
do
  case $OPTION in
   h)
      usage;
      exit 0;
      ;;
   o)
     outfile=$OPTARG;
     ;;
   f)
     logfile=$OPTARG;
     ;;
   s)
     docheck_mem=1;
     ;;
   c)
     docheck=1;
     docheck_mem=1;
     ;;
   O)
     optimizations=("-O$OPTARG");
     ;;
   ?)
     usage;
     exit 1;
     ;;
  esac
done

if [[ -z $outfile ]] || [[ -z $logfile ]]
then
  usage;
  exit 1;
fi

outfile=`realpath -s $outfile`;
logfile=`realpath -s $logfile`;


BOOST_ROOT=/opt/packages/boost;

function configure(){
  echo "Configuring";
  CXXFLAGS="$2" $1/configure --prefix=$PWD >> $logfile 2>&1;
}

function compile(){
  echo "Making";
  make all install >>$logfile 2>&1;
}

function check(){
  echo "Checking";
  make check >> $logfile 2>&1;
}

function memcheck(){
  echo "Valgrinding";
  retval=0;
  for test in molecuilder/src/unittests/*
  do
    if [ -x "$test" ]
    then
      echo -n "    $test: " >> $outfile;
      valgrind -v --error-exitcode=255 $test >> $logfile 2>&1;
      if $?
      then
      	echo "OK" >> $outfile
      else
        echo "FAIL" >> $outfile;
        retval=1;
      fi
    fi
  done
  return $retval
}

function test(){

  echo "Testing with \"$2\"";

  echo -n "  Configuring: " >> $outfile;
  if configure "$1" "$2"
  then
    echo "OK" >> $outfile;
  else
    echo "FAIL" >> $outfile;
    return;
  fi

  echo -n "  Compiling: " >> $outfile;
  if compile
  then
    echo "OK" >> $outfile;
  else
    echo "FAIL" >> $outfile;
    return;
  fi

  if docheck
  then
    echo -n "  Running testsuite: " >> $outfile;
    if check
    then
      echo "OK" >> $outfile;
    else
      echo "FAIL" >> $outfile;
      return;
    fi
  fi

  if docheck_mem
  then
    echo  "  Checking memory Errors:..." >> $outfile;
    if memcheck
    then
      echo "  ...OK" >> $outfile
    else
      echo "  ...FAIL" >> $outfile
      return
    fi
  fi
}



function run(){
  echo "Testing with \"$1\":" >> $outfile;
  testdir=`mktemp -d`;
  basedir=$PWD;
  cd $testdir;
  test "$basedir" "$1";
  cd $basedir;
  #rm -rf $testdir;
  echo "" >> $outfile;
}


echo -n "Full compilation test for Molecuilder started on " > $outfile;
date >> $outfile;
echo "" >> $logfile;

for optimization in "${optimizations[@]}"
do
  for option in "${options[@]}"
  do
    run "$optimization $option";
  done
done

echo -n "Full compilation test for Molecuilder on " >> $outfile
date >> $outfile
