/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AtomDescriptorUnitTest.cpp
 *
 *  Created on: Feb 9, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AtomDescriptorUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>

#include <Descriptors/AtomDescriptor.hpp>
#include <Descriptors/AtomIdDescriptor.hpp>

#include "World.hpp"
#include "atom.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AtomDescriptorTest );

// set up and tear down
void AtomDescriptorTest::setUp(){
  World::getInstance();
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::getInstance().createAtom();
    atomIds[i]= atoms[i]->getId();
  }
}

void AtomDescriptorTest::tearDown(){
  World::purgeInstance();
}

// some helper functions
static bool hasAllAtoms(std::vector<atom*> atoms,atomId_t ids[ATOM_COUNT], std::set<atomId_t> excluded = std::set<atomId_t>()){
  for(int i=0;i<ATOM_COUNT;++i){
    atomId_t id = ids[i];
    if(!excluded.count(id)){
      std::vector<atom*>::iterator iter;
      bool res=false;
      for(iter=atoms.begin();iter!=atoms.end();++iter){
        res |= (*iter)->getId() == id;
      }
      if(!res) {
        cout << "Atom " << id << " missing in returned list" << endl;
        return false;
      }
    }
  }
  return true;
}

static bool hasNoDuplicateAtoms(std::vector<atom*> atoms){
  std::set<atomId_t> found;
  std::vector<atom*>::iterator iter;
  for(iter=atoms.begin();iter!=atoms.end();++iter){
    int id = (*iter)->getId();
    if(found.count(id))
      return false;
    found.insert(id);
  }
  return true;
}


void AtomDescriptorTest::AtomBaseSetsTest(){
  std::vector<atom*> allAtoms = World::getInstance().getAllAtoms(AllAtoms());
  CPPUNIT_ASSERT_EQUAL( true , hasAllAtoms(allAtoms,atomIds));
  CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateAtoms(allAtoms));

  std::vector<atom*> noAtoms = World::getInstance().getAllAtoms(NoAtoms());
  CPPUNIT_ASSERT_EQUAL( true , noAtoms.empty());
}
void AtomDescriptorTest::AtomIdTest(){
  // test Atoms from boundaries and middle of the set
  atom* testAtom;
  testAtom = World::getInstance().getAtom(AtomById(atomIds[0]));
  CPPUNIT_ASSERT(testAtom);
  CPPUNIT_ASSERT_EQUAL( atomIds[0], testAtom->getId());
  testAtom = World::getInstance().getAtom(AtomById(atomIds[ATOM_COUNT/2]));
  CPPUNIT_ASSERT(testAtom);
  CPPUNIT_ASSERT_EQUAL( atomIds[ATOM_COUNT/2], testAtom->getId());
  testAtom = World::getInstance().getAtom(AtomById(atomIds[ATOM_COUNT-1]));
  CPPUNIT_ASSERT(testAtom);
  CPPUNIT_ASSERT_EQUAL( atomIds[ATOM_COUNT-1], testAtom->getId());

  // find some ID that has not been created
  atomId_t outsideId=0;
  bool res = false;
  for(outsideId=0;!res;++outsideId) {
    res = true;
    for(int i = 0; i < ATOM_COUNT; ++i){
      res &= atomIds[i]!=outsideId;
    }
  }
  // test from outside of set
  testAtom = World::getInstance().getAtom(AtomById(outsideId));
  CPPUNIT_ASSERT(!testAtom);
}
void AtomDescriptorTest::AtomCalcTest(){
  // test some elementary set operations
  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(AllAtoms()||NoAtoms());
    CPPUNIT_ASSERT_EQUAL( true , hasAllAtoms(testAtoms,atomIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateAtoms(testAtoms));
  }

  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(NoAtoms()||AllAtoms());
    CPPUNIT_ASSERT_EQUAL( true , hasAllAtoms(testAtoms,atomIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateAtoms(testAtoms));
  }

  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(NoAtoms()&&AllAtoms());
    CPPUNIT_ASSERT_EQUAL( true , testAtoms.empty());
  }

  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(AllAtoms()&&NoAtoms());
    CPPUNIT_ASSERT_EQUAL( true , testAtoms.empty());
  }

  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(!AllAtoms());
    CPPUNIT_ASSERT_EQUAL( true , testAtoms.empty());
  }

  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(!NoAtoms());
    CPPUNIT_ASSERT_EQUAL( true , hasAllAtoms(testAtoms,atomIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateAtoms(testAtoms));
  }
  // exclude and include some atoms
  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(AllAtoms()&&(!AtomById(atomIds[ATOM_COUNT/2])));
    std::set<atomId_t> excluded;
    excluded.insert(atomIds[ATOM_COUNT/2]);
    CPPUNIT_ASSERT_EQUAL( true , hasAllAtoms(testAtoms,atomIds,excluded));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicateAtoms(testAtoms));
    CPPUNIT_ASSERT_EQUAL( (size_t)(ATOM_COUNT-1), testAtoms.size());
  }

  {
    std::vector<atom*> testAtoms = World::getInstance().getAllAtoms(NoAtoms()||(AtomById(atomIds[ATOM_COUNT/2])));
    CPPUNIT_ASSERT_EQUAL( (size_t)1, testAtoms.size());
    CPPUNIT_ASSERT_EQUAL( atomIds[ATOM_COUNT/2], testAtoms[0]->getId());
  }
}
