/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * SendResultsOperation.cpp
 *
 *  Created on: Dec 11, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include <boost/bind.hpp>
#include <boost/foreach.hpp>
#include <iostream>
#include <vector>
#include "Connection.hpp" // Must come before boost/serialization headers.
#include <boost/serialization/vector.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "ControllerChoices.hpp"
#include "FragmentResult.hpp"

#include "Controller/Commands/SendResultsOperation.hpp"

/** Handle connect operation to send results.
 *
 * \param e error code if something went wrong
 * \param endpoint_iterator endpoint of the connection
 */
void SendResultsOperation::handle_connect_get(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  Info info(__FUNCTION__);
  if (!e)
  {
    // Successfully established connection. Give choice.
    enum ControllerChoices choice = SendResults;
    connection_.async_write(choice,
      boost::bind(&SendResultsOperation::handle_ReceivingResults, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&SendResultsOperation::handle_connect_get, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }
}

/** Callback function when results have been received.
 *
 * \param e error code if something went wrong
 */
void SendResultsOperation::handle_ReceivingResults(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);
  if (!e)
  {
     // The connection::async_read() function will automatically
     // decode the data that is written to the underlying socket.
     connection_.async_read(results,
        boost::bind(&SendResultsOperation::handle_ReceivedResults, this,
        boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    Exitflag = ErrorFlag;
    ELOG(1, e.message());
  }

  // Since we are not starting a new operation the io_service will run out of
  // work to do and the client will exit.
}


/** Callback function when results have been received.
 *
 * \param e error code if something went wrong
 */
void SendResultsOperation::handle_ReceivedResults(const boost::system::error_code& e)
{
  Info info(__FUNCTION__);

  LOG(1, "INFO: Received "+toString(results.size())+" results.");

  Operation::handle_FinishOperation(e);
}

/** Internal function to connect to the endpoint of the server asynchronuously.
 *
 * We require internal connetion_ and host and service to be set up for this.
 */
void SendResultsOperation::connect_get()
{
  Info info(__FUNCTION__);
  // Resolve the host name into an IP address.
  boost::asio::ip::tcp::resolver::iterator endpoint_iterator = getEndpointIterator();
  boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;

  // Start an asynchronous connect operation.
  std::cout << "Connecting to endpoint " << endpoint << " to get results " << std::endl;
  connection_.socket().async_connect(endpoint,
    boost::bind(&SendResultsOperation::handle_connect_get, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
}

/** Getter for results.
 *
 * \sa calculateResults()
 * \return vector of results for the added jobs (\sa addJobs()).
 */
std::vector<FragmentResult> SendResultsOperation::getResults()
{
  Info info(__FUNCTION__);
  return results;
}

/** Function to initiate receival of results.
 *
 */
void SendResultsOperation::operator()()
{
  // connect
  connect_get();
  //disconnect
  disconnect();
}

void SendResultsOperation::handle_connect(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{}
