/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FragmentQueueUnitTest.cpp
 *
 *  Created on: Oct 23, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "FragmentQueueUnitTest.hpp"

#include <vector>

#include "CodePatterns/Assert.hpp"
#include "FragmentQueue.hpp"

#include "CodePatterns/Observer/Channels.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

#include "stubs/FragmentJobStub.hpp"
#include "unittests/stubs/ObserverStub.hpp"

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( FragmentQueueTest );


void FragmentQueueTest::setUp()
{
  // Throw assertions
  ASSERT_DO(Assert::Throw);

  queue = new FragmentQueue();
  addobserver = new NotificationObserver(queue->getChannel(FragmentQueue::JobAdded));
  removeobserver = new NotificationObserver(queue->getChannel(FragmentQueue::JobRemoved));

  // and sign on
  queue->signOn(addobserver, FragmentQueue::JobAdded);
  queue->signOn(removeobserver, FragmentQueue::JobRemoved);
}


void FragmentQueueTest::tearDown()
{
  queue->signOff(addobserver, FragmentQueue::JobAdded);
  queue->signOff(removeobserver, FragmentQueue::JobRemoved);

  delete queue;
  delete removeobserver;
  delete addobserver;
}

/** UnitTest for working JobQueue
 */
void FragmentQueueTest::JobTest()
{
  FragmentJob::ptr testJob(new FragmentJobStub(JobId::IllegalJob));
  /// check for illegal id
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->pushJob(testJob), Assert::AssertionFailure );
#endif
  // set to valid id
  testJob->setId(1);

  CPPUNIT_ASSERT_EQUAL(false, queue->isJobPresent() );

#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushJob(testJob) );
#else
  queue->pushJob(testJob);
#endif
  CPPUNIT_ASSERT( addobserver->wasNotified );
  CPPUNIT_ASSERT( !removeobserver->wasNotified );
  addobserver->wasNotified = false;

  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL(true, queue->isJobPresent() );
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());
  {
    FragmentQueue::ResultMap::const_iterator iter = queue->results.find((JobId_t)1);
    CPPUNIT_ASSERT( iter != queue->results.end() );
    CPPUNIT_ASSERT( FragmentQueue::NoResult == iter->second );
  }

  // push same id again
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->pushJob(testJob), Assert::AssertionFailure );
#endif

  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());

  FragmentJob::ptr poppedJob;
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( poppedJob = queue->popJob() );
#else
  poppedJob = queue->popJob();
#endif
  CPPUNIT_ASSERT( !addobserver->wasNotified );
  CPPUNIT_ASSERT( removeobserver->wasNotified );
  CPPUNIT_ASSERT( poppedJob == testJob );

  CPPUNIT_ASSERT_EQUAL((size_t)0, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL(false, queue->isJobPresent() );
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());
  {
    FragmentQueue::ResultMap::const_iterator iter = queue->results.find((JobId_t)1);
    CPPUNIT_ASSERT( iter != queue->results.end() );
    CPPUNIT_ASSERT( FragmentQueue::NoResultQueued == iter->second );
  }

#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->popJob(), Assert::AssertionFailure );
#endif
}

/** UnitTest for adding multiple jobs at a time.
 */
void FragmentQueueTest::JobsTest()
{
  // prepare some jobs
  FragmentJob::ptr testJob(new FragmentJobStub(JobId::IllegalJob));
  testJob->setId((JobId_t)1);
  FragmentJob::ptr anothertestJob(new FragmentJobStub(JobId::IllegalJob));
  anothertestJob->setId((JobId_t)2);

  // prepare a vector of them
  std::vector<FragmentJob::ptr> testjobs;
  testjobs.push_back(testJob);
  testjobs.push_back(anothertestJob);

  // prepare another vector of them
  std::vector<FragmentJob::ptr> sametestjobs;
  sametestjobs.push_back(testJob);
  sametestjobs.push_back(anothertestJob);

  // push the vector
  CPPUNIT_ASSERT_EQUAL( (size_t)0, queue->jobs.size() );
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushJobs(testjobs) );
#else
  queue->pushJobs(testjobs);
#endif
  CPPUNIT_ASSERT_EQUAL( (size_t)2, queue->jobs.size() );
  CPPUNIT_ASSERT_EQUAL((size_t)2, queue->results.size());
  {
    FragmentQueue::ResultMap::const_iterator iter = queue->results.find((JobId_t)1);
    CPPUNIT_ASSERT( iter != queue->results.end() );
    CPPUNIT_ASSERT( FragmentQueue::NoResult == iter->second );
  }
  {
    FragmentQueue::ResultMap::const_iterator iter = queue->results.find((JobId_t)2);
    CPPUNIT_ASSERT( iter != queue->results.end() );
    CPPUNIT_ASSERT( FragmentQueue::NoResult == iter->second );
  }
  // push again
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->pushJobs(testjobs), Assert::AssertionFailure );
#endif

  CPPUNIT_ASSERT_EQUAL( (size_t)2, queue->jobs.size() );
  CPPUNIT_ASSERT_EQUAL((size_t)2, queue->results.size());
}

/** UnitTest for working ResultMap
 */
void FragmentQueueTest::ResultsTest()
{
  // prepare a job
  FragmentJob::ptr testJob(new FragmentJobStub(JobId::IllegalJob));
  testJob->setId(1);
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushJob(testJob) );
#else
  queue->pushJob(testJob);
#endif

  // check for present job to do
  CPPUNIT_ASSERT_EQUAL( (size_t)1, queue->getPresentJobs() );

#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->popJob() );
#else
  queue->popJob();
#endif

  // check for present job to do
  CPPUNIT_ASSERT_EQUAL( (size_t)0, queue->getPresentJobs() );

  // prepare a result
  FragmentResult::ptr testResult( new FragmentResult(1) );
  FragmentResult::ptr wrongIdResult( new FragmentResult(2) );

  // check that none are present and we can't get result yet
  CPPUNIT_ASSERT_EQUAL( (size_t)0, queue->getDoneJobs() );
  CPPUNIT_ASSERT( !queue->isResultPresent(1) );
  CPPUNIT_ASSERT( !queue->isResultPresent(2) );
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->getResult(1), Assert::AssertionFailure );
#endif

  /// check for admonishing wrong id
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->pushResult(wrongIdResult), Assert::AssertionFailure );
#endif

  // push correct result
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushResult(testResult) );
#else
  queue->pushResult(testResult);
#endif

  // check presence again
  CPPUNIT_ASSERT( queue->isResultPresent(1) );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, queue->getDoneJobs() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, queue->getPresentJobs() );

  // obtain result again
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->getResult(1) );
#else
  queue->getResult(1);
#endif

  // check presence one more time
  CPPUNIT_ASSERT_EQUAL( (size_t)0, queue->getDoneJobs() );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, queue->getPresentJobs() );
  CPPUNIT_ASSERT( !queue->isResultPresent(1) );
  CPPUNIT_ASSERT( !queue->isResultPresent(2) );
#ifndef NDEBUG
  std::cout << "The following assertion is intended and does not indicate a failure." << std::endl;
  CPPUNIT_ASSERT_THROW( queue->getResult(1), Assert::AssertionFailure );
#endif
}

/** UnitTest for working ResultMap
 */
void FragmentQueueTest::AllResultsTest()
{
  // prepare a job
  FragmentJob::ptr testJob( new FragmentJobStub(JobId::IllegalJob) );
  testJob->setId((JobId_t)1);
  FragmentJob::ptr anothertestJob( new FragmentJobStub(JobId::IllegalJob) );
  anothertestJob->setId((JobId_t)2);

#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushJob(testJob) );
  CPPUNIT_ASSERT_NO_THROW( queue->pushJob(anothertestJob) );
#else
  queue->pushJob(testJob);
  queue->pushJob(anothertestJob);
#endif

  // check that no results are returned.
  {
    const std::vector<FragmentResult::ptr> results = queue->getAllResults();
    CPPUNIT_ASSERT_EQUAL( (size_t)0, results.size() );
  }

  // pop both as if some work was being done
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->popJob() );
  CPPUNIT_ASSERT_NO_THROW( queue->popJob() );
#else
  queue->popJob();
  queue->popJob();
#endif

  // prepare a result
  FragmentResult::ptr testResult( new FragmentResult(1) );
  FragmentResult::ptr anothertestResult( new FragmentResult(2) );

  // push correct result
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushResult(testResult) );
  CPPUNIT_ASSERT_NO_THROW( queue->pushResult(anothertestResult) );
#else
  queue->pushResult(testResult);
  queue->pushResult(anothertestResult);
#endif

  // check that two results are returned.
  {
    const std::vector<FragmentResult::ptr> results = queue->getAllResults();
    CPPUNIT_ASSERT_EQUAL( (size_t)2, results.size() );
  }
}

/** Unit test for resubmitJob().
 *
 */
void FragmentQueueTest::resubmitTest()
{
  FragmentJob::ptr testJob(new FragmentJobStub(1));

  // push a Job into queue
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->pushJob(testJob) );
#else
  queue->pushJob(testJob);
#endif
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());

  // pop the job
  // pop both as if some work was being done
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->popJob() );
#else
  queue->popJob();
#endif
  CPPUNIT_ASSERT_EQUAL((size_t)0, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());

  // resubmit
#ifndef NDEBUG
  CPPUNIT_ASSERT_NO_THROW( queue->resubmitJob((JobId_t)1) );
#else
  queue->resubmitJob((JobId_t)1);
#endif

  // check whethers it's present again
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->jobs.size());
  CPPUNIT_ASSERT_EQUAL((size_t)1, queue->results.size());
}

