/*
 * FragmentQueue.hpp
 *
 *  Created on: Oct 19, 2011
 *      Author: heber
 */

#ifndef FRAGMENTQUEUE_HPP_
#define FRAGMENTQUEUE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <deque>
#include <map>
#include <vector>

#include "CodePatterns/Observer/Observable.hpp"

#include "types.hpp"
#include "Jobs/FragmentJob.hpp"
#include "Results/FragmentResult.hpp"

class FragmentQueueTest;

/** This class contains a deque for temporarily storing the FragmentJob
 * instances until they are received by the server. Also all results are
 * contained herein.
 */
class FragmentQueue : public Observable
{
  friend class FragmentQueueTest;
public:
  FragmentQueue();
  ~FragmentQueue();

  enum NotificationType {
    JobAdded,
    JobRemoved,
    NotificationType_MAX  // denotes the maximum of available notification types
  };

  // entering jobs into queue
  void pushJob(FragmentJob::ptr job);
  void pushJobs(std::vector<FragmentJob::ptr> &_jobs);
  FragmentJob::ptr popJob();
  bool isJobPresent() const;
  void resubmitJob(const JobId_t jobid);

  // querying for results
  bool isResultPresent(JobId_t jobid) const;
  FragmentResult::ptr getResult(JobId_t jobid);
  void pushResult(FragmentResult::ptr &result);
  std::vector<FragmentResult::ptr> getAllResults();

  size_t getDoneJobs() const;
  size_t getPresentJobs() const;

private:
  bool isPresentResult(const FragmentResult::ptr result) const;

  //!> result that takes place in ResultQueue after job has arrived before it has been popped.
  static FragmentResult::ptr NoResult;
  //!> result that takes place in ResultQueue until real result has arrived.
  static FragmentResult::ptr NoResultQueued;
  //!> result that takes place in ResultQueue after real result has been delivered.
  static FragmentResult::ptr ResultDelivered;
  //!> maximum number of attempts before job is not resubmitted
  static size_t Max_Attempts;

  typedef std::deque<FragmentJob::ptr> JobQueue;
  typedef std::map<JobId_t, FragmentJob::ptr> BackupMap;
  typedef std::map<JobId_t, FragmentResult::ptr> ResultMap;
  typedef std::map<JobId_t, size_t> AttemptsMap;

  //!> queue for all jobs
  JobQueue jobs;
  //!> backup map for popped jobs that workers failed to calculate
  BackupMap backup;
  //!> map for all results that have been polled by the server
  ResultMap results;
  //!> map for all jobs that had to be resubmitted storing how often this has been tried
  AttemptsMap attempts;
};


#endif /* FRAGMENTQUEUE_HPP_ */
