/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * PairPotential_Morse.cpp
 *
 *  Created on: Oct 03, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "PairPotential_Morse.hpp"

#include <boost/assign/list_of.hpp> // for 'map_list_of()'
#include <boost/bind.hpp>
#include <cmath>
#include <string>

#include "CodePatterns/Assert.hpp"

#include "FunctionApproximation/Extractors.hpp"
#include "FunctionApproximation/TrainingData.hpp"
#include "Potentials/helpers.hpp"
#include "Potentials/ParticleTypeCheckers.hpp"

class Fragment;

// static definitions
const PairPotential_Morse::ParameterNames_t
PairPotential_Morse::ParameterNames =
      boost::assign::list_of<std::string>
      ("spring_constant")
      ("equilibrium_distance")
      ("dissociation_energy")
      ("") //energy_offset
    ;
const std::string PairPotential_Morse::potential_token("morse");

PairPotential_Morse::PairPotential_Morse(
    const ParticleTypes_t &_ParticleTypes
    ) :
  SerializablePotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  // have some decent defaults for parameter_derivative checking
  params[spring_constant] = 1.;
  params[equilibrium_distance] = 1.;
  params[dissociation_energy] = 0.1;
  params[energy_offset] = 0.1;
}

PairPotential_Morse::PairPotential_Morse(
    const ParticleTypes_t &_ParticleTypes,
    const double _spring_constant,
    const double _equilibrium_distance,
    const double _dissociation_energy,
    const double _energy_offset) :
  SerializablePotential(_ParticleTypes),
  params(parameters_t(MAXPARAMS, 0.))
{
  params[spring_constant] = _spring_constant;
  params[equilibrium_distance] = _equilibrium_distance;
  params[dissociation_energy] = _dissociation_energy;
  params[energy_offset] = _energy_offset;
}

void PairPotential_Morse::setParameters(const parameters_t &_params)
{
  const size_t paramsDim = _params.size();
  ASSERT( paramsDim <= getParameterDimension(),
      "PairPotential_Morse::setParameters() - we need not more than "
      +toString(getParameterDimension())+" parameters.");
  for(size_t i=0;i<paramsDim;++i)
    params[i] = _params[i];

#ifndef NDEBUG
  parameters_t check_params(getParameters());
  check_params.resize(paramsDim); // truncate to same size
  ASSERT( check_params == _params,
      "PairPotential_Morse::setParameters() - failed, mismatch in to be set "
      +toString(_params)+" and set "+toString(check_params)+" params.");
#endif
}

PairPotential_Morse::results_t
PairPotential_Morse::operator()(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 1,
      "PairPotential_Morse::operator() - requires exactly one argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Morse::operator() - types don't match with ones in arguments.");
  const argument_t &r_ij = arguments[0];
  // Maple: f(r,D,k,R,c) := D * (1 - exp(-k*(r-R)))^(2)+c
  const result_t result =
      params[dissociation_energy] * Helpers::pow( 1.
          - exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance])),2)
            + params[energy_offset];
  return std::vector<result_t>(1, result);
}

PairPotential_Morse::derivative_components_t
PairPotential_Morse::derivative(
    const arguments_t &arguments
    ) const
{
  ASSERT( arguments.size() == 1,
      "PairPotential_Morse::operator() - requires exactly one argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Morse::operator() - types don't match with ones in arguments.");
  derivative_components_t result;
  const argument_t &r_ij = arguments[0];
  // Maple result: 2*D*(1-exp(-k*(r-R)))*k*exp(-k*(r-R))
  result.push_back(
      2. * params[dissociation_energy]
      * ( 1. - exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance])))
      * (- params[spring_constant]) * exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance]))
  );
  ASSERT( result.size() == 1,
      "PairPotential_Morse::operator() - we did not create exactly one component.");
  return result;
}

PairPotential_Morse::results_t
PairPotential_Morse::parameter_derivative(
    const arguments_t &arguments,
    const size_t index
    ) const
{
  ASSERT( arguments.size() == 1,
      "PairPotential_Morse::parameter_derivative() - requires exactly one argument.");
  ASSERT( ParticleTypeChecker::checkArgumentsAgainstParticleTypes(
      arguments, getParticleTypes()),
      "PairPotential_Morse::operator() - types don't match with ones in arguments.");
  const argument_t &r_ij = arguments[0];
  switch (index) {
    case spring_constant:
    {
      // Maple result: -2*D*(1-exp(-k*(r-R)))*(-r+R)*exp(-k*(r-R))
      const result_t result =
          - 2. * params[dissociation_energy]
          * ( 1. - exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance])))
          * (- r_ij.distance + params[equilibrium_distance])
          * exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance]))
          ;
      return std::vector<result_t>(1, result);
      break;
    }
    case equilibrium_distance:
    {
      // Maple result: -2*D*(1-exp(-k*(r-R)))*k*exp(-k*(r-R))
      const result_t result =
          - 2. * params[dissociation_energy]
          * ( 1. - exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance])))
          * params[spring_constant] * exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance]))
          ;
      return std::vector<result_t>(1, result);
      break;
    }
    case dissociation_energy:
    {
      // Maple result: (1-exp(-k*(r-R)))^2
      const result_t result =
          Helpers::pow(1.
              - exp( - params[spring_constant] * ( r_ij.distance - params[equilibrium_distance])),2);
      return std::vector<result_t>(1, result);
      break;
    }
    case energy_offset:
    {
      // Maple result: 1
      const result_t result = +1.;
      return std::vector<result_t>(1, result);
      break;
    }
    default:
      break;
  }
  return std::vector<result_t>(1, 0.);
}

FunctionModel::extractor_t
PairPotential_Morse::getFragmentSpecificExtractor(
    const charges_t &charges) const
{
  ASSERT(charges.size() == (size_t)2,
      "PairPotential_Morse::getFragmentSpecificExtractor() - requires 2 charges.");
  FunctionModel::extractor_t returnfunction =
      boost::bind(&Extractors::gatherDistancesFromFragment,
          boost::bind(&Fragment::getPositions, _1),
          boost::bind(&Fragment::getCharges, _1),
          boost::cref(charges),
          _2);
  return returnfunction;
}

void
PairPotential_Morse::setParametersToRandomInitialValues(
    const TrainingData &data)
{
  params[PairPotential_Morse::dissociation_energy] = 1e+0*rand()/(double)RAND_MAX;// 0.5;
  params[PairPotential_Morse::energy_offset] =
      data.getTrainingOutputAverage()[0];// -1.;
  params[PairPotential_Morse::spring_constant] = 1e+0*rand()/(double)RAND_MAX;// 1.;
  params[PairPotential_Morse::equilibrium_distance] =  3e+0*rand()/(double)RAND_MAX;//2.9;
}

