/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CommandLineParser.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/program_options/option.hpp>
#include <boost/program_options/value_semantic.hpp>
#include <boost/program_options.hpp>
#include <fstream>
#include <iostream>
#include <set>
#include <map>

#include "Actions/Action.hpp"
#include "Actions/ActionQueue.hpp"
#include "Actions/ActionTrait.hpp"
#include "Actions/OptionRegistry.hpp"
#include "Actions/OptionTrait.hpp"
#include "Actions/Values.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "CommandLineParser.hpp"
#include "CommandLineParser_validate.hpp"
#include "Parameters/Specifics/KeyValuePair.hpp"
#include "World.hpp"

#include "CodePatterns/Singleton_impl.hpp"

using namespace MoleCuilder;

class element;

/** Constructor of class CommandLineParser.
 *
 */
CommandLineParser::CommandLineParser() :
  analysis("Analysis options"),
  atom("Atom options"),
  bond("Bond options"),
  command("Command options"),
  fill("fill options"),
  shape("shape options"),
  fragmentation("Fragmentation options"),
  geometry("Geometry options"),
  graph("Graph options"),
  molecule("Molecule options"),
  options("Secondary options"),
  parser("Parser options"),
  potential("Potential options"),
  selection("Selection options"),
  tesselation("Tesselation options"),
  world("World options")
{
  // put all options lists into a lookup
  CmdParserLookup["analysis"] = &analysis;
  CmdParserLookup["atom"] = &atom;
  CmdParserLookup["bond"] = &bond;
  CmdParserLookup["command"] = &command;
  CmdParserLookup["edit"] = &edit;
  CmdParserLookup["fill"] = &fill;
  CmdParserLookup["shape"] = &shape;
  CmdParserLookup["fragmentation"] = &fragmentation;
  CmdParserLookup["geometry"] = &geometry;
  CmdParserLookup["graph"] = &graph;
  CmdParserLookup["options"] = &options;
  CmdParserLookup["molecule"] = &molecule;
  CmdParserLookup["parser"] = &parser;
  CmdParserLookup["potential"] = &potential;
  CmdParserLookup["selection"] = &selection;
  CmdParserLookup["tesselation"] = &tesselation;
  CmdParserLookup["world"] = &world;
}

/** Destructor of class CommandLineParser.
 *
 */
CommandLineParser::~CommandLineParser()
{}

/** Initializes command arguments to accept.
 * Goes through ActionRegistry and puts all actions therein into the map.
 */
void CommandLineParser::InitializeCommandArguments()
{
  // we need a list of already added options, otherwise we get ambigious exceptions
  std::set<std::string> AlreadyAddedOptionNames;

  bool ActionAlreadyAdded_flag = false;
  ActionQueue &AQ = ActionQueue::getInstance();
  ActionQueue::ActionTokens_t tokens = AQ.getListOfActions();
  for (ActionQueue::ActionTokens_t::const_iterator iter = tokens.begin();
      iter != tokens.end(); ++iter) {
    const ActionTrait &CurrentTrait = AQ.getActionsTrait(*iter);
    ActionAlreadyAdded_flag = false;
    //std::cout << "Current Action to initialize is: " << actioniter->first << std::endl;

    for (ActionTrait::options_const_iterator optioniter = CurrentTrait.getBeginIter();
        optioniter != CurrentTrait.getEndIter();
        ++optioniter) {
      if (optioniter->first == *iter)
        ActionAlreadyAdded_flag = true;
      ASSERT( OptionRegistry::getInstance().isOptionPresentByName(optioniter->first),
          "CommandLineParser::Init() - Option "+optioniter->first+" not present in OptionRegistry." );
      const OptionTrait* const currentOption = OptionRegistry::getInstance().getOptionByName(optioniter->first);

      if (AlreadyAddedOptionNames.find(optioniter->first) == AlreadyAddedOptionNames.end()) {
        // add the option
//        std::cout << "Registering Option "
//            << currentOption->getName()
//            << " with type '" << currentOption->getTypeName() << "' "
//            << " with description '" << currentOption->getDescription() << "' ";
//        if (currentOption->hasShortForm())
//          std::cout << ", with short form " << currentOption->getShortForm();
//        else
//          std::cout << ", with no short form ";
//        if (currentOption->hasDefaultValue())
//          std::cout << ", with default value " << currentOption->getDefaultValue();
//        else
//          std::cout << ", with no default value ";
//        std::cout << std::endl;

        AddOptionToParser(currentOption, (CmdParserLookup["options"]),
            (optioniter->first == *iter) ?
                true : false);

        AlreadyAddedOptionNames.insert(optioniter->first);
      } else {
//        std::cout << "Option " << currentOption->getName() << " already registered." << std::endl;
      }
    }

    if (!ActionAlreadyAdded_flag) {
      // add the action
//      std::cout << "Registering Action "
//          << currentAction->Traits.getName()
//          << " in menu " << currentAction->Traits.getMenuName()
//          << " with type '" << currentAction->Traits.getTypeName() << "' "
//          << " with description '" << currentAction->Traits.getDescription() << "' ";
//      if (currentAction->Traits.hasShortForm())
//        std::cout << ", with short form " << currentAction->Traits.getShortForm();
//      else
//        std::cout << ", with no short form ";
//      if (currentAction->Traits.hasDefaultValue())
//        std::cout << ", with default value " << currentAction->Traits.getDefaultValue();
//      else
//        std::cout << ", with no default value ";
//      std::cout  << std::endl;

      ASSERT(CmdParserLookup.find(CurrentTrait.getMenuName()) != CmdParserLookup.end(),
          "CommandLineParser: boost::program_options::options_description for this Action not present.");
      AddOptionToParser(
          dynamic_cast<const OptionTrait * const>(&CurrentTrait),
          (CmdParserLookup[CurrentTrait.getMenuName()]),
          false);
    }
  }
  // note: positioning is not important on the command line
}

/**
 * This is a fix by Benedek Thaler given on the boost mailing list to get back
 * to the old behavior before the breaking change in 1.63 for
 * boost::program_options.
 *
 * The problem is that for an implicit value "-a foo" will not recognize the
 * value for "a". It now needs to read "-a=foo" to store the value "foo" for
 * the action "a". This broke our behavior for all action that have same-named
 * options.
 *
 * See https://lists.boost.org/Archives/boost/2017/01/232272.php
 */
#if BOOST_VERSION >= 106300
template <typename T>
struct greedy_implicit_value : public po::typed_value<T>
{
  using base = po::typed_value<T>;

  greedy_implicit_value(const T& value) : base(nullptr)
  {
    base::implicit_value(value);
  }

  bool adjacent_tokens_only() const override { return false; }
unsigned max_tokens() const override { return 1; }
};

template <typename T>
po::typed_value<T>* implicit_value(const T& value)
{
  return new greedy_implicit_value<T>(value);
}
#else
template <typename T>
po::typed_value<T>* implicit_value(const T& value)
{
  return new po::typed_value<T>()->implicit_value(value);
}
#endif


/** Adds an Action or Option to the CommandLineParser.
 * Note that Action is derived from Option(Trait)
 *
 * This ugly switch function is necessary because of the compile-time problem:
 * po::value<T> has to be instantiated at compile-time however we do know the type not until run-time.
 * Not even a templated function like po::value<T> getProgramOptionValuefromType() does help, specialized
 * to each available type, as the signatures of all the functions differ. Hence, they cannot not put into
 * one type_info -> po::value<T> map ...
 *
 * \param *currentOption pointer to Action/Option to add
 * \param *OptionList program_options list to add to
 * \param _DefaultAsImplicit whether to add a default value as default_value or
 *        as implicit_value (default = option token present or not, implicit =
          option token present but not necessarily followed by argument)
 */
void CommandLineParser::AddOptionToParser(
    const OptionTrait * const currentOption, 
    po::options_description* OptionList,
    const bool _DefaultAsImplicit)
{
  // check whether dynamic_cast in Init() suceeded
  ASSERT(currentOption != NULL, "CommandLineParser::AddOptionToParser() - currentOption is NULL!");
  // add other options
//  std::cout << "Adding Action " << currentOption->getName() << " with type "
//      << currentOption->getType()->name() << ", " << (currentOption->hasDefaultValue() ? "with" : "without")
//      << " default value, and KeyandShortform " << currentOption->getKeyAndShortForm()
//      << " to CommandLineParser." << std::endl;
  switch(TypeToEnums.getEnumforType(currentOption->getType())) {
    default:
    case TypeEnumContainer::NoneType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(), currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::BooleanType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  (_DefaultAsImplicit ? 
                      implicit_value(bool(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str()))) :
                      po::value < bool >()->default_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str()))) :
                  po::value < bool >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::FileType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < boost::filesystem::path >()->default_value(boost::lexical_cast<boost::filesystem::path>(currentOption->getDefaultValue().c_str())) :
                  po::value < boost::filesystem::path >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfFilesType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<boost::filesystem::path> >()->default_value(boost::lexical_cast< std::vector<boost::filesystem::path> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<boost::filesystem::path> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::IntegerType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  (_DefaultAsImplicit ? 
                      implicit_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str())) :
                      po::value < int >()->default_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str()))) :
                  po::value < int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfIntegersType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<int> >()->default_value(boost::lexical_cast< std::vector<int> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::UnsignedIntegerType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  (_DefaultAsImplicit ? 
                      implicit_value(boost::lexical_cast<unsigned int>(currentOption->getDefaultValue().c_str())) :
                      po::value < unsigned int >()->default_value(boost::lexical_cast<unsigned int>(currentOption->getDefaultValue().c_str()))) :
                  po::value < unsigned int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfUnsignedIntegersType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<unsigned int> >()->default_value(boost::lexical_cast< std::vector<unsigned int> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<unsigned int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::DoubleType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  (_DefaultAsImplicit ? 
                      implicit_value(boost::lexical_cast<double>(currentOption->getDefaultValue().c_str())) :
                      po::value < double >()->default_value(boost::lexical_cast<double>(currentOption->getDefaultValue().c_str()))) :
                  po::value < double >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfDoublesType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<double> >()->default_value(boost::lexical_cast< std::vector<double> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<double> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::StringType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  (_DefaultAsImplicit ? 
                      implicit_value(currentOption->getDefaultValue()) :
                      po::value < std::string >()->default_value(currentOption->getDefaultValue())) :
                  po::value < std::string >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfStringsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<std::string> >()->default_value(boost::lexical_cast< std::vector<std::string> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<std::string> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::VectorType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < VectorValue >()->default_value(boost::lexical_cast<VectorValue>(currentOption->getDefaultValue().c_str())) :
                  po::value < VectorValue >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfVectorsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<VectorValue> >()->default_value(boost::lexical_cast< std::vector<VectorValue> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<VectorValue> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::MoleculeType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < const molecule * >()->default_value(boost::lexical_cast<const molecule *>(currentOption->getDefaultValue().c_str())) :
                  po::value < int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfMoleculesType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<const molecule *> >()->default_value(boost::lexical_cast< std::vector<const molecule *> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::AtomType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  (_DefaultAsImplicit ? 
                      implicit_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str())) :
                      po::value < int >()->default_value(boost::lexical_cast<int>(currentOption->getDefaultValue().c_str()))) :
                  po::value < int >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfAtomsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<const atom *> >()->default_value(boost::lexical_cast< std::vector<const atom *> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<int> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ElementType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < const element * >()->default_value(boost::lexical_cast<const element *>(currentOption->getDefaultValue().c_str())) :
                  po::value < std::string >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfElementsType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<const element *> >()->default_value(boost::lexical_cast< std::vector<const element *> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<std::string> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::RealSpaceMatrixType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < RealSpaceMatrixValue >()->default_value(boost::lexical_cast<BoxValue>(currentOption->getDefaultValue().c_str())) :
                  po::value < RealSpaceMatrixValue >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::KeyValueType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
            currentOption->hasDefaultValue() ?
                  (_DefaultAsImplicit ?
                      implicit_value(boost::lexical_cast< KeyValuePair >(currentOption->getDefaultValue().c_str())) :
                      po::value < KeyValuePair >()->default_value(boost::lexical_cast< KeyValuePair >(currentOption->getDefaultValue().c_str()))) :
                  po::value < KeyValuePair >(),
                  currentOption->getDescription().c_str())
        ;
      break;
    case TypeEnumContainer::ListOfKeyValuesType:
      OptionList->add_options()
        (currentOption->getKeyAndShortForm().c_str(),
//            currentOption->hasDefaultValue() ?
//                  po::value < std::vector<KeyValuePair> >()->default_value(boost::lexical_cast< std::vector<KeyValuePair> >(currentOption->getDefaultValue().c_str())) :
                  po::value < std::vector<KeyValuePair> >()->multitoken(),
                  currentOption->getDescription().c_str())
        ;
      break;
  }
}

/** States whether there are command line arguments.
 * \return true - there are none, false - there is at least one command line argument
 */
bool CommandLineParser::isEmpty()
{
  return vm.empty();
}

/** Sets the options.
 * \param _argc arg count from main()
 * \param **_argv argument array from main()
 */
void CommandLineParser::setOptions(int _argc, char **_argv)
{
  argc = _argc;
  argv = _argv;
  config_file_options.add(options);
  // append all option_descriptions to both cmdline_options and visible
  for (CmdParserLookupMap::iterator iter = CmdParserLookup.begin();
      iter != CmdParserLookup.end();
      ++iter) {
    cmdline_options.add(*(iter->second));
    visible.add(*(iter->second));
  }
}

/** This is due to the answer by Aleksey Vitebskiy
 * in http://stackoverflow.com/questions/4107087/accepting-negative-doubles-with-boostprogram-options
 *
 */
std::vector<po::option> ignore_numbers(std::vector<std::string>& args)
{
    std::vector<po::option> result;
    int pos = 0;
    while(!args.empty()) {
        const std::string& arg = args[0];
        bool isNumber = true;
        try {
          boost::lexical_cast<double>(arg);
        } catch(boost::bad_lexical_cast) {
          isNumber = false;
        }
        if (isNumber) {
            result.push_back(po::option());
            po::option& opt = result.back();

            opt.position_key = pos++;
            opt.value.push_back(arg);
            opt.original_tokens.push_back(arg);

            args.erase(args.begin());
        } else {
            break;
        }
    }

    return result;
}

/** Parses the command line arguments.
 * Calls program_options::store() and program_options::notify()
 *
 * @return true - all is ok, false - command-line options could not be parsed 
 *         correctly
 */
bool CommandLineParser::Parse()
{
  bool status = true;
  try {
    po::store(po::command_line_parser(argc,argv).extra_style_parser(&ignore_numbers).options(cmdline_options).run(), vm);
  } catch (std::exception &e) {
    std::cerr << "Something went wrong with parsing the command-line arguments: " 
        << e.what() << std::endl;
    World::getInstance().setExitFlag(134);
#ifdef HAVE_ACTION_THREAD
    // force action queue to stop thread
    ActionQueue::getInstance().stop();
    ActionQueue::getInstance().clearTempQueue();
#endif
    ActionQueue::getInstance().clearQueue();
    status = false;
  }
  if (status) {
    std::ifstream input;
    input.open("example.cfg");
    if (!input.fail())
      po::store(po::parse_config_file(input, config_file_options), vm);
    input.close();
    po::notify(vm);
  }
  return status;
}

/** Scan the argument list for -a or --arguments and store their order for later use.
 */
void CommandLineParser::scanforSequenceOfArguments()
{
  std::map <std::string, std::string> ShortFormToActionMap = getShortFormToActionMap();
  LOG(0, "Scanning command line arguments and recognizing Actions.");
  // go through all arguments
  for (int i=1;i<argc;i++) {
    LOG(2, "Checking on " << argv[i]);
    // check whether they
    if (argv[i][0] == '-') { // .. begin with -
      LOG(2, "Possible argument: " << argv[i]);
      if (argv[i][1] == '-') { // .. or --
        LOG(1, "Putting " << argv[i] << " into the sequence.");
        std::vector<std::string> fields;
        boost::split(fields, argv[i], boost::is_any_of("="));
        SequenceOfActions.push_back(&(fields[0][2]));
        //  .. and check that next letter is not numeric, if so insert
      } else if (((argv[i][1] < '0') || (argv[i][1] > '9')) && ((argv[i][1] != '.'))) {
        std::map <std::string, std::string>::iterator iter = ShortFormToActionMap.find(&(argv[i][1]));
        if (iter != ShortFormToActionMap.end()) {
          LOG(1, "Putting " << iter->second << " for " << iter->first << " into the sequence.");
          SequenceOfActions.push_back(iter->second);
        }
      }
    }
  }
}

/** Makes the Parser parse the command line options with current known options.
 * \param _argc arg count from main()
 * \param **_argv argument array from main()
 */
void CommandLineParser::Run(int _argc, char **_argv)
{
  setOptions(_argc,_argv);
  const bool status = Parse();
  if (status)
    scanforSequenceOfArguments();
}

/** Go through all Actions and create a map from short form to their token.
 * \return map from Action's ShortForm to token.
 */
std::map <std::string, std::string> CommandLineParser::getShortFormToActionMap() const
{
  std::map <std::string, std::string> result;

  ActionQueue &AQ = ActionQueue::getInstance();
  ActionQueue::ActionTokens_t tokens = AQ.getListOfActions();
  for (ActionQueue::ActionTokens_t::const_iterator iter = tokens.begin();
      iter != tokens.end(); ++iter) {
    const ActionTrait &CurrentTrait = AQ.getActionsTrait(*iter);
    if (CurrentTrait.hasShortForm()) {
      ASSERT(result.find(CurrentTrait.getShortForm()) == result.end(),
          "Short form "+toString(CurrentTrait.getShortForm())+
          " for action "+toString(*iter)+" already present from "+
          std::string(result[CurrentTrait.getShortForm()])+"!");
      result[CurrentTrait.getShortForm()] = *iter;
    }
  }

  return result;
}

CONSTRUCT_SINGLETON(CommandLineParser)
