/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * MenuDescription.cpp
 *
 *  Created on: Oct 26, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <iostream>
#include <map>
#include <string>

#include "Actions/ActionRegistry.hpp"
#include "Menu/MenuDescription.hpp"

/** Constructor of class MenuDescription.
 *
 */
MenuDescription::MenuDescription()
{
  // put each menu into its place, "" means top level
  MenuPositionMap["analysis"] = TopPosition("tools",1);
  MenuPositionMap["atom"] = TopPosition("edit",1);
  MenuPositionMap["command"] = TopPosition("",3);
  MenuPositionMap["edit"] = TopPosition("",2);
  MenuPositionMap["fragmentation"] = TopPosition("tools",3);
  MenuPositionMap["molecule"] = TopPosition("edit",2);
  MenuPositionMap["parser"] = TopPosition("edit",3);
  MenuPositionMap["selection"] = TopPosition("edit",4);
  MenuPositionMap["tesselation"] = TopPosition("tools",2);
  MenuPositionMap["tools"] = TopPosition("",4);
  MenuPositionMap["world"] = TopPosition("",1);

  // put menu description into each menu category
  MenuDescriptionsMap["analysis"] = "Analysis (pair correlation, volume)";
  MenuDescriptionsMap["atom"] = "Edit atoms";
  MenuDescriptionsMap["command"] = "Configuration";
  MenuDescriptionsMap["edit"] = "Edit";
  MenuDescriptionsMap["fragmentation"] = "Fragmentation";
  MenuDescriptionsMap["molecule"] = "Parse files into system";
  MenuDescriptionsMap["parser"] = "Edit molecules (load, parse, save)";
  MenuDescriptionsMap["selection"] = "Select atoms/molecules";
  MenuDescriptionsMap["tesselation"] = "Tesselate molecules";
  MenuDescriptionsMap["tools"] = "Various tools";
  MenuDescriptionsMap["world"] = "Edit world";

  // put menu name into each menu category
  MenuNameMap["analysis"] = "Analysis";
  MenuNameMap["atom"] = "Atoms";
  MenuNameMap["command"] = "Configuration";
  MenuNameMap["edit"] = "Edit";
  MenuNameMap["fragmentation"] = "Fragmentation";
  MenuNameMap["molecule"] = "Molecules";
  MenuNameMap["parser"] = "Input/Output";
  MenuNameMap["selection"] = "Selection";
  MenuNameMap["tesselation"] = "Tesselation";
  MenuNameMap["tools"] = "Tools";
  MenuNameMap["world"] = "Globals";
}

/** Destructor of class MenuDescription.
 *
 */
MenuDescription::~MenuDescription()
{}

/** Getter for MenuDescriptionsMap.
 * \param token name of menu
 * \return description string of the menu or empty
 */
const std::string MenuDescription::getDescription(const std::string &token) const
{
  if (MenuDescriptionsMap.find(token) != MenuDescriptionsMap.end())
    return MenuDescriptionsMap.find(token)->second;
  else
    return std::string();
}

/** Getter for MenuNameMap.
 * \param token name of menu
 * \return description string of the menu or empty
 */
const std::string MenuDescription::getName(const std::string &token) const
{
  if (MenuNameMap.find(token) != MenuNameMap.end())
    return MenuNameMap.find(token)->second;
  else
    return std::string();
}

/** Constructs a multimap of all menus running over all actions belonging to it.
 * \return multimap with which actions belongs to which menu.
 */
std::multimap <std::string, std::string> MenuDescription::getMenuItemsMap() const
{
  std::multimap <std::string, std::string> result;

  ActionRegistry &AR = ActionRegistry::getInstance();
  for (ActionRegistry::const_iterator iter = AR.getBeginIter();iter != AR.getEndIter();++iter) {
    std::cout << "Inserting " << (iter->second)->getName() << " into menu " << (iter->second)->Traits.getMenuName() << std::endl;
    result.insert( std::pair<std::string, std::string> ((iter->second)->Traits.getMenuName(), (iter->second)->getName()));
  }
  // TODO: MenuPosition is not yet realized.
  return result;
}

/** Forward iterator from beginning of list of descriptions.
 * \return iterator
 */
MenuDescription::iterator MenuDescription::getBeginIter()
{
  return MenuPositionMap.begin();
}

/** Forward iterator at end of list of descriptions.
 * \return iterator
 */
MenuDescription::iterator MenuDescription::getEndIter()
{
  return MenuPositionMap.end();
}

/** Constant forward iterator from beginning of list of descriptions.
 * \return constant iterator
 */
MenuDescription::const_iterator MenuDescription::getBeginIter() const
{
  return MenuPositionMap.begin();
}

/** Constant forward iterator at end of list of descriptions.
 * \return constant iterator
 */
MenuDescription::const_iterator MenuDescription::getEndIter() const
{
  return MenuPositionMap.end();
}


