/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * InterfaceVMGJob.cpp
 *
 *  Created on: 10.06.2012
 *      Author: Frederik Heber
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <cmath>
#include <iostream>

#include "CodePatterns/Log.hpp"

#include "base/vector.hpp"
#include "grid/grid.hpp"
#include "grid/multigrid.hpp"

#include "InterfaceVMGJob.hpp"

using namespace VMG;
using VMGInterfaces::InterfaceVMGJob;

void InterfaceVMGJob::ImportRightHandSide(Multigrid& multigrid)
{
  Index i;
  Vector pos;

//  VMG::TempGrid *temp_grid = new VMG::TempGrid(129, 0, 0., 1.);

  Grid& grid = multigrid(multigrid.MaxLevel());
  grid.ClearBoundary();

  const Index begin_local = grid.Global().LocalBegin() - grid.Local().HaloSize1();

  LOG(1, "INFO: Mesh has extent " << grid.Extent().MeshWidth() << ".");
  const int gridpoints = pow(2, level);
  LOG(1, "INFO: gridpoints on finest level are " << gridpoints << ".");
  assert( (grid.Extent().MeshWidth().X() * gridpoints) == 1 );
  assert( (grid.Extent().MeshWidth().Y() * gridpoints) == 1 );
  assert( (grid.Extent().MeshWidth().Z() * gridpoints) == 1 );
  LOG(1, "INFO: "
      << "X in [" << grid.Local().Begin().X() << "," << grid.Local().End().X() << "],"
      << "Y in [" << grid.Local().Begin().Y() << "," << grid.Local().End().Y() << "],"
      << "Z in [" << grid.Local().Begin().Z() << "," << grid.Local().End().Z() << "].");

  std::vector<double>::const_iterator sample_iter = sampled_input.begin();
  for (i.X()=grid.Local().Begin().X(); i.X()<grid.Local().End().X(); ++i.X())
    for (i.Y()=grid.Local().Begin().Y(); i.Y()<grid.Local().End().Y(); ++i.Y())
      for (i.Z()=grid.Local().Begin().Z(); i.Z()<grid.Local().End().Z(); ++i.Z()) {
        pos = grid.Extent().MeshWidth() * static_cast<Vector>(begin_local + i);
//        R.x() = pos.X();
//        R.y() = pos.Y();
//        R.z() = pos.Z();
        grid(i) = *sample_iter; //temp_grid(i);
        ++sample_iter;
      }
  assert( sample_iter == sampled_input.end() );

  Grid::iterator grid_iter;
  double charge_sum = 0.0;
  for (grid_iter=grid.Iterators().Local().Begin(); grid_iter!=grid.Iterators().Local().End(); ++grid_iter)
    charge_sum += grid.GetVal(*grid_iter);
//  charge_sum = MG::GetComm()->GlobalSum(charge_sum);
//  comm.PrintStringOnce("Grid charge sum: %e", charge_sum);
  std::cout << "Grid charge sum: " << charge_sum << std::endl;

//  delete temp_grid;
}

void InterfaceVMGJob::ExportSolution(Grid& grid)
{
  // grid now contains the sough-for potential

  const Index begin_local = grid.Global().LocalBegin() - grid.Local().HaloSize1();
  Index i;

  for (i.X()=grid.Local().Begin().X(); i.X()<grid.Local().End().X(); ++i.X())
    for (i.Y()=grid.Local().Begin().Y(); i.Y()<grid.Local().End().Y(); ++i.Y())
      for (i.Z()=grid.Local().Begin().Z(); i.Z()<grid.Local().End().Z(); ++i.Z()) {
        grid(i);
      }

  Grid::iterator grid_iter;
  double potential_sum = 0.0;
  for (grid_iter=grid.Iterators().Local().Begin(); grid_iter!=grid.Iterators().Local().End(); ++grid_iter)
    potential_sum += grid.GetVal(*grid_iter);
//  charge_sum = MG::GetComm()->GlobalSum(potential_sum);
//  comm.PrintStringOnce("Grid potential sum: %e", potential_sum);
  std::cout << "Grid potential sum: " << potential_sum << std::endl;

  //Particle::CommMPI& comm = *dynamic_cast<Particle::CommMPI*>(MG::GetComm());
  //e_long = comm.GlobalSumRoot(e_long);

}
