/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2011 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CommandLineParser_validate.cpp
 *
 *  Created on: Nov 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <boost/version.hpp>

#include <iostream>
#include <string>

#include "Actions/Values.hpp"
#include "CommandLineParser_validate.hpp"

/** boost::program_options validator specialization for VectorValue.
 * \param &v  reference for return value
 * \param &values string vector of scanned options
 * \param *
 * \param
 *
 */
void validate(boost::any& v, const std::vector<std::string>& values, VectorValue *, int)
{
  VectorValue VV;
  std::vector<std::string> components;

  // split comma-separated values
  if (values.size() != 1) {
    std::cerr <<  "Not one vector but " << values.size() << " given " << std::endl;
#if BOOST_VERSION < 104200
    throw boost::program_options::validation_error("Unequal to one vector given");
#else
    throw boost::program_options::validation_error(
        boost::program_options::validation_error::invalid_option_value,
        std::string("value"),
        std::string("VectorValue")
    );
#endif
  }
  std::string argument(values.at(0));
  std::string::iterator Aiter = argument.begin();
  std::string::iterator Biter = argument.begin();
  for (; Aiter != argument.end(); ++Aiter) {
    if (*Aiter == ',') {
      components.push_back(std::string(Biter,Aiter));
      do {
        Aiter++;
      } while (*Aiter == ' ' || *Aiter == '\t');
      Biter = Aiter;
    }
  }
  components.push_back(std::string(Biter,argument.end()));

  if (components.size() != 3) {
    std::cerr <<  "Specified vector does not have three components but " << components.size() << std::endl;
#if BOOST_VERSION < 104200
    throw boost::program_options::validation_error("Specified vector does not have three components");
#else
    throw boost::program_options::validation_error(
        boost::program_options::validation_error::invalid_option_value,
        std::string("value"),
        std::string("VectorValue")
    );
#endif
  }
  for (size_t i=0;i<NDIM;++i)
    VV.vector[i] = boost::lexical_cast<double>(components.at(i));
  v = boost::any(VectorValue(VV));
}

void validate(boost::any& v, const std::vector<std::string>& values, BoxValue *, int)
{
  BoxValue BV;
  std::vector<std::string> components;

  // split comma-separated values
  if (values.size() != 1) {
    std::cerr <<  "Not one vector but " << values.size() << " given " << std::endl;
#if BOOST_VERSION < 104200
    throw boost::program_options::validation_error("Unequal to one vector given");
#else
    throw boost::program_options::validation_error(
        boost::program_options::validation_error::invalid_option_value,
        std::string("value"),
        std::string("BoxValue")
    );
#endif
  }
  std::string argument(values.at(0));
  std::string::iterator Aiter = argument.begin();
  std::string::iterator Biter = argument.begin();
  for (; Aiter != argument.end(); ++Aiter) {
    if (*Aiter == ',') {
      components.push_back(std::string(Biter,Aiter));
      do {
        Aiter++;
      } while (*Aiter == ' ' || *Aiter == '\t');
      Biter = Aiter;
    }
  }
  components.push_back(std::string(Biter,argument.end()));

  if (components.size() != 6) {
    std::cerr <<  "Specified vector does not have three components but " << components.size() << std::endl;
#if BOOST_VERSION < 104200
    throw boost::program_options::validation_error("Specified symmetric box matrix does not have six components");
#else
    throw boost::program_options::validation_error(
        boost::program_options::validation_error::invalid_option_value,
        std::string("value"),
        std::string("BoxValue")
    );
#endif
  }
  for (size_t i=0;i<(NDIM*(NDIM+1))/2; ++i)
    BV.matrix[i] = boost::lexical_cast<double>(components.at(i));
  v = boost::any(BoxValue(BV));
}

/** boost::program_options validator specialization for boost::filesystem::path.
 * \param &v  reference for return value
 * \param &values string vector of scanned options
 * \param *
 * \param
 *
 */
void validate(boost::any& v, const std::vector<std::string>& values, boost::filesystem::path *, int)
{
  boost::filesystem::path filename;

  std::cerr << "boost::filesystem::path validator used." << std::endl;

  // split comma-separated values
  if (values.size() != 1) {
    std::cerr <<  "Not one file but " << values.size() << " given " << std::endl;
#if BOOST_VERSION < 104200
    throw boost::program_options::validation_error("Unequal to one file given");
#else
    if (values.size() == 0) {
      throw boost::program_options::validation_error(
          boost::program_options::validation_error::at_least_one_value_required,
          std::string("value"),
          std::string("boost::filesystem::path")
      );
    } else {
      throw boost::program_options::validation_error(
          boost::program_options::validation_error::multiple_values_not_allowed,
          std::string("value"),
          std::string("boost::filesystem::path")
      );
    }
#endif
  }
  filename = values.at(0);
  v = boost::any(boost::filesystem::path(filename));
}

