/*
 * CommandLineParser.hpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */

#ifndef COMMANDLINEPARSER_HPP_
#define COMMANDLINEPARSER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <boost/program_options.hpp>

namespace po = boost::program_options;

#include "CodePatterns/Singleton.hpp"
#include "UIElements/CommandLineUI/TypeEnumContainer.hpp"

#include <map>

class Action;
class CommandLineParser_ActionRegistry_ConsistenyTest;
class OptionTrait;

/** This class is a wrapper for boost::program_options.
 *
 * <h1> CommandLine Howto </h1>
 * <h2> Introduction </h2>
 *
 * The UIFactory is a base class for the User Interaction. There are three UI specializations:
 * Text, GUI and CommandLine. Accessing functionality via the CommandLine UI is explained here.
 *
 * First, an Action has to be written for the specific functionality. This Action should
 * be added in Actions/...Action in the respective subdirectory of the following types:
 *  -# Analysis: Analysis actions like evaluating pair correlation, bonds, ...
 *  -# Atom: adding, removing, manipulating atoms
 *  -# Cmd: specifying data bases, verbosity, ...
 *  -# Fragmentation: fragmenting a system, performing graph analysis, ...
 *  -# Molecule: adding, removing, manipulating molecules
 *  -# Parser: Parsing files (loading, saving)
 *  -# Tesselation: obtaining (non)convex surface of a molecule, embedding, ...
 *  -# World: Setting Box dimensions, default name of new molecules, ...
 *
 *  The CommandLineUIFactory is a specialization of the UIFactory for parsing command
 *  line parameters, generating and executing actions there from.
 *
 *  The idea of the CommandLineFactory is explained elsewhere, here we would like to give a
 *  receipe for creating new actions.
 *
 * <h3>Introducing new actions</h3>
 *
 * Let us now introduce what to do if a new action is to be implemented. Here, we use the
 * CommandLineVersionAction as an example.
 * This consists if basically three parts:
 * 1. Create the files, write the classes and make them compilable
 *   - Create new source and header files in one of the above subfolders in the Actions folder,
 *     e.g. create VersionAction.cpp and VersionAction.hpp in Actions/Cmd/
 *   - Give it a sensible class name, the convention is <type><what it does>Action,
 *     where <type> is basically the naming (written out) of the subdirectory,
 *     e.g. class CommandLineVersionAction.
 *   - Add the source and header file to the respective variables in molecuilder/src/Makefile.am,
 *     e.g. if you add a Cmd action the variables are CMDACTIONSOURCE and CMDACTIONHEADER,
 *     such that they get compiled.
 * 2. Add an instance to the CommandLineUIFactory, such that they are known to the UI.
 *   - Add the header file as an include to UIElements/CommandLineWindow.cpp, e.g.
 *     #include "Actions/Cmd/VersionAction.hpp"
 *   - Add an instance of your class to the specific populater-function in
 *     UIElements/CommandLineWindow.cpp, e.g. for the above Cmd action, add to populateCommandActions()
 *     add new CommandLineVersionAction().
 *     This will automatically register in the ActionRegistry.
 * 3. Give them an option name, short hand an description, such that they can be referenced from
 *    the command line.
 *   - think of a new key name, e.g. "version", which is the long form of the command parameter,
 *     i.e. --version).
 *   - add this key to every map of MapofActions, i.e. to
 *     - MapofActions::DescriptionMap: the description which appears as help and tooltip
 *     - MapofActions::ShortFormMap: the short form of the command parameter (e.g. -v)
 *     - MapofActions::ValueMap: the value the command parameter has (do not create if it does not need one)
 *   - If your action requires additional parameters, these need to be added in the same manner as in
 *     the list item above.
 *
 *  Don't forget to write the actual code. :)
 *
 * <h3>Writing an action</h3>
 *
 *  As you write a new action you may think in terms of the command line, i.e. you want to use this
 *  new functionality you add by calling molecuilder as: ./molecuilder --super-action foobar.txt, where
 *  the key of your new action would be "super-action". While this is fine, keep in mind, that your action
 *  should be useable for the other UI specializations as well, i.e. from the menu and the GUI. Therefore,
 *  -# Don't use cin to ask the user for input: Use Query...()!
 *  -# Rather don't use cout/cerrs, but either give Log() or eLog() or use QueryEmpty() if you want to give
 *     the user specific information what you ask of him.
 *
 */
class CommandLineParser : public Singleton<CommandLineParser> {
  friend class Singleton<CommandLineParser>;

  //!> test needs to access CmdParserLookup to see whether menus are missing
  friend class CommandLineParser_ActionRegistry_ConsistenyTest;
public:

  // Parses the command line arguments in CommandLineParser::**argv with currently known options.
  void Run(int _argc, char **_argv);

  // Initialises all options from ActionRegistry.
  void InitializeCommandArguments();

  // Checks whether there have been any commands on the command line.
  bool isEmpty();

  /* boost's program_options are sorted into three categories:
   * -# generic options: option available to both command line and config
   * -# config options: only available in the config file
   * -# hidden options: options which the user is not shown on "help"
   */
  po::options_description analysis;
  po::options_description atom;
  po::options_description command;
  po::options_description edit;
  po::options_description fragmentation;
  po::options_description graph;
  po::options_description molecule;
  po::options_description options;
  po::options_description parser;
  po::options_description selection;
  po::options_description tesselation;
  po::options_description world;

  po::options_description visible;

  po::variables_map vm;

  // private sequence of actions as they appeared on the command line
  std::list<std::string> SequenceOfActions;

private:
  // private constructor and destructor
  CommandLineParser();
  virtual ~CommandLineParser();

  /* The following program_options options_decriptions are used to
   * generate the various cases and call differently in Parse().
   */
  po::options_description cmdline_options;
  po::options_description config_file_options;

  // adds options to the parser
  void AddOptionToParser(const OptionTrait * const currentOption, po::options_description* OptionList);

  // creates a map from short forms to action tokens needed to parse command line
  std::map <std::string, std::string> getShortFormToActionMap() const;

  typedef std::map< std::string , po::options_description *> CmdParserLookupMap;

  // lookup list from "configmenus" to the ones of CommandLineParser
  CmdParserLookupMap CmdParserLookup;

  // Sets the options from the three cases.
  void setOptions(int _argc, char **_argv);

  // Parses all options from command line and config file
  void Parse();

  // as boost's program_options does not care about of order of appearance but we do for actions,
  // we have to have a list and a function to obtain it.
  void scanforSequenceOfArguments();

  TypeEnumContainer TypeToEnums;

  // argument counter and array passed on from main()
  int argc;
  char **argv;
};

#endif /* COMMANDLINEPARSER_HPP_ */
