/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FillVoidWithMoleculeAction.cpp
 *
 *  Created on: May 10, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "atom.hpp"
#include "Graph/BondGraph.hpp"
#include "boundary.hpp"
#include "CodePatterns/Verbose.hpp"
#include "config.hpp"
#include "Descriptors/MoleculeIdDescriptor.hpp"
#include "Descriptors/MoleculeOrderDescriptor.hpp"
#include "molecule.hpp"
#include "Parser/FormatParserStorage.hpp"
#include "World.hpp"


#include <algorithm>
#include <iostream>
#include <string>

#include "Actions/MoleculeAction/FillVoidWithMoleculeAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "FillVoidWithMoleculeAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr MoleculeFillVoidWithMoleculeAction::performCall() {
  // obtain information
  getParametersfromValueStorage();

  if (!boost::filesystem::exists(params.fillername)) {
    DoeLog(1) && (eLog() << Verbose(1) << "File with filler molecule " << params.fillername << " does not exist!" << endl);
    return Action::failure;
  }

  DoLog(1) && (Log() << Verbose(1) << "Filling Box with water molecules, "
      << " minimum distance to molecules" << params.boundary
      << ", random atom displacement " << params.RandAtomDisplacement
      << ", random molecule displacement " << params.RandMoleculeDisplacement
      << ", distances between fillers (" << params.distances[0] << "," << params.distances[1] << "," << params.distances[2]
      << "), MinDistance " << params.MinDistance
      << ", DoRotate " << params.DoRotate << "." << endl);
  // construct water molecule
  std::vector<molecule *> presentmolecules = World::getInstance().getAllMolecules();
//  DoLog(0) && (Log() << Verbose(0) << presentmolecules.size() << " molecules initially are present." << std::endl);
  std::string FilenameSuffix = params.fillername.string().substr(params.fillername.string().find_last_of('.')+1, params.fillername.string().length());
  ifstream input;
  input.open(params.fillername.string().c_str());
  ParserTypes type = FormatParserStorage::getInstance().getTypeFromSuffix(FilenameSuffix);
  FormatParser &parser = FormatParserStorage::getInstance().get(type);
  parser.load(&input);

  // search the filler molecule that has been just parsed
  molecule *filler = World::getInstance().getMolecule(MoleculeByOrder(-1)); // get last molecule
  filler->SetNameFromFilename(params.fillername.string().c_str());
  molecule::atomVector Set = filler->getAtomSet();
  World::getInstance().getBondGraph()->CreateAdjacency(Set);

  // call routine
  double distance[NDIM];
  for (int i=0;i<NDIM;i++)
    distance[i] = params.distances[i];
  FillVoidWithMolecule(
      filler,
      *(World::getInstance().getConfig()),
      distance,
      params.boundary,
      params.RandAtomDisplacement,
      params.RandMoleculeDisplacement,
      params.MinDistance,
      params.DoRotate);

  // generate list of newly created molecules
  // (we can in general remove more quickly from a list than a vector)
  std::vector<molecule *> fillermolecules = World::getInstance().getAllMolecules();
//  DoLog(0) && (Log() << Verbose(0) << fillermolecules.size() << " molecules are present." << std::endl);
  std::list<molecule *> fillermolecules_list;
  std::copy( fillermolecules.begin(),  fillermolecules.end(), std::back_inserter( fillermolecules_list ));
//  DoLog(0) && (Log() << Verbose(0) << fillermolecules_list.size() << " molecules have been copied." << std::endl);
  for (std::vector<molecule *>::const_iterator iter = presentmolecules.begin();
      iter != presentmolecules.end();
      ++iter) {
    fillermolecules_list.remove(*iter);
  }
//  DoLog(0) && (Log() << Verbose(0) << fillermolecules_list.size() << " molecules left after removal." << std::endl);
  fillermolecules.clear();
  std::copy(fillermolecules_list.begin(), fillermolecules_list.end(), std::back_inserter( fillermolecules ));

//  DoLog(0) && (Log() << Verbose(0) << fillermolecules.size() << " molecules have been inserted." << std::endl);

  return Action::state_ptr(new MoleculeFillVoidWithMoleculeState(fillermolecules,params));
}

Action::state_ptr MoleculeFillVoidWithMoleculeAction::performUndo(Action::state_ptr _state) {
  MoleculeFillVoidWithMoleculeState *state = assert_cast<MoleculeFillVoidWithMoleculeState*>(_state.get());

  MoleculeListClass *MolList = World::getInstance().getMolecules();

  BOOST_FOREACH(molecule *_mol, state->fillermolecules) {
    MolList->erase(_mol);
    if ((_mol != NULL) && (!(World::getInstance().getAllMolecules(MoleculeById(_mol->getId()))).empty())) {
      for(molecule::iterator iter = _mol->begin();
          !_mol->empty();
          iter = _mol->begin()) {
        atom *Walker = *iter;
        World::getInstance().destroyAtom(Walker);
      }
      World::getInstance().destroyMolecule(_mol);
    }
  }

  // as molecules and atoms from state are removed, we have to create a new one
  std::vector<molecule *> fillermolecules;
  return Action::state_ptr(new MoleculeFillVoidWithMoleculeState(fillermolecules,state->params));
}

Action::state_ptr MoleculeFillVoidWithMoleculeAction::performRedo(Action::state_ptr _state){
  //MoleculeFillVoidWithMoleculeState *state = assert_cast<MoleculeFillVoidWithMoleculeState*>(_state.get());

  return Action::failure;
  //return Action::state_ptr(_state);
}

bool MoleculeFillVoidWithMoleculeAction::canUndo() {
  return true;
}

bool MoleculeFillVoidWithMoleculeAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
