/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * Plane.cpp
 *
 *  Created on: Apr 7, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <cmath>
#include <limits>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "defs.hpp"
#include "Exceptions.hpp"
#include "fast_functions.hpp"
#include "Line.hpp"
#include "Plane.hpp"
#include "Vector.hpp"

/**
 * generates a plane from three given vectors defining three points in space
 */
Plane::Plane(const Vector &y1, const Vector &y2, const Vector &y3) throw(LinearDependenceException) :
  normalVector(new Vector())
{
  Vector x1 = y1 - y2;
  Vector x2 = y3 - y2;
  if ((x1.Norm() <= LINALG_MYEPSILON())) {
    throw LinearDependenceException() << LinearAlgebraVectorPair( make_pair(&y1, &y2) );
  }
  if ((x2.Norm() <= LINALG_MYEPSILON())) {
    throw LinearDependenceException() << LinearAlgebraVectorPair( make_pair(&y2, &y3) );
  }
  if((fabs(x1.Angle(x2)) <= LINALG_MYEPSILON())) {
    throw LinearDependenceException() << LinearAlgebraVectorPair( make_pair(&x1, &x2) );
  }
//  Log() << Verbose(4) << "relative, first plane coordinates:";
//  x1.Output((ofstream *)&cout);
//  Log() << Verbose(0) << endl;
//  Log() << Verbose(4) << "second plane coordinates:";
//  x2.Output((ofstream *)&cout);
//  Log() << Verbose(0) << endl;

  normalVector->at(0) = (x1[1]*x2[2] - x1[2]*x2[1]);
  normalVector->at(1) = (x1[2]*x2[0] - x1[0]*x2[2]);
  normalVector->at(2) = (x1[0]*x2[1] - x1[1]*x2[0]);
  normalVector->Normalize();

  offset=normalVector->ScalarProduct(y1);
}
/**
 * Constructs a plane from two direction vectors and a offset.
 */
Plane::Plane(const Vector &y1, const Vector &y2, double _offset) throw(ZeroVectorException,LinearDependenceException) :
    normalVector(new Vector()),
    offset(_offset)
{
  Vector x1 = y1;
  Vector x2 = y2;
  if ((x1.Norm() <= LINALG_MYEPSILON())) {
    throw ZeroVectorException() << LinearAlgebraVector(&x1);
  }
  if ((x2.Norm() <= LINALG_MYEPSILON())) {
    throw ZeroVectorException() << LinearAlgebraVector(&x2);
  }
  if((fabs(x1.Angle(x2)) <= LINALG_MYEPSILON())) {
    throw LinearDependenceException() << LinearAlgebraVectorPair( make_pair(&x1, &x2) );
  }
//  Log() << Verbose(4) << "relative, first plane coordinates:";
//  x1.Output((ofstream *)&cout);
//  Log() << Verbose(0) << endl;
//  Log() << Verbose(4) << "second plane coordinates:";
//  x2.Output((ofstream *)&cout);
//  Log() << Verbose(0) << endl;

  normalVector->at(0) = (x1[1]*x2[2] - x1[2]*x2[1]);
  normalVector->at(1) = (x1[2]*x2[0] - x1[0]*x2[2]);
  normalVector->at(2) = (x1[0]*x2[1] - x1[1]*x2[0]);
  normalVector->Normalize();
}

Plane::Plane(const Vector &_normalVector, double _offset) throw(ZeroVectorException):
  normalVector(new Vector(_normalVector)),
  offset(_offset)
{
  if(normalVector->IsZero())
    throw ZeroVectorException() << LinearAlgebraVector(&(*normalVector));
  double factor = 1/normalVector->Norm();
  // normalize the plane parameters
  (*normalVector)*=factor;
  offset*=factor;
}

Plane::Plane(const Vector &_normalVector, const Vector &_offsetVector) throw(ZeroVectorException):
    normalVector(new Vector(_normalVector))
{
  if(normalVector->IsZero()){
    throw ZeroVectorException() << LinearAlgebraVector(&(*normalVector));
  }
  normalVector->Normalize();
  offset = normalVector->ScalarProduct(_offsetVector);
}

/**
 * copy constructor
 */
Plane::Plane(const Plane& plane) :
  normalVector(new Vector(*plane.normalVector)),
  offset(plane.offset)
{}


Plane::~Plane()
{}

Plane &Plane::operator=(const Plane &rhs){
  if(&rhs!=this){
    normalVector.reset(new Vector(*rhs.normalVector));
    offset = rhs.offset;
  }
  return *this;
}


Vector Plane::getNormal() const{
  return *normalVector;
}

double Plane::getOffset() const{
  return offset;
}

Vector Plane::getOffsetVector() const {
  return getOffset()*getNormal();
}

vector<Vector> Plane::getPointsOnPlane() const{
  std::vector<Vector> res;
  res.reserve(3);
  // first point on the plane
  res.push_back(getOffsetVector());
  // get a vector that has direction of plane
  Vector direction;
  direction.GetOneNormalVector(getNormal());
  res.push_back(res[0]+direction);
  // get an orthogonal vector to direction and normal (has direction of plane)
  direction.VectorProduct(getNormal());
  direction.Normalize();
  res.push_back(res[0] +direction);
  return res;
}


/** Calculates the intersection point between a line defined by \a *LineVector and \a *LineVector2 and a plane defined by \a *Normal and \a *PlaneOffset.
 * According to [Bronstein] the vectorial plane equation is:
 *   -# \f$\stackrel{r}{\rightarrow} \cdot \stackrel{N}{\rightarrow} + D = 0\f$,
 * where \f$\stackrel{r}{\rightarrow}\f$ is the vector to be testet, \f$\stackrel{N}{\rightarrow}\f$ is the plane's normal vector and
 * \f$D = - \stackrel{a}{\rightarrow} \stackrel{N}{\rightarrow}\f$, the offset with respect to origin, if \f$\stackrel{a}{\rightarrow}\f$,
 * is an offset vector onto the plane. The line is parametrized by \f$\stackrel{x}{\rightarrow} + k \stackrel{t}{\rightarrow}\f$, where
 * \f$\stackrel{x}{\rightarrow}\f$ is the offset and \f$\stackrel{t}{\rightarrow}\f$ the directional vector (NOTE: No need to normalize
 * the latter). Inserting the parametrized form into the plane equation and solving for \f$k\f$, which we insert then into the parametrization
 * of the line yields the intersection point on the plane.
 * \param *Origin first vector of line
 * \param *LineVector second vector of line
 * \return true -  \a this contains intersection point on return, false - line is parallel to plane (even if in-plane)
 */
Vector Plane::GetIntersection(const Line& line) const
{
  //Info FunctionInfo(__func__);
  Vector res;

  res = getNormal();
  const Vector direction = line.getDirection();
  double factor1 = res.ScalarProduct(direction);
  if(fabs(factor1) <= LINALG_MYEPSILON()){
    // the plane is parallel... under all circumstances this is bad luck
    // we no have either no or infinite solutions
    if(isContained(line.getOrigin())){
      const Vector origin = line.getOrigin();
      throw MultipleSolutionsException() << LinearAlgebraVector(&origin);
    }
    else{
      throw LinearDependenceException() << LinearAlgebraVectorPair( make_pair(&res,  &direction) );
    }
  }

  double factor2 = getNormal().ScalarProduct(line.getOrigin());
  double scaleFactor = (offset-factor2)/factor1;

  res = line.getOrigin() + scaleFactor * line.getDirection();

  // tests to make sure the resulting vector really is on plane and line
  ASSERT(isContained(res),"Calculated line-Plane intersection does not lie on plane.");
  ASSERT(line.isContained(res),"Calculated line-Plane intersection does not lie on line.");
  return res;
};

Vector Plane::mirrorVector(const Vector &rhs) const {
  Vector helper = getVectorToPoint(rhs);
  // substract twice the Vector to the plane
  return rhs+2*helper;
}

Line Plane::getOrthogonalLine(const Vector &origin) const{
  return Line(origin,getNormal());
}

bool Plane::onSameSide(const Vector &point1,const Vector &point2) const{
  return sign(point1.ScalarProduct(*normalVector)-offset) ==
         sign(point2.ScalarProduct(*normalVector)-offset);
}

/************ Methods inherited from Space ****************/

double Plane::distance(const Vector &point) const{
  double res = point.ScalarProduct(*normalVector)-offset;
  return fabs(res);
}

Vector Plane::getClosestPoint(const Vector &point) const{
  double factor = point.ScalarProduct(*normalVector)-offset;
  if(fabs(factor) <= LINALG_MYEPSILON()){
    // the point itself lies on the plane
    return point;
  }
  Vector difference = factor * (*normalVector);
  return (point - difference);
}

// Operators

bool operator==(const Plane &x,const Plane &y){
  return *x.normalVector == *y.normalVector && x.offset == y.offset;
}

ostream &operator << (ostream &ost,const Plane &p){
  ost << "<" << p.getNormal() << ";x> - " << p.getOffset() << "=0";
  return ost;
}
