/** \file atom.cpp
 * 
 * Function implementations for the class atom.
 * 
 */

#include "molecules.hpp"
 
/************************************* Functions for class atom *************************************/

/** Constructor of class atom.
 */
atom::atom() 
{
  Name = NULL;
  previous = NULL;
  next = NULL;
  father = this;  // generally, father is itself
  Ancestor = NULL;
  type = NULL;
  sort = NULL;
  FixedIon = 0;
  nr = -1;
  GraphNr = -1;
  ComponentNr = NULL;
  IsCyclic = false;
  SeparationVertex = false;
  LowpointNr = -1;
  AdaptiveOrder = 0;
  MaxOrder = false;
};

/** Destructor of class atom.
 */
atom::~atom() 
{
  Free((void **)&Name, "atom::~atom: *Name");
  Free((void **)&ComponentNr, "atom::~atom: *ComponentNr");
};


/** Climbs up the father list until NULL, last is returned.
 * \return true father, i.e. whose father points to itself, NULL if it could not be found or has none (added hydrogen)
 */
atom *atom::GetTrueFather()
{
  atom *walker = this;
  do {
    if (walker == walker->father) // top most father is the one that points on itself
      break;
    walker = walker->father;
  } while (walker != NULL);
  return walker;
};

/** Output of a single atom.
 * \param ElementNo cardinal number of the element
 * \param AtomNo cardinal number among these atoms of the same element
 * \param *out stream to output to
 */
bool atom::Output(int ElementNo, int AtomNo, ofstream *out) const
{
  if (out != NULL) {
    *out << "Ion_Type" << ElementNo << "_" << AtomNo << "\t"  << fixed << setprecision(9) << showpoint;
    *out << x.x[0] << "\t" << x.x[1] << "\t" << x.x[2];
    *out << "\t" << FixedIon;
    if (v.Norm() > MYEPSILON)
      *out << "\t" << scientific << setprecision(6) << v.x[0] << "\t" << v.x[1] << "\t" << v.x[2] << "\t";
    *out << " # Number in molecule " << nr << endl;
    return true;
  } else
    return false;
};

/** Output of a single atom as one lin in xyz file.
 * \param *out stream to output to
 */
bool atom::OutputXYZLine(ofstream *out) const
{
  if (out != NULL) {
    *out << type->symbol << "\t" << x.x[0] << "\t" << x.x[1] << "\t" << x.x[2] << "\t" << endl;
    return true;
  } else
    return false;
};

ostream & operator << (ostream &ost, const atom &a) 
{
  ost << "[" << a.Name << "|" << &a << "]";
  return ost;
};

/** Compares the indices of \a this atom with a given \a ptr.
 * \param ptr atom to compare index against
 * \return true - this one's is smaller, false - not
 */ 
bool atom::Compare(atom &ptr)
{
  if (nr < ptr.nr)
    return true;
  else
    return false;
};

bool operator < (atom &a, atom &b) 
{
  return a.Compare(b);
};

