/*
 * QTStatusBar.cpp
 *
 *  Created on: Feb 17, 2010
 *      Author: crueger
 */

#include <sstream>

#include <QtGui/QLabel>
#include <QtGui/QBoxLayout>
#include <QtGui/QProgressBar>

#include "QTStatusBar.hpp"
#include "World.hpp"
#include "helpers.hpp"
#include "Actions/Process.hpp"


QTStatusBar::QTStatusBar(QWidget *_parent) :
  QStatusBar(_parent),
  parent(_parent),
  atomCount(World::getInstance().numAtoms()),
  moleculeCount(World::getInstance().numMolecules())
{
  World::getInstance().signOn(this);
  Process::AddObserver(this);
  statusLabel = new QLabel(this);
  statusLabel->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
  addPermanentWidget(statusLabel);
  redrawStatus();
}

QTStatusBar::~QTStatusBar()
{
  Process::RemoveObserver(this);
  World::getInstance().signOff(this);
}

void QTStatusBar::update(Observable *subject){
  if (subject==World::getPointer()){
    atomCount = World::getInstance().numAtoms();
    moleculeCount = World::getInstance().numMolecules();
    redrawStatus();
  }
  else {
    // we probably have some process
    Process *proc;
    if((proc=dynamic_cast<Process*>(subject))){
      redrawProcess(proc);
    }
  }
}

void QTStatusBar::subjectKilled(Observable *subject){
  // Processes don't notify when they are killed
  atomCount = World::getInstance().numAtoms();
  moleculeCount = World::getInstance().numMolecules();
  World::getInstance().signOn(this);
  redrawStatus();
}

void QTStatusBar::redrawStatus(){
  stringstream sstr;
  sstr << "You have " << atomCount << " atom" << PLURAL_S(atomCount)
       <<" in " << moleculeCount << " molecule" << PLURAL_S(moleculeCount);
  statusLabel->setText(QString(sstr.str().c_str()));
}

void QTStatusBar::redrawProcess(Process *proc){
  progressIndicator *ind=0;
  // see what we have to do with the process
  if(proc->doesStart()){
    ind = new progressIndicator(proc->getName());
    ind->bar->setMaximum(proc->getMaxSteps());
    progressBars.insert(pair<Process*,progressIndicator*>(proc,ind));
  }
  else {
    ind = progressBars[proc];
  }
  if(activeProcess!=proc){
    addWidget(ind->container);
    activeProcess = proc;
  }
  ind->bar->setValue(proc->getCurrStep());
  parent->repaint();
  if(proc->doesStop()){
    removeWidget(ind->container);
    activeProcess = 0;
    progressBars.erase(proc);
    delete ind;
  }
}



QTStatusBar::progressIndicator::progressIndicator(string name){
  stringstream sstr;
  sstr << "Busy (" << name << ")";
  container = new QWidget();
  layout = new QHBoxLayout(container);
  label = new QLabel(QString(sstr.str().c_str()));
  bar = new QProgressBar();

  layout->addWidget(label);
  layout->addWidget(bar);
  container->setLayout(layout);
}

QTStatusBar::progressIndicator::~progressIndicator(){
  delete container;
}
