/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ParserTremoloUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserTremoloUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include "CodePatterns/Log.hpp"

#include "Atom/atom.hpp"
#include "Atom/AtomObserver.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Parser/TremoloParser.hpp"
#include "Parser/ChangeTracker.hpp"
#include "World.hpp"
#include "WorldTime.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserTremoloUnitTest );

static string Tremolo_Atomdata1 = "\
# ATOMDATA\tId\tname\ttype\tx=3\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n";
static string Tremolo_Atomdata2 = "\
#\n\
#ATOMDATA Id name type x=3\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n\
1 hydrogen H 3.0 4.5 0.1\n\
\n";
static string Tremolo_invalidkey = "\
#\n\
#ATOMDATA Id name foo type x=3\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n\
\n\n";
static string Tremolo_velocity = "\
#\n\
#ATOMDATA Id name type u=3\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n\
1 hydrogen H 3.0 4.5 0.1\n\
\n";
static string Tremolo_neighbours = "#\n\
#ATOMDATA Id type neighbors=2\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n\
1 H 3 0\n\
2 H 3 0\n\
3 O 1 2\n";
static string Tremolo_improper = "\
#\n\
#ATOMDATA Id type imprData\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n\
8 H 9-10\n\
9 H 10-8,8-10\n\
10 O -\n";
static string Tremolo_torsion = "\
#\n\
#ATOMDATA Id type torsion\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n\
8 H 9-10\n\
9 H 10-8,8-10\n\
10 O -\n";
static string Tremolo_full = "\
# ATOMDATA\tx=3\tu=3\tF\tstress\tId\tneighbors=5\timprData\tGroupMeasureTypeNo\ttype\textType\tname\tresName\tchainID\tresSeq\toccupancy\ttempFactor\tsegID\tCharge\tcharge\tGrpTypeNo\ttorsion\n\
# Box\t20\t0\t0\t0\t20\t0\t0\t0\t20\n\
0\t0\t0\t0\t0\t0\t0\t0\t1\t0\t0\t0\t0\t0\t-\t0\tH\t-\t-\t-\t0\t0\t0\t0\t0\t0\t0\t0\t-\t\n";

void ParserTremoloUnitTest::setUp() {
  World::getInstance();

  setVerbosity(5);

  parser = new FormatParser<tremolo>();

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserTremoloUnitTest::tearDown() 
{
  delete parser;
  ChangeTracker::purgeInstance();
  AtomObserver::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserTremoloUnitTest::readTremoloPreliminaryCommentsTest() {
  cout << "Testing the tremolo parser." << endl;
  stringstream input, output;

  // Atomdata beginning with "# ATOMDATA"
  {
    input << Tremolo_Atomdata1;
    parser->load(&input);
    std::vector<atom *> atoms = World::getInstance().getAllAtoms();
    parser->save(&output, atoms);
//    std::cout << output.str() << std::endl;
//    std::cout << Tremolo_Atomdata1 << std::endl;
    CPPUNIT_ASSERT(Tremolo_Atomdata1 == output.str());
    input.clear();
    output.clear();
  }

  // Atomdata beginning with "#ATOMDATA"
  {
    input << Tremolo_Atomdata2;
    parser->load(&input);
    std::vector<atom *> atoms = World::getInstance().getAllAtoms();
    parser->save(&output, atoms);
    std::cout << output.str() << std::endl;
    CPPUNIT_ASSERT(output.str().find("hydrogen") != string::npos);
    input.clear();
    output.clear();
  }

  // Invalid key in Atomdata line
  input << Tremolo_invalidkey;
  CPPUNIT_ASSERT_THROW( parser->load(&input), IllegalParserKeyException);
  //TODO: prove invalidity
  input.clear();
}

void ParserTremoloUnitTest::getsetAtomDataTest() {
  stringstream input;
  input << Tremolo_Atomdata1;
  parser->load(&input);

  CPPUNIT_ASSERT_EQUAL( std::string("Id name type x=3"), parser->getAtomData() );

  // overwrite keys
  const std::string fewkeys("Id type x=3 neighbors=2");
  parser->resetAtomData(fewkeys);
  CPPUNIT_ASSERT_EQUAL( fewkeys, parser->getAtomData() );

  // add some keys
  const std::string morekeys("charge");
  parser->setAtomData(morekeys);
  CPPUNIT_ASSERT_EQUAL( fewkeys+std::string(" ")+morekeys, parser->getAtomData() );

  // add similar key
  const std::string otherkey("neighbors=4");
  parser->setAtomData(otherkey);
  CPPUNIT_ASSERT( fewkeys+std::string(" ")+morekeys != parser->getAtomData() );
}

void ParserTremoloUnitTest::readTremoloCoordinatesTest() {
  stringstream input;

  // One simple data line
  input << Tremolo_Atomdata2;
  parser->load(&input);
  CPPUNIT_ASSERT(World::getInstance().getAtom(AtomByType(1))->at(0) == 3.0);
  input.clear();
}

void ParserTremoloUnitTest::readTremoloVelocityTest() {
  stringstream input;

  // One simple data line
  input << Tremolo_velocity;
  parser->load(&input);
  CPPUNIT_ASSERT(World::getInstance().getAtom(AtomByType(1))->getAtomicVelocity()[0] == 3.0);
  input.clear();
}

void ParserTremoloUnitTest::readTremoloNeighborInformationTest() {
  stringstream input;

  // Neighbor data
  input << Tremolo_neighbours;
  parser->load(&input);

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
  CPPUNIT_ASSERT(World::getInstance().getAtom(AtomByType(8))->
      IsBondedTo(WorldTime::getTime(), World::getInstance().getAtom(AtomByType(1))));
  input.clear();
}

void ParserTremoloUnitTest::readAndWriteTremoloImprDataInformationTest() {
  stringstream input, output;

  // Neighbor data
  {
    input << Tremolo_improper;
    parser->load(&input);
    std::vector<atom *> atoms = World::getInstance().getAllAtoms();
    parser->save(&output, atoms);
    CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
    std::cout << output.str() << std::endl;
    CPPUNIT_ASSERT(output.str().find("2-0,0-2") != string::npos);
    input.clear();
    output.clear();
  }
}

void ParserTremoloUnitTest::readAndWriteTremoloTorsionInformationTest() {
  stringstream input, output;

  // Neighbor data
  {
    input << Tremolo_torsion;
    parser->load(&input);
    std::vector<atom *> atoms = World::getInstance().getAllAtoms();
    parser->save(&output, atoms);
    CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
    std::cout << output.str() << std::endl;
    CPPUNIT_ASSERT(output.str().find("2-0,0-2") != string::npos);
    input.clear();
    output.clear();
  }
}

void ParserTremoloUnitTest::writeTremoloTest() {
  stringstream output;

  // with the maximum number of fields and minimal information, default values are printed
  {
    atom* newAtom = World::getInstance().createAtom();
    newAtom->setType(1);
    parser->setAtomData("x=3 u=3 F stress Id neighbors=5 imprData GroupMeasureTypeNo type extType name resName chainID resSeq occupancy tempFactor segID Charge charge GrpTypeNo torsion");
    std::vector<atom *> atoms = World::getInstance().getAllAtoms();
    parser->save(&output, atoms);
//    std::cout << output.str() << std::endl;
//    std::cout << Tremolo_full << std::endl;
    CPPUNIT_ASSERT(output.str() == Tremolo_full);
  }

  cout << "testing the tremolo parser is done" << endl;
}

