#ifndef VECTOR_HPP_
#define VECTOR_HPP_

using namespace std;

/*********************************************** includes ***********************************/

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_multimin.h>

#include <memory>
#include <vector>

#include "defs.hpp"
#include "Space.hpp"

/********************************************** declarations *******************************/

class Vector;
class Matrix;

typedef std::vector<Vector> pointset;

/** Single vector.
 * basically, just a x[3] but with helpful functions
 */
class Vector : public Space{
  friend Vector operator*(const Matrix&,const Vector&);
public:
  Vector();
  Vector(const double x1, const double x2, const double x3);
  Vector(const Vector& src);
  virtual ~Vector();

  // Method implemented by forwarding to the Representation

  double DistanceSquared(const Vector &y) const;
  double DistanceToSpace(const Space& space) const;
  double PeriodicDistance(const Vector &y, const double * const cell_size) const;
  double PeriodicDistanceSquared(const Vector &y, const double * const cell_size) const;
  double ScalarProduct(const Vector &y) const;
  double Angle(const Vector &y) const;
  bool IsZero() const;
  bool IsOne() const;
  bool IsNormalTo(const Vector &normal) const;
  bool IsEqualTo(const Vector &a) const;

  void AddVector(const Vector &y);
  void SubtractVector(const Vector &y);
  void VectorProduct(const Vector &y);
  void ProjectOntoPlane(const Vector &y);
  void ProjectIt(const Vector &y);
  Vector Projection(const Vector &y) const;
  void ScaleAll(const double *factor);
  void Scale(const double factor);
  void MatrixMultiplication(const Matrix &M);
  bool InverseMatrixMultiplication(const double * const M);
  void KeepPeriodic(const double * const matrix);
  bool GetOneNormalVector(const Vector &x1);
  bool MakeNormalTo(const Vector &y1);
  bool IsInParallelepiped(const Vector &offset, const double * const parallelepiped) const;
  void WrapPeriodically(const double * const M, const double * const Minv);
  std::pair<Vector,Vector> partition(const Vector&) const;
  std::pair<pointset,Vector> partition(const pointset&) const;

  // Accessors ussually come in pairs... and sometimes even more than that
  double& operator[](size_t i);
  const double& operator[](size_t i) const;
  double& at(size_t i);
  const double& at(size_t i) const;

  // Assignment operator
  Vector &operator=(const Vector& src);

  // Access to internal structure
  gsl_vector* get();

  // Methods that are derived directly from other methods
  double Norm() const;
  double NormSquared() const;
  void Normalize();
  void Zero();
  void One(const double one);
  void LinearCombinationOfVectors(const Vector &x1, const Vector &x2, const Vector &x3, const double * const factors);

  // operators for mathematical operations
  bool operator==(const Vector& b) const;
  bool operator!=(const Vector& b) const;
  const Vector& operator+=(const Vector& b);
  const Vector& operator-=(const Vector& b);
  Vector const operator+(const Vector& b) const;
  Vector const operator-(const Vector& b) const;
  Vector& operator*=(const Matrix&);

  // Methods inherited from Space
  virtual double distance(const Vector &point) const;
  virtual Vector getClosestPoint(const Vector &point) const;

protected:

private:
  Vector(gsl_vector *);
  gsl_vector *content;

};

// some commonly used and fixed vectors
const extern Vector zeroVec;
const extern Vector e1;
const extern Vector e2;
const extern Vector e3;

ostream & operator << (ostream& ost, const Vector &m);
const Vector& operator*=(Vector& a, const double m);
Vector const operator*(const Vector& a, const double m);
Vector const operator*(const double m, const Vector& a);
Vector operator*(const Matrix&,const Vector&);


#endif /*VECTOR_HPP_*/
