/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2016 Frederik Heber. All rights reserved.
 *
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ExportGraph_ToAtomFragments.cpp
 *
 *  Created on: Mar 07, 2016
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio required before MemDebug due to placement new
#include <boost/asio.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "Fragmentation/Exporters/ExportGraph_ToAtomFragments.hpp"

#include <algorithm>

#include "Fragmentation/Homology/AtomFragmentsMap.hpp"
#include "Fragmentation/KeySet.hpp"
#include "World.hpp"

ExportGraph_ToAtomFragments::ExportGraph_ToAtomFragments(
    const Graph &_graph,
    const enum HydrogenTreatment _treatment,
    const enum HydrogenSaturation _saturation,
    const SaturatedFragment::GlobalSaturationPositions_t &_globalsaturationpositions) :
    ExportGraph(_graph, _treatment, _saturation,_globalsaturationpositions)
{}

ExportGraph_ToAtomFragments::~ExportGraph_ToAtomFragments()
{}

bool ExportGraph_ToAtomFragments::operator()()
{
  AtomFragmentsMap &atomfragments = AtomFragmentsMap::getInstance();
  LOG(1, "INFO: Creating " << TotalGraph.size() << " possible fragments.");

  // go through all fragments, output to stream and create job therefrom
  ExportGraph::SaturatedFragment_ptr CurrentFragment = getNextFragment();
  for (; (CurrentFragment != NULL) && (CurrentFragment->getKeySet() != ExportGraph::EmptySet);
      CurrentFragment = getNextFragment()) {
    const KeySet &set = CurrentFragment->getKeySet();
    LOG(3, "DEBUG: Creating fragment keysets for set " << set << ".");
    {
      // order is the same as the number of non-hydrogen atoms
      const KeySet &keyset = CurrentFragment->getKeySet();
      const KeySet &fullmolecule = CurrentFragment->getFullMolecule();
      const KeySet &saturationhydrogens = CurrentFragment->getSaturationHydrogens();
      AtomFragmentsMap::indices_t forceindices(fullmolecule.begin(), fullmolecule.end());
      {
        // replace all saturated hydrogen indices by "-1"
        for (AtomFragmentsMap::indices_t::iterator iter = forceindices.begin();
            iter != forceindices.end();
            ++iter)
          if (saturationhydrogens.find(*iter) != saturationhydrogens.end())
            *iter = -1;
      }
      const bool status = atomfragments.addFullKeyset(keyset, forceindices);
      if (!status)
        ELOG(1, "Something went wrong with insertion for keyset " << keyset);
    }
    // store force index reference file
    // explicitly release fragment
    CurrentFragment.reset();
  }
  if (CurrentFragment == NULL) {
    ELOG(1, "Some error while obtaining the next fragment occured.");
    return false;
  }
  return true;
}
