/*
 * ExportGraph.hpp
 *
 *  Created on: 08.03.2012
 *      Author: heber
 */

#ifndef EXPORTGRAPH_HPP_
#define EXPORTGRAPH_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <vector>

#include <boost/shared_ptr.hpp>

#include "Fragmentation/HydrogenSaturation_enum.hpp"
#include "Fragmentation/Exporters/ExportGraph.hpp"
#include "Fragmentation/Exporters/HydrogenPool.hpp"
#include "Fragmentation/Exporters/SaturatedFragment.hpp"

#include "Fragmentation/Graph.hpp"

class ListOfLocalAtoms_t;
class molecule;

/** ExportGraph is an interface definition of a class that takes a fragmented
 * system contained in an instance of Graph and exports it to some place else.
 *
 * Guidelines are as follows:
 * - all derived classes should have their specific properties set by extra
 *   setters and not by passing them through the cstor
 * - operator() is called without arguments.
 */
class ExportGraph
{
public:
  ExportGraph(
      const Graph &_graph,
      const enum HydrogenTreatment _treatment,
      const enum HydrogenSaturation _saturation,
      const SaturatedFragment::GlobalSaturationPositions_t &_globalsaturationpositions);
  virtual ~ExportGraph();

  typedef boost::shared_ptr<SaturatedFragment> SaturatedFragment_ptr;

  virtual bool operator()()=0;

  /** Returns a saturated fragment for immediate use only.
   *
   * An empty KeySet indicates the end of all possible fragments
   * A shared_ptr containing NULL indicate an error
   *
   * \return RAII instance of a saturated fragment, NULL - failure, empty set - end
   */
  SaturatedFragment_ptr getNextFragment();

  /** Sets the next fragment to the first graph again.
   *
   */
  void reset();

private:

  /** Helper function to create a fragment from a keyset and note down its in use.
   *
   * \param _set KeySet to create saturated fragment from
   * \return RAII instance of a saturated fragment
   */
   SaturatedFragment_ptr leaseFragment(const KeySet &_set);

  /** Helper function to create a fragment from a keyset and note down its in use.
   *
   * \param _ptr RAII instance of a saturated fragment to release again
   */
  void releaseFragment(SaturatedFragment_ptr &_ptr);

//  void prepareMolecule();
//  molecule * StoreFragmentFromKeySet(KeySet &Leaflet, bool IsAngstroem);
//  int StoreFragmentFromKeySet_Init(molecule *Leaf, KeySet &Leaflet, ListOfLocalAtoms_t &SonList);
//  void CreateInducedSubgraphOfFragment(molecule *Leaf, ListOfLocalAtoms_t &SonList, bool IsAngstroem);

protected:
  //!> empty set to indicate last and one fragment
  const KeySet EmptySet;

protected:
  //!> set of all KeySets that are leased as saturated fragments
  SaturatedFragment::KeySetsInUse_t KeySetsInUse;

  //!> pool containing hydrogens for saturation
  HydrogenPool hydrogens;

  //!> graph containing all keysets
  const Graph &TotalGraph;
  //!> internal list of created molecules
  std::vector<molecule *> BondFragments;
  //!> whether to always add already present hydrogens or not
  const enum HydrogenTreatment treatment;
  //!> whether to saturate dangling bonds or not
  const enum HydrogenSaturation saturation;

  //!> Global information over all atoms with saturation positions to be used per fragment
  const SaturatedFragment::GlobalSaturationPositions_t &globalsaturationpositions;

private:
  //!> iterator pointing at the CurrentKeySet to be exported
  Graph::const_iterator CurrentKeySet;
};

#endif /* EXPORTGRAPH_HPP_ */
