/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ParserMpqcUnitTest.cpp
 *
 *  Created on: Mar 3, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ParserMpqcUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <boost/any.hpp>

#include "World.hpp"
#include "atom.hpp"
#include "element.hpp"
#include "periodentafel.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include "CodePatterns/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

using namespace std;

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ParserMpqcUnitTest );

static string waterMpqc_CLHF ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<CLHF>: (\n\
\t\tmolecule = $:molecule\n\
\t\tbasis = $:basis\n\
\t\tmaxiter = 1000\n\
\t\tmemory = 16000000\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n"; // tested with mpqc 3.0.0-alpha
static string waterMpqc_CLKS ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<CLKS>: (\n\
\t\tfunctional<StdDenFunctional>:(name=B3LYP)\n\
\t\tmolecule = $:molecule\n\
\t\tbasis = $:basis\n\
\t\tmaxiter = 1000\n\
\t\tmemory = 16000000\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n"; // tested with mpqc 3.0.0-alpha
static string waterMpqc_MBPT2 ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<MBPT2>: (\n\
\t\tbasis = $:basis\n\
\t\tmolecule = $:molecule\n\
\t\tmemory = 16000000\n\
\t\treference<CLHF>: (\n\
\t\t\tmaxiter = 1000\n\
\t\t\tbasis = $:basis\n\
\t\t\tmolecule = $:molecule\n\
\t\t\tmemory = 16000000\n\
\t\t)\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n"; // tested with mpqc 3.0.0-alpha
static string waterMpqc_MBPT2_R12 ="% Created by MoleCuilder\n\
mpqc: (\n\
\tsavestate = no\n\
\tdo_gradient = yes\n\
\tmole<MBPT2_R12>: (\n\
\t\tmolecule = $:molecule\n\
\t\tbasis = $:basis\n\
\t\taux_basis = $:abasis\n\
\t\tstdapprox = \"A'\"\n\
\t\tnfzc = 1\n\
\t\tmemory = 16000000\n\
\t\tintegrals<IntegralCints>:()\n\
\t\treference<CLHF>: (\n\
\t\t\tmolecule = $:molecule\n\
\t\t\tbasis = $:basis\n\
\t\t\tmaxiter = 1000\n\
\t\t\tmemory = 16000000\n\
\t\t\tintegrals<IntegralCints>:()\n\
\t\t)\n\
\t)\n\
)\n\
molecule<Molecule>: (\n\
\tunit = angstrom\n\
\t{ atoms geometry } = {\n\
\t\tO [ -0.505735\t0\t0 ]\n\
\t\tH [ 0.252867\t0\t0.504284 ]\n\
\t\tH [ 0.252867\t0\t-0.504284 ]\n\
\t}\n\
)\n\
basis<GaussianBasisSet>: (\n\
\tname = \"3-21G\"\n\
\tmolecule = $:molecule\n\
)\n\
% auxiliary basis set specification\n\
\tabasis<GaussianBasisSet>: (\n\
\tname = \"aug-cc-pVDZ\"\n\
\tmolecule = $:molecule\n\
)\n"; // basically tested with mpqc 3.0.0-alpha (no parse errors but did not calculate due to missing code)

void ParserMpqcUnitTest::setUp() {
  mpqc = new MpqcParser();

  World::getInstance();

  setVerbosity(2);

  // we need hydrogens and oxygens in the following tests
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(1) != NULL);
  CPPUNIT_ASSERT(World::getInstance().getPeriode()->FindElement(8) != NULL);
}

void ParserMpqcUnitTest::tearDown() {
  delete mpqc;
  ChangeTracker::purgeInstance();
  World::purgeInstance();
}

/************************************ tests ***********************************/

void ParserMpqcUnitTest::ParameterTypeTest() {
  // check types in boost::any map
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::hessianParam].type() == typeid(bool));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::hessianParam].type() != typeid(int));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::savestateParam].type() == typeid(bool));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::do_gradientParam].type() == typeid(bool));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::maxiterParam].type() == typeid(int));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::memoryParam].type() == typeid(int));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::stdapproxParam].type() == typeid(std::string));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::nfzcParam].type() == typeid(int));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::basisParam].type() == typeid(std::string));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::aux_basisParam].type() == typeid(std::string));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::integrationParam].type() == typeid(MpqcParser_Parameters::IntegralCints));
  CPPUNIT_ASSERT(mpqc->getParams().params[MpqcParser_Parameters::theoryParam].type() == typeid(MpqcParser_Parameters::MBPT2));
}

void ParserMpqcUnitTest::ParameterDefaultTest() {
  // check default values
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::hessianParam) == "no");
  CPPUNIT_ASSERT(!mpqc->getParams().getBool(MpqcParser_Parameters::hessianParam));
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::savestateParam) == "no");
  CPPUNIT_ASSERT(!mpqc->getParams().getBool(MpqcParser_Parameters::savestateParam));
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::do_gradientParam) == "yes");
  CPPUNIT_ASSERT(mpqc->getParams().getBool(MpqcParser_Parameters::do_gradientParam));
  CPPUNIT_ASSERT(mpqc->getParams().getInt(MpqcParser_Parameters::maxiterParam) == 1000);
  CPPUNIT_ASSERT(mpqc->getParams().getInt(MpqcParser_Parameters::memoryParam) == 16000000);
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::stdapproxParam) == "A'");
  CPPUNIT_ASSERT(mpqc->getParams().getInt(MpqcParser_Parameters::nfzcParam) == 1);
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::basisParam) == "3-21G");
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::aux_basisParam) == "aug-cc-pVDZ");
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::integrationParam) == "IntegralCints");
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::theoryParam) == "MBPT2");
  CPPUNIT_ASSERT(mpqc->getParams().getTheory() == MpqcParser_Parameters::MBPT2);
  CPPUNIT_ASSERT(mpqc->getParams().getIntegration() == MpqcParser_Parameters::IntegralCints);

  // check that values are not removed
  CPPUNIT_ASSERT(!mpqc->getParams().params[MpqcParser_Parameters::theoryParam].empty());

  // check throw, for the moment aren't, are caught in getInt()
  CPPUNIT_ASSERT_THROW(mpqc->getParams().getInt(MpqcParser_Parameters::integrationParam), boost::bad_any_cast);
  CPPUNIT_ASSERT_THROW(mpqc->getParams().getInt(MpqcParser_Parameters::theoryParam), boost::bad_any_cast);

}

void ParserMpqcUnitTest::ParameterCloneTest() {
  FormatParser_Parameters *clone = mpqc->getParams().clone();
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::theoryParam) == "MBPT2");
  std::stringstream setvalue("theory = CLHF");
  setvalue >> mpqc->getParams();
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::theoryParam) == "CLHF");
  mpqc->getParams().makeClone(*clone);
  CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::theoryParam) == "MBPT2");
}

void ParserMpqcUnitTest::ParameterSetterTest() {
  // test a string
  {
    std::stringstream setvalue("theory = CLHF");
    setvalue >> mpqc->getParams();
//    std::cout << "integration method is "
//        << mpqc->getParams().getString(MpqcParser_Parameters::theoryParam) << std::endl;
    CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::theoryParam) == "CLHF");
  }
  // test a bool
  {
    std::stringstream setvalue("Hessian = yes");
    setvalue >> mpqc->getParams();
//    std::cout << "Hessian is "
//        << mpqc->getParams().getString(MpqcParser_Parameters::hessianParam) << std::endl;
    CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::hessianParam) == "yes");
  }
  // test int
  {
    std::stringstream setvalue("maxiter = 500");
    setvalue >> mpqc->getParams();
//    std::cout << "maxiter is "
//        << mpqc->getParams().getString(MpqcParser_Parameters::maxiterParam) << std::endl;
    CPPUNIT_ASSERT(mpqc->getParams().getInt(MpqcParser_Parameters::maxiterParam) == 500);
  }
  // test whether unknown key fails
  std::cout << "The following Assertion warning is desired and does not indicate a failure of the test." << std::endl;
  {
    std::stringstream setvalue("hessian = no");
#ifndef NDEBUG
    ASSERT_DO(Assert::Throw);
    CPPUNIT_ASSERT_THROW(setvalue >> mpqc->getParams(), Assert::AssertionFailure);
#else
    setvalue >> mpqc->getParams();
#endif
//    std::cout << "Hessian is still "
//        << mpqc->getParams().getString(MpqcParser_Parameters::hessianParam) << std::endl;
    CPPUNIT_ASSERT(mpqc->getParams().getString(MpqcParser_Parameters::hessianParam) == "yes");
  }
}

void ParserMpqcUnitTest::readMpqcTest() {
  stringstream input(waterMpqc_CLHF);
  mpqc->getParams().setTheory(MpqcParser_Parameters::CLHF);
  mpqc->load(&input);

  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());
}

void ParserMpqcUnitTest::writeMpqcTest() {
  // build up water molecule
  string first;
  string second;
  atom *Walker = NULL;
  Walker = World::getInstance().createAtom();
  Walker->setType(8);
  Walker->setPosition(Vector(0,0,0));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,0.504284));
  Walker = World::getInstance().createAtom();
  Walker->setType(1);
  Walker->setPosition(Vector(0.758602,0,-0.504284));
  CPPUNIT_ASSERT_EQUAL(3, World::getInstance().numAtoms());

  // create two stringstreams, one stored, one created

  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  {
    // compare both configs for CLHF
    stringstream output;
    mpqc->getParams().setTheory(MpqcParser_Parameters::CLHF);
    mpqc->save(&output, atoms);
    stringstream input(waterMpqc_CLHF);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
  {
    // compare both configs for CLKS
    stringstream output;
    mpqc->getParams().setTheory(MpqcParser_Parameters::CLKS);
    mpqc->save(&output, atoms);
    stringstream input(waterMpqc_CLKS);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
  {
    // compare both configs for MBPT2
    stringstream output;
    mpqc->getParams().setTheory(MpqcParser_Parameters::MBPT2);
    mpqc->save(&output, atoms);
    stringstream input(waterMpqc_MBPT2);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
  {
    // compare both configs for MBPT2_R12
    stringstream output;
    mpqc->getParams().setTheory(MpqcParser_Parameters::MBPT2_R12);
    mpqc->save(&output, atoms);
    stringstream input(waterMpqc_MBPT2_R12);
    for (; std::getline(input, first) && std::getline(output, second); ) {
      //std::cout << "Comparing '" << first << "' to '" << second << "'" << std::endl;
      CPPUNIT_ASSERT(first == second);
    }
  }
}
