/*
 * MpqcParser_Parameters.hpp
 *
 *  Created on: Feb 3, 2011
 *      Author: heber
 */

#ifndef MPQCPARSER_PARAMETERS_HPP_
#define MPQCPARSER_PARAMETERS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <list>
#include <typeinfo>
#include <vector>

#include "CodePatterns/Clone.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include <boost/any.hpp>

#include "Parser/FormatParser_Parameters.hpp"

class MpqcParser;

class MpqcParser_Parameters : public FormatParser_Parameters
{
  // MpqcParser should be friend to access params directly for types.
  friend class MpqcParser;
  // ParserMpqcUnitTest needs to be friend to check types contained in params.
  friend class ParserMpqcUnitTest;

public:
  /** Constructor of MpqcParser_Parameters.
   *
   */
  MpqcParser_Parameters();

  /** Copy Constructor of MpqcParser_Parameters.
   *
   * @param state ref to instance to copy
   */
  MpqcParser_Parameters(const MpqcParser_Parameters & state);

  /** Destructor of MpqcParser_Parameters.
   *
   */
  virtual ~MpqcParser_Parameters();

  /** Enumeration of all known theories.
   *
   */
  enum Theory {
    CLHF,       //!< Closed Shell Hartree-Fock equations
    CLKS,       //!< Closed Shell Kohn-Sham equations
    MBPT2,      //!< Moeller Plesset Perturbation Theory second order
    MBPT2_R12,  //!< Moeller Plesset Perturbation Theory second order with R12 integral
    unknownTheory //!< designates an unknown theory
  };

  /** Enumeration of all known integration methods
   *
   */
  enum IntegrationMethod {
    IntegralCints,  //!< Integration method Cints in MBPT2 R12?
    unknownIntegration //!< designates unknown integration method
  };

  /** Enumeration of all known Parameters to allow placing them in vectors, maps.
   *
   */
  enum Parameters {
    hessianParam,    //!< HessianParam, whether hessian should be calculated or not
    savestateParam,  //!< savestateParam, whether intermediate/final states (wave function) should be stored
    do_gradientParam,//!< do_gradientParam, whether a gradient should be calculated
    maxiterParam,    //!< maxiterParam, number of maximum iterations for CG
    memoryParam,     //!< memoryParam, maximum amount of memory to use
    stdapproxParam,  //!< stdapproxParam, standard approximation in MBPT2 R12
    nfzcParam,       //!< nfzcParam, nfzc parameter in MBPT2 R12
    basisParam,      //!< basisParam, basis set to use
    aux_basisParam,  //!< aux_basisParam, auxiliary baseis set to use in MBPT2 R12
    integrationParam,//!< integrationParam, integration method to use in MBPT2 R12
    theoryParam,     //!< theoryParam, level of theory to use
    unknownParam};   //!< unknownParam, designates an unknown parameter

  bool checkWorldElementsAgainstCurrentBasis() const;

  /** Sets the desired level of solving theory to use.
   *
   * \param _theory shorthand of the theory
   */
  void setTheory(enum Theory _theory);

  /** Sets the desired level of solving integration to use.
   *
   * \param _integration shorthand of the integration
   */
  void setIntegration(enum IntegrationMethod _integration);

  /** Getter for integration method in params.
   *
   * @return enumeration index of IntegrationMethod.
   */
  enum IntegrationMethod getIntegration() const;

  /** Getter for current Theory in params.
   *
   * @return enumeration index of Theory
   */
  enum Theory getTheory() const;

  /** Getter for a parameter in params as a string.
   *
   * @param _param enumeration index of desired Parameter
   * @return string value
   */
  std::string getString(enum Parameters _param) const;

  /** Getter for integer value of desired Parameter in params.
   *
   * Only if type in params matches int!
   *
   * @param _param enumeration index in Parameter
   * @return integer value of parameter
   */
  int getInt(enum Parameters _param) const;

  /** Getter for double value of desired Parameter in params.
   *
   * Only if type in params matches double!
   *
   * @param _param enumeration index in Parameter
   * @return double value of parameter
   */
  double getDouble(enum Parameters _param) const;

  /** Getter for bool value of desired Parameter in params.
   *
   * Only if type in params matches bool!
   *
   * @param _param enumeration index in Parameter
   * @return bool value of parameter
   */
  bool getBool(enum Parameters _param) const;

  /** Setter for a desired value of its type is known.
   *
   * We check whether given type matches present type in params.
   *
   * @param _param enumeration index of Parameter
   * @param _desired desired value to set to
   * @return true - type match, value set, false - type mismatch
   */
  template <class T> bool setter(enum Parameters _param, T _desired) {
    if (typeid(T) == params[_param].type()) {
      params[_param] = _desired;
      return true;
    } else
      return false;
  }

  /** Sets a desired value in the params from a string.
   *
   * This is due to strict typing of C++ very ugly and boost::any does not make
   * it any better because it offers to functions to use values directly from
   * stringstream. Probably, because value is unknown to is as well and hence
   * the author could not implement it beautifully, so he dropped it altogether.
   * Grrr ....
   *
   * @param _param param to set
   * @param _desired stringstream containing value as next argument
   * @return true - type ok, false - unknown type in params.
   */
  bool setter(enum Parameters _param, std::stringstream& _desired);

  /** Grants access to ParamLookup.
   *
   *  Does not check for unknown parameter.
   *
   * @param _name name of parameter
   * @return enumeration index of Parameters
   */
  enum Parameters getParam(std::string _name) const;

  /** Checker whether parameter with name is known.
   *
   * @param _name
   * @return true - parameter known, false - parameter unknown
   */
  bool haveParam(std::string _name) const;

  /** Creates a clone of the class.
   *
   * @return
   */
  FormatParser_Parameters* clone() const;

  /** Applies a before returned undo state.
   *
   * @param undo state to set
   */
  void makeClone(const FormatParser_Parameters & _state);

  /** Set the internal parameters to the one's from the given \a state.
   *
   * @param state set of parameters
   */
  void copyParameters(const MpqcParser_Parameters & state);

private:
  /** Global initialization in cstor.
   *
   */
  void Init();

  /** Initializes BasisList.
   *
   */
  void initBasis();

  /** Initializes params.
   * Sets the type and the associated enumeration index.
   */
  void initParameters();

  /** Internal function used by initParameters() to add parameters to params.
   *
   * @param _enum enumeration index to set
   * @param _p (default) value to set with certain type
   */
  template <class T> void appendParameter(enum Parameters _enum, T _p) {
    boost::any _p_value = _p;
    params[_enum] = _p_value;
  }

  // all internal typedefs for lists below
  typedef std::map<std::string, std::list<std::string> > BasisMapType;
  typedef std::map<enum Theory, std::string> TheoryNamesType;
  typedef std::map<std::string, enum Theory> TheoryLookupType;
  typedef std::map<enum IntegrationMethod, std::string> IntegrationNamesType;
  typedef std::map<std::string, enum IntegrationMethod> IntegrationLookupType;
  typedef std::map<enum Parameters, std::string> ParamNamesType;
  typedef std::map<std::string, enum Parameters> ParamLookupType;
  typedef std::map<enum Parameters, boost::any> parameterlist;

  //!> boost::any container for all the parameters
  parameterlist params;

  // maps from names to enumerations

  //!> contains basis and all elements the basis knows about
  BasisMapType BasisList;
  //!> contains the name of a theory as string
  TheoryNamesType TheoryNames;
  //!> contains a lookup from theory name to enumeration index
  TheoryLookupType TheoryLookup;
  //!> contains the name of an integration method as string
  IntegrationNamesType IntegrationNames;
  //!> contains a lookup from integration method name to enumeration index
  IntegrationLookupType IntegrationLookup;
  //!> contains the name of a parameter
  ParamNamesType ParamNames;
  //!> contains a lookup from parameter name to enumeration index
  ParamLookupType ParamLookup;
};

/** Output operator for the contents of MpqcParser_Parameters::params.
 *
 * @param ost output stream
 * @param params reference to MpqcParser_Parameters containing params.
 * @return reference to output stream for concatenation
 */
std::ostream & operator << (std::ostream& ost, const MpqcParser_Parameters &params);

/** Input operator for a list of parameters to place into \a params.
 *
 * @param ist input stream
 * @param params parameters to parse into
 * @return input stream for concatenation
 */
std::istream & operator >> (std::istream& ist, MpqcParser_Parameters &params);

#endif /* MPQCPARSER_PARAMETERS_HPP_ */
