/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * FormatParser.cpp
 *
 *  Created on: Mar 1, 2010
 *      Author: metzler
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "FormatParser.hpp"
#include <iostream>

using namespace std;

/**
 * Constructor.
 */
FormatParser::FormatParser() :
  Observer("FormatParser"),
  parameters(NULL),
  saveStream(NULL)
{
  ChangeTracker::getInstance().signOn(this);
  World::getInstance().signOn(this, World::getInstance().getChannel(World::AtomInserted));
  World::getInstance().signOn(this, World::getInstance().getChannel(World::AtomRemoved));
}

/**
 * Destructor.
 */
FormatParser::~FormatParser() {
  ChangeTracker::getInstance().signOff(this);
  World::getInstance().signOff(this, World::getInstance().getChannel(World::AtomInserted));
  World::getInstance().signOff(this, World::getInstance().getChannel(World::AtomRemoved));
}

/**
 * Update operation which can be invoked by the observable (which should be the
 * change tracker here).
 */
void FormatParser::update(Observable *publisher) {
  if (!saveStream) {
    cerr << "Please invoke setOstream() so the parser knows where to save the World's data." << endl;
    return;
  }

  std::vector<atom *> atoms = World::getInstance().getAllAtoms();
  save(saveStream, atoms);
}

/**
 * With this, each format parser is informed about specific changes in the World.
 */
void FormatParser::recieveNotification(Observable *publisher, Notification_ptr notification) {
  switch (notification->getChannelNo()) {
    case World::AtomInserted:
      AtomInserted(World::getInstance().lastChanged<atom>()->getId());
      break;
    case World::AtomRemoved:
      AtomRemoved(World::getInstance().lastChanged<atom>()->getId());
      break;
    default:
      ASSERT(0,
          "FormatParser::recieveNotification() - unknown notification "
          +toString(notification->getChannelNo())+" received.");
      break;
  }
}

/**
 * The observable can tell when it dies.
 */
void FormatParser::subjectKilled(Observable *publisher) {}


/**
 * Sets the output stream for save, so the save() method can be invoked on update
 * automatically.
 *
 * \param ostream where to save the World's state
 */
void FormatParser::setOstream(ostream* output) {
  saveStream = output;
}
