/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RandomNumberEngine_Parameters.cpp
 *
 *  Created on: Jan 6, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <boost/tokenizer.hpp>
#include <string>
#include <map>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include "RandomNumberEngine_Parameters.hpp"

#include "RandomNumberEngine.hpp"

RandomNumberEngine_Parameters::ParamNames RandomNumberEngine_Parameters::params;
const double RandomNumberEngine_Parameters::noset_value = -1.;

RandomNumberEngine_Parameters::RandomNumberEngine_Parameters() :
  seed(noset_value)
{
  // if param names map not yet filled, do so
  if (params.begin() == params.end()) {
    params["seed"] = param_seed;
  }
}

RandomNumberEngine_Parameters::RandomNumberEngine_Parameters(
    const RandomNumberEngine_Parameters &_ref)
{
  // check self assignment
  if (&_ref != this) {
    seed = _ref.seed;
  }
}

RandomNumberEngine_Parameters::~RandomNumberEngine_Parameters()
{}

void RandomNumberEngine_Parameters::getParameters(const RandomNumberEngine* const _engine)
{
  seed = _engine->getseed();
}

void RandomNumberEngine_Parameters::update(const RandomNumberEngine_Parameters &_params)
{
  seed = _params.seed != noset_value ? _params.seed : seed;
}

bool RandomNumberEngine_Parameters::isDefault() const
{
  bool status = true;
  status = status && (seed == noset_value);
  return status;
}

std::ostream & operator << (std::ostream& ost, const RandomNumberEngine_Parameters &params)
{
  std::ostringstream output;
  output << "seed=" << params.seed << ";";
  ost << output.str();
  return ost;
}

// TODO: These should throw exceptions such that invalid user entry can be detected.
std::istream & operator >> (std::istream& ist, RandomNumberEngine_Parameters &params)
{
  typedef boost::tokenizer<boost::char_separator<char> >
      tokenizer;
  boost::char_separator<char> semicolonsep(";");
  boost::char_separator<char> equalitysep("=");
  std::string line;
  double tmp;
  std::getline( ist, line );
  //DoLog(0) && (Log() << Verbose(0) << "INFO: full line of parameters is '" << line << "'" << std::endl);
  tokenizer tokens(line, semicolonsep);
  ASSERT(tokens.begin() != tokens.end(),
      "operator<< on RandomNumberEngine_Parameters - empty string, need at least ';'!");
  for (tokenizer::iterator tok_iter = tokens.begin();
        tok_iter != tokens.end(); ++tok_iter) {
    tokenizer paramtokens(*tok_iter, equalitysep);
    if (paramtokens.begin() != paramtokens.end()) {
      tokenizer::iterator tok_paramiter = paramtokens.begin();
      tokenizer::iterator tok_valueiter = tok_paramiter;
      tokenizer::iterator tok_checkiter = ++tok_valueiter;
      ASSERT(tok_paramiter != paramtokens.end(),
          "operator<< on RandomNumberEngine_Parameters - missing value before '=' in token"
          +*tok_iter+"!");
      ASSERT(tok_valueiter != paramtokens.end(),
          "operator<< on RandomNumberEngine_Parameters - missing value after '=' in token"
          +*tok_iter+"!");
      ++tok_checkiter;
      ASSERT(tok_checkiter == paramtokens.end(),
          "operator<< on RandomNumberEngine_Parameters - still tokens before ';' in token"
          +*tok_iter+": "+*tok_checkiter+"!");
      DoLog(0) && (Log() << Verbose(0) << "INFO: Token pair is " << *tok_paramiter << "," << *tok_valueiter << std::endl);
      std::stringstream value(*tok_valueiter);
      value >> tmp;
      if ((RandomNumberEngine_Parameters::params.count(*tok_paramiter) != 0)
          && (tmp != RandomNumberEngine_Parameters::noset_value)) {
        switch (RandomNumberEngine_Parameters::params[*tok_paramiter]) {
          case (RandomNumberEngine_Parameters::param_seed):
              params.seed = tmp;
              break;
          default:
              std::cerr << "This is weird, we cannot get here!" << std::endl;
        }
        DoLog(1) && (Log() << Verbose(1) << "ACCEPT: Value for token " << *tok_paramiter << " set." << std::endl);
      } else {
        DoLog(1) && (Log() << Verbose(1) << "REJECT: Value for token " << *tok_paramiter << " not specified." << std::endl);
      }
    } else {
      ist.setstate(std::ios::eofbit);
    }
  }
  return ist;
}
