/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * RandomNumberDistribution_Parameters.cpp
 *
 *  Created on: Jan 6, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <boost/tokenizer.hpp>
#include <string>
#include <map>

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"

#include "RandomNumberDistribution_Parameters.hpp"

#include "RandomNumberDistribution.hpp"

RandomNumberDistribution_Parameters::ParamNames RandomNumberDistribution_Parameters::params;
const double RandomNumberDistribution_Parameters::noset_value = -1.;

RandomNumberDistribution_Parameters::RandomNumberDistribution_Parameters() :
  min(noset_value),
  max(noset_value),
  p(noset_value),
  t(noset_value),
  median(noset_value),
  sigma(noset_value),
  alpha(noset_value),
  mean(noset_value),
  a(noset_value),
  b(noset_value),
  c(noset_value),
  lambda(noset_value)
{
  // if param names map not yet filled, do so
  if (params.begin() == params.end()) {
    params["min"] = param_min;
    params["max"] = param_max;
    params["p"] = param_p;
    params["t"] = param_t;
    params["median"] = param_median;
    params["sigma"] = param_sigma;
    params["alpha"] = param_alpha;
    params["mean"] = param_mean;
    params["a"] = param_a;
    params["b"] = param_b;
    params["c"] = param_c;
    params["lambda"] = param_lambda;
  }
}

RandomNumberDistribution_Parameters::RandomNumberDistribution_Parameters(
    const RandomNumberDistribution_Parameters &_ref)
{
  // check self assignment
  if (&_ref != this) {
    min = _ref.min;
    max = _ref.max;
    p = _ref.p;
    p = _ref.t;
    median = _ref.median;
    sigma = _ref.sigma;
    alpha = _ref.alpha;
    mean = _ref.mean;
    a = _ref.a;
    b = _ref.b;
    c = _ref.c;
    lambda = _ref.lambda;
  }
}

RandomNumberDistribution_Parameters::~RandomNumberDistribution_Parameters()
{}

void RandomNumberDistribution_Parameters::getParameters(const RandomNumberDistribution* const _distribution)
{
  min = _distribution->min();
  max = _distribution->max();
  p = _distribution->p();
  t = _distribution->t();
  median = _distribution->median();
  sigma = _distribution->sigma();
  alpha = _distribution->alpha();
  mean = _distribution->mean();
  a = _distribution->a();
  b = _distribution->b();
  c = _distribution->c();
  lambda = _distribution->lambda();
}

void RandomNumberDistribution_Parameters::update(const RandomNumberDistribution_Parameters &_params)
{
  min = _params.min != noset_value ? _params.min : min;
  max = _params.max != noset_value ? _params.max : max;
  p = _params.p != noset_value ? _params.p : p;
  p = _params.t != noset_value ? _params.t : t;
  median = _params.median != noset_value ? _params.median : median;
  sigma = _params.sigma != noset_value ? _params.sigma : sigma;
  alpha = _params.alpha != noset_value ? _params.alpha : alpha;
  mean = _params.mean != noset_value ? _params.mean : mean;
  a = _params.a != noset_value ? _params.a : a;
  b = _params.b != noset_value ? _params.b : b;
  c = _params.c != noset_value ? _params.c : c;
  lambda = _params.lambda != noset_value ? _params.lambda : lambda;
}

bool RandomNumberDistribution_Parameters::isDefault() const
{
  bool status = true;
  status = status && (min == noset_value);
  status = status && (max == noset_value);
  status = status && (p == noset_value);
  status = status && (t == noset_value);
  status = status && (median == noset_value);
  status = status && (sigma == noset_value);
  status = status && (alpha == noset_value);
  status = status && (mean == noset_value);
  status = status && (a == noset_value);
  status = status && (b == noset_value);
  status = status && (c == noset_value);
  status = status && (lambda == noset_value);
  return status;
}

std::ostream & operator << (std::ostream& ost, const RandomNumberDistribution_Parameters &params)
{
  std::ostringstream output;
  output << "min=" << params.min << ";";
  output << "max=" << params.max << ";";
  output << "p=" << params.p << ";";
  output << "t=" << params.t << ";";
  output << "median=" << params.median << ";";
  output << "sigma=" << params.sigma << ";";
  output << "alpha=" << params.alpha << ";";
  output << "mean=" << params.mean << ";";
  output << "a=" << params.a << ";";
  output << "b=" << params.b << ";";
  output << "c=" << params.c << ";";
  output << "lambda=" << params.lambda << ";" << std::endl;
  ost << output.str();
  return ost;
}

// TODO: These should throw exceptions such that invalid user entry can be detected.
std::istream & operator >> (std::istream& ist, RandomNumberDistribution_Parameters &params)
{
  typedef boost::tokenizer<boost::char_separator<char> >
      tokenizer;
  boost::char_separator<char> semicolonsep(";");
  boost::char_separator<char> equalitysep("=");
  std::string line;
  double tmp;
  std::getline( ist, line );
  //DoLog(0) && (Log() << Verbose(0) << "INFO: full line of parameters is '" << line << "'" << std::endl);
  tokenizer tokens(line, semicolonsep);
  ASSERT(tokens.begin() != tokens.end(),
      "operator<< on RandomNumberDistribution_Parameters - empty string, need at least ';'!");
  for (tokenizer::iterator tok_iter = tokens.begin();
        tok_iter != tokens.end(); ++tok_iter) {
    tokenizer paramtokens(*tok_iter, equalitysep);
    if (paramtokens.begin() != paramtokens.end()) {
      tokenizer::iterator tok_paramiter = paramtokens.begin();
      tokenizer::iterator tok_valueiter = tok_paramiter;
      tokenizer::iterator tok_checkiter = ++tok_valueiter;
      ASSERT(tok_paramiter != paramtokens.end(),
          "operator<< on RandomNumberDistribution_Parameters - missing value before '='!");
      ASSERT(tok_valueiter != paramtokens.end(),
          "operator<< on RandomNumberDistribution_Parameters - missing value after '='!");
      ++tok_checkiter;
      ASSERT(tok_checkiter == paramtokens.end(),
          "operator<< on RandomNumberDistribution_Parameters - still tokens before ';'!");
      DoLog(0) && (Log() << Verbose(0) << "INFO: Token pair is " << *tok_paramiter << "," << *tok_valueiter << std::endl);
      std::stringstream value(*tok_valueiter);
      value >> tmp;
      if ((RandomNumberDistribution_Parameters::params.count(*tok_paramiter) != 0)
          && (tmp != RandomNumberDistribution_Parameters::noset_value)) {
        switch (RandomNumberDistribution_Parameters::params[*tok_paramiter]) {
          case (RandomNumberDistribution_Parameters::param_min):
              params.min = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_max):
              params.max = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_p):
              params.p = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_t):
              params.t = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_median):
              params.median = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_sigma):
              params.sigma = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_alpha):
              params.alpha = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_mean):
              params.mean = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_a):
              params.a = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_b):
              params.b = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_c):
              params.c = tmp;
              break;
          case (RandomNumberDistribution_Parameters::param_lambda):
              params.lambda = tmp;
              break;
          default:
              std::cerr << "This is weird, we cannot get here!" << std::endl;
        }
        DoLog(1) && (Log() << Verbose(1) << "ACCEPT: Value for token " << *tok_paramiter << " set." << std::endl);
      } else {
        DoLog(1) && (Log() << Verbose(1) << "REJECT: Value for token " << *tok_paramiter << " not specified." << std::endl);
      }
    } else {
      ist.setstate(std::ios::eofbit);
    }
  }
  return ist;
}
