/*
 * tesselation.cpp
 *
 *  Created on: Aug 3, 2009
 *      Author: heber
 */

#include <fstream>

#include "linkedcell.hpp"
#include "tesselation.hpp"
#include "tesselationhelpers.hpp"
#include "vector.hpp"
#include "verbose.hpp"

class molecule;

// ======================================== Points on Boundary =================================

/** Constructor of BoundaryPointSet.
 */
BoundaryPointSet::BoundaryPointSet()
{
  LinesCount = 0;
  Nr = -1;
  value = 0.;
};

/** Constructor of BoundaryPointSet with Tesselpoint.
 * \param *Walker TesselPoint this boundary point represents
 */
BoundaryPointSet::BoundaryPointSet(TesselPoint *Walker)
{
  node = Walker;
  LinesCount = 0;
  Nr = Walker->nr;
  value = 0.;
};

/** Destructor of BoundaryPointSet.
 * Sets node to NULL to avoid removing the original, represented TesselPoint.
 * \note When removing point from a class Tesselation, use RemoveTesselationPoint()
 */
BoundaryPointSet::~BoundaryPointSet()
{
  //cout << Verbose(5) << "Erasing point nr. " << Nr << "." << endl;
  if (!lines.empty())
    cerr << "WARNING: Memory Leak! I " << *this << " am still connected to some lines." << endl;
  node = NULL;
};

/** Add a line to the LineMap of this point.
 * \param *line line to add
 */
void BoundaryPointSet::AddLine(class BoundaryLineSet *line)
{
  cout << Verbose(6) << "Adding " << *this << " to line " << *line << "."
      << endl;
  if (line->endpoints[0] == this)
    {
      lines.insert(LinePair(line->endpoints[1]->Nr, line));
    }
  else
    {
      lines.insert(LinePair(line->endpoints[0]->Nr, line));
    }
  LinesCount++;
};

/** output operator for BoundaryPointSet.
 * \param &ost output stream
 * \param &a boundary point
 */
ostream & operator <<(ostream &ost, BoundaryPointSet &a)
{
  ost << "[" << a.Nr << "|" << a.node->Name << " at " << *a.node->node << "]";
  return ost;
}
;

// ======================================== Lines on Boundary =================================

/** Constructor of BoundaryLineSet.
 */
BoundaryLineSet::BoundaryLineSet()
{
  for (int i = 0; i < 2; i++)
    endpoints[i] = NULL;
  Nr = -1;
};

/** Constructor of BoundaryLineSet with two endpoints.
 * Adds line automatically to each endpoints' LineMap
 * \param *Point[2] array of two boundary points
 * \param number number of the list
 */
BoundaryLineSet::BoundaryLineSet(class BoundaryPointSet *Point[2], int number)
{
  // set number
  Nr = number;
  // set endpoints in ascending order
  SetEndpointsOrdered(endpoints, Point[0], Point[1]);
  // add this line to the hash maps of both endpoints
  Point[0]->AddLine(this); //Taken out, to check whether we can avoid unwanted double adding.
  Point[1]->AddLine(this); //
  // clear triangles list
  cout << Verbose(5) << "New Line with endpoints " << *this << "." << endl;
};

/** Destructor for BoundaryLineSet.
 * Removes itself from each endpoints' LineMap, calling RemoveTrianglePoint() when point not connected anymore.
 * \note When removing lines from a class Tesselation, use RemoveTesselationLine()
 */
BoundaryLineSet::~BoundaryLineSet()
{
  int Numbers[2];

  // get other endpoint number of finding copies of same line
  if (endpoints[1] != NULL)
    Numbers[0] = endpoints[1]->Nr;
  else
    Numbers[0] = -1;
  if (endpoints[0] != NULL)
    Numbers[1] = endpoints[0]->Nr;
  else
    Numbers[1] = -1;

  for (int i = 0; i < 2; i++) {
    if (endpoints[i] != NULL) {
      if (Numbers[i] != -1) { // as there may be multiple lines with same endpoints, we have to go through each and find in the endpoint's line list this line set
        pair<LineMap::iterator, LineMap::iterator> erasor = endpoints[i]->lines.equal_range(Numbers[i]);
        for (LineMap::iterator Runner = erasor.first; Runner != erasor.second; Runner++)
          if ((*Runner).second == this) {
            //cout << Verbose(5) << "Removing Line Nr. " << Nr << " in boundary point " << *endpoints[i] << "." << endl;
            endpoints[i]->lines.erase(Runner);
            break;
          }
      } else { // there's just a single line left
        if (endpoints[i]->lines.erase(Nr)) {
          //cout << Verbose(5) << "Removing Line Nr. " << Nr << " in boundary point " << *endpoints[i] << "." << endl;
        }
      }
      if (endpoints[i]->lines.empty()) {
        //cout << Verbose(5) << *endpoints[i] << " has no more lines it's attached to, erasing." << endl;
        if (endpoints[i] != NULL) {
          delete(endpoints[i]);
          endpoints[i] = NULL;
        }
      }
    }
  }
  if (!triangles.empty())
    cerr << "WARNING: Memory Leak! I " << *this << " am still connected to some triangles." << endl;
};

/** Add triangle to TriangleMap of this boundary line.
 * \param *triangle to add
 */
void BoundaryLineSet::AddTriangle(class BoundaryTriangleSet *triangle)
{
  cout << Verbose(6) << "Add " << triangle->Nr << " to line " << *this << "." << endl;
  triangles.insert(TrianglePair(triangle->Nr, triangle));
};

/** Checks whether we have a common endpoint with given \a *line.
 * \param *line other line to test
 * \return true - common endpoint present, false - not connected
 */
bool BoundaryLineSet::IsConnectedTo(class BoundaryLineSet *line)
{
  if ((endpoints[0] == line->endpoints[0]) || (endpoints[1] == line->endpoints[0]) || (endpoints[0] == line->endpoints[1]) || (endpoints[1] == line->endpoints[1]))
    return true;
  else
    return false;
};

/** Checks whether the adjacent triangles of a baseline are convex or not.
 * We sum the two angles of each height vector with respect to the center of the baseline.
 * If greater/equal M_PI than we are convex.
 * \param *out output stream for debugging
 * \return true - triangles are convex, false - concave or less than two triangles connected
 */
bool BoundaryLineSet::CheckConvexityCriterion(ofstream *out)
{
  Vector BaseLineCenter, BaseLineNormal, BaseLine, helper[2], NormalCheck;
  // get the two triangles
  if (triangles.size() != 2) {
    *out << Verbose(1) << "ERROR: Baseline " << *this << " is connected to less than two triangles, Tesselation incomplete!" << endl;
    return true;
  }
  // check normal vectors
  // have a normal vector on the base line pointing outwards
  //*out << Verbose(3) << "INFO: " << *this << " has vectors at " << *(endpoints[0]->node->node) << " and at " << *(endpoints[1]->node->node) << "." << endl;
  BaseLineCenter.CopyVector(endpoints[0]->node->node);
  BaseLineCenter.AddVector(endpoints[1]->node->node);
  BaseLineCenter.Scale(1./2.);
  BaseLine.CopyVector(endpoints[0]->node->node);
  BaseLine.SubtractVector(endpoints[1]->node->node);
  //*out << Verbose(3) << "INFO: Baseline is " << BaseLine << " and its center is at " << BaseLineCenter << "." << endl;

  BaseLineNormal.Zero();
  NormalCheck.Zero();
  double sign = -1.;
  int i=0;
  class BoundaryPointSet *node = NULL;
  for(TriangleMap::iterator runner = triangles.begin(); runner != triangles.end(); runner++) {
    //*out << Verbose(3) << "INFO: NormalVector of " << *(runner->second) << " is " << runner->second->NormalVector << "." << endl;
    NormalCheck.AddVector(&runner->second->NormalVector);
    NormalCheck.Scale(sign);
    sign = -sign;
    if (runner->second->NormalVector.NormSquared() > MYEPSILON)
      BaseLineNormal.CopyVector(&runner->second->NormalVector);   // yes, copy second on top of first
    else {
      *out << Verbose(1) << "CRITICAL: Triangle " << *runner->second << " has zero normal vector!" << endl;
      exit(255);
    }
    node = runner->second->GetThirdEndpoint(this);
    if (node != NULL) {
      //*out << Verbose(3) << "INFO: Third node for triangle " << *(runner->second) << " is " << *node << " at " << *(node->node->node) << "." << endl;
      helper[i].CopyVector(node->node->node);
      helper[i].SubtractVector(&BaseLineCenter);
      helper[i].MakeNormalVector(&BaseLine);  // we want to compare the triangle's heights' angles!
      //*out << Verbose(4) << "INFO: Height vector with respect to baseline is " << helper[i] << "." << endl;
      i++;
    } else {
      //*out << Verbose(2) << "ERROR: I cannot find third node in triangle, something's wrong." << endl;
      return true;
    }
  }
  //*out << Verbose(3) << "INFO: BaselineNormal is " << BaseLineNormal << "." << endl;
  if (NormalCheck.NormSquared() < MYEPSILON) {
    *out << Verbose(3) << "ACCEPT: Normalvectors of both triangles are the same: convex." << endl;
    return true;
  }
  BaseLineNormal.Scale(-1.);
  double angle = GetAngle(helper[0], helper[1], BaseLineNormal);
  if ((angle - M_PI) > -MYEPSILON) {
    *out << Verbose(3) << "ACCEPT: Angle is greater than pi: convex." << endl;
    return true;
  } else {
    *out << Verbose(3) << "REJECT: Angle is less than pi: concave." << endl;
    return false;
  }
}

/** Checks whether point is any of the two endpoints this line contains.
 * \param *point point to test
 * \return true - point is of the line, false - is not
 */
bool BoundaryLineSet::ContainsBoundaryPoint(class BoundaryPointSet *point)
{
  for(int i=0;i<2;i++)
    if (point == endpoints[i])
      return true;
  return false;
};

/** Returns other endpoint of the line.
 * \param *point other endpoint
 * \return NULL - if endpoint not contained in BoundaryLineSet, or pointer to BoundaryPointSet otherwise
 */
class BoundaryPointSet *BoundaryLineSet::GetOtherEndpoint(class BoundaryPointSet *point)
{
  if (endpoints[0] == point)
    return endpoints[1];
  else if (endpoints[1] == point)
    return endpoints[0];
  else
    return NULL;
};

/** output operator for BoundaryLineSet.
 * \param &ost output stream
 * \param &a boundary line
 */
ostream & operator <<(ostream &ost, BoundaryLineSet &a)
{
  ost << "[" << a.Nr << "|" << a.endpoints[0]->node->Name << " at " << *a.endpoints[0]->node->node << "," << a.endpoints[1]->node->Name << " at " << *a.endpoints[1]->node->node << "]";
  return ost;
};

// ======================================== Triangles on Boundary =================================

/** Constructor for BoundaryTriangleSet.
 */
BoundaryTriangleSet::BoundaryTriangleSet()
{
  for (int i = 0; i < 3; i++)
    {
      endpoints[i] = NULL;
      lines[i] = NULL;
    }
  Nr = -1;
};

/** Constructor for BoundaryTriangleSet with three lines.
 * \param *line[3] lines that make up the triangle
 * \param number number of triangle
 */
BoundaryTriangleSet::BoundaryTriangleSet(class BoundaryLineSet *line[3], int number)
{
  // set number
  Nr = number;
  // set lines
  cout << Verbose(5) << "New triangle " << Nr << ":" << endl;
  for (int i = 0; i < 3; i++)
    {
      lines[i] = line[i];
      lines[i]->AddTriangle(this);
    }
  // get ascending order of endpoints
  map<int, class BoundaryPointSet *> OrderMap;
  for (int i = 0; i < 3; i++)
    // for all three lines
    for (int j = 0; j < 2; j++)
      { // for both endpoints
        OrderMap.insert(pair<int, class BoundaryPointSet *> (
            line[i]->endpoints[j]->Nr, line[i]->endpoints[j]));
        // and we don't care whether insertion fails
      }
  // set endpoints
  int Counter = 0;
  cout << Verbose(6) << " with end points ";
  for (map<int, class BoundaryPointSet *>::iterator runner = OrderMap.begin(); runner
      != OrderMap.end(); runner++)
    {
      endpoints[Counter] = runner->second;
      cout << " " << *endpoints[Counter];
      Counter++;
    }
  if (Counter < 3)
    {
      cerr << "ERROR! We have a triangle with only two distinct endpoints!"
          << endl;
      //exit(1);
    }
  cout << "." << endl;
};

/** Destructor of BoundaryTriangleSet.
 * Removes itself from each of its lines' LineMap and removes them if necessary.
 * \note When removing triangles from a class Tesselation, use RemoveTesselationTriangle()
 */
BoundaryTriangleSet::~BoundaryTriangleSet()
{
  for (int i = 0; i < 3; i++) {
    if (lines[i] != NULL) {
      if (lines[i]->triangles.erase(Nr)) {
        //cout << Verbose(5) << "Triangle Nr." << Nr << " erased in line " << *lines[i] << "." << endl;
      }
      if (lines[i]->triangles.empty()) {
          //cout << Verbose(5) << *lines[i] << " is no more attached to any triangle, erasing." << endl;
          delete (lines[i]);
          lines[i] = NULL;
      }
    }
  }
  //cout << Verbose(5) << "Erasing triangle Nr." << Nr << " itself." << endl;
};

/** Calculates the normal vector for this triangle.
 * Is made unique by comparison with \a OtherVector to point in the other direction.
 * \param &OtherVector direction vector to make normal vector unique.
 */
void BoundaryTriangleSet::GetNormalVector(Vector &OtherVector)
{
  // get normal vector
  NormalVector.MakeNormalVector(endpoints[0]->node->node, endpoints[1]->node->node, endpoints[2]->node->node);

  // make it always point inward (any offset vector onto plane projected onto normal vector suffices)
  if (NormalVector.ScalarProduct(&OtherVector) > 0.)
    NormalVector.Scale(-1.);
};

/** Finds the point on the triangle \a *BTS the line defined by \a *MolCenter and \a *x crosses through.
 * We call Vector::GetIntersectionWithPlane() to receive the intersection point with the plane
 * This we test if it's really on the plane and whether it's inside the triangle on the plane or not.
 * The latter is done as follows: We calculate the cross point of one of the triangle's baseline with the line
 * given by the intersection and the third basepoint. Then, we check whether it's on the baseline (i.e. between
 * the first two basepoints) or not.
 * \param *out output stream for debugging
 * \param *MolCenter offset vector of line
 * \param *x second endpoint of line, minus \a *MolCenter is directional vector of line
 * \param *Intersection intersection on plane on return
 * \return true - \a *Intersection contains intersection on plane defined by triangle, false - zero vector if outside of triangle.
 */
bool BoundaryTriangleSet::GetIntersectionInsideTriangle(ofstream *out, Vector *MolCenter, Vector *x, Vector *Intersection)
{
  Vector CrossPoint;
  Vector helper;

  if (!Intersection->GetIntersectionWithPlane(out, &NormalVector, endpoints[0]->node->node, MolCenter, x)) {
    *out << Verbose(1) << "Alas! Intersection with plane failed - at least numerically - the intersection is not on the plane!" << endl;
    return false;
  }

  // Calculate cross point between one baseline and the line from the third endpoint to intersection
  int i=0;
  do {
    if (CrossPoint.GetIntersectionOfTwoLinesOnPlane(out, endpoints[i%3]->node->node, endpoints[(i+1)%3]->node->node, endpoints[(i+2)%3]->node->node, Intersection, &NormalVector)) {
      helper.CopyVector(endpoints[(i+1)%3]->node->node);
      helper.SubtractVector(endpoints[i%3]->node->node);
    } else 
      i++;
    if (i>2)
      break;
  } while (CrossPoint.NormSquared() < MYEPSILON);
  if (i==3) {
    *out << Verbose(1) << "ERROR: Could not find any cross points, something's utterly wrong here!" << endl;
    exit(255);
  }
  CrossPoint.SubtractVector(endpoints[i%3]->node->node);  // cross point was returned as absolute vector

  // check whether intersection is inside or not by comparing length of intersection and length of cross point
  if ((CrossPoint.NormSquared() - helper.NormSquared()) < MYEPSILON) { // inside
    return true;
  } else { // outside!
    Intersection->Zero();
    return false;
  }
};

/** Checks whether lines is any of the three boundary lines this triangle contains.
 * \param *line line to test
 * \return true - line is of the triangle, false - is not
 */
bool BoundaryTriangleSet::ContainsBoundaryLine(class BoundaryLineSet *line)
{
  for(int i=0;i<3;i++)
    if (line == lines[i])
      return true;
  return false;
};

/** Checks whether point is any of the three endpoints this triangle contains.
 * \param *point point to test
 * \return true - point is of the triangle, false - is not
 */
bool BoundaryTriangleSet::ContainsBoundaryPoint(class BoundaryPointSet *point)
{
  for(int i=0;i<3;i++)
    if (point == endpoints[i])
      return true;
  return false;
};

/** Checks whether point is any of the three endpoints this triangle contains.
 * \param *point TesselPoint to test
 * \return true - point is of the triangle, false - is not
 */
bool BoundaryTriangleSet::ContainsBoundaryPoint(class TesselPoint *point)
{
  for(int i=0;i<3;i++)
    if (point == endpoints[i]->node)
      return true;
  return false;
};

/** Checks whether three given \a *Points coincide with triangle's endpoints.
 * \param *Points[3] pointer to BoundaryPointSet
 * \return true - is the very triangle, false - is not
 */
bool BoundaryTriangleSet::IsPresentTupel(class BoundaryPointSet *Points[3])
{
  return (((endpoints[0] == Points[0])
            || (endpoints[0] == Points[1])
            || (endpoints[0] == Points[2])
          ) && (
            (endpoints[1] == Points[0])
            || (endpoints[1] == Points[1])
            || (endpoints[1] == Points[2])
          ) && (
            (endpoints[2] == Points[0])
            || (endpoints[2] == Points[1])
            || (endpoints[2] == Points[2])

          ));
};

/** Checks whether three given \a *Points coincide with triangle's endpoints.
 * \param *Points[3] pointer to BoundaryPointSet
 * \return true - is the very triangle, false - is not
 */
bool BoundaryTriangleSet::IsPresentTupel(class BoundaryTriangleSet *T)
{
  return (((endpoints[0] == T->endpoints[0])
            || (endpoints[0] == T->endpoints[1])
            || (endpoints[0] == T->endpoints[2])
          ) && (
            (endpoints[1] == T->endpoints[0])
            || (endpoints[1] == T->endpoints[1])
            || (endpoints[1] == T->endpoints[2])
          ) && (
            (endpoints[2] == T->endpoints[0])
            || (endpoints[2] == T->endpoints[1])
            || (endpoints[2] == T->endpoints[2])

          ));
};

/** Returns the endpoint which is not contained in the given \a *line.
 * \param *line baseline defining two endpoints
 * \return pointer third endpoint or NULL if line does not belong to triangle.
 */
class BoundaryPointSet *BoundaryTriangleSet::GetThirdEndpoint(class BoundaryLineSet *line)
{
  // sanity check
  if (!ContainsBoundaryLine(line))
    return NULL;
  for(int i=0;i<3;i++)
    if (!line->ContainsBoundaryPoint(endpoints[i]))
      return endpoints[i];
  // actually, that' impossible :)
  return NULL;
};

/** Calculates the center point of the triangle.
 * Is third of the sum of all endpoints.
 * \param *center central point on return.
 */
void BoundaryTriangleSet::GetCenter(Vector *center)
{
  center->Zero();
  for(int i=0;i<3;i++)
    center->AddVector(endpoints[i]->node->node);
  center->Scale(1./3.);
}

/** output operator for BoundaryTriangleSet.
 * \param &ost output stream
 * \param &a boundary triangle
 */
ostream &operator <<(ostream &ost, BoundaryTriangleSet &a)
{
  ost << "[" << a.Nr << "|" << a.endpoints[0]->node->Name << " at " << *a.endpoints[0]->node->node << ","
      << a.endpoints[1]->node->Name << " at " << *a.endpoints[1]->node->node << "," << a.endpoints[2]->node->Name << " at " << *a.endpoints[2]->node->node << "]";
  return ost;
};

// =========================================================== class TESSELPOINT ===========================================

/** Constructor of class TesselPoint.
 */
TesselPoint::TesselPoint()
{
  node = NULL;
  nr = -1;
  Name =  NULL;
};

/** Destructor for class TesselPoint.
 */
TesselPoint::~TesselPoint()
{
};

/** Prints LCNode to screen.
 */
ostream & operator << (ostream &ost, const TesselPoint &a)
{
  ost << "[" << (a.Name) << "|" << a.Name << " at " << *a.node << "]";
  return ost;
};

/** Prints LCNode to screen.
 */
ostream & TesselPoint::operator << (ostream &ost)
{
  ost << "[" << (Name) << "|" << this << "]";
  return ost;
};


// =========================================================== class POINTCLOUD ============================================

/** Constructor of class PointCloud.
 */
PointCloud::PointCloud()
{

};

/** Destructor for class PointCloud.
 */
PointCloud::~PointCloud()
{

};

// ============================ CandidateForTesselation =============================

/** Constructor of class CandidateForTesselation.
 */
CandidateForTesselation::CandidateForTesselation(TesselPoint *candidate, BoundaryLineSet* line, Vector OptCandidateCenter, Vector OtherOptCandidateCenter) {
  point = candidate;
  BaseLine = line;
  OptCenter.CopyVector(&OptCandidateCenter);
  OtherOptCenter.CopyVector(&OtherOptCandidateCenter);
};

/** Destructor for class CandidateForTesselation.
 */
CandidateForTesselation::~CandidateForTesselation() {
  point = NULL;
  BaseLine = NULL;
};

// =========================================================== class TESSELATION ===========================================

/** Constructor of class Tesselation.
 */
Tesselation::Tesselation()
{
  PointsOnBoundaryCount = 0;
  LinesOnBoundaryCount = 0;
  TrianglesOnBoundaryCount = 0;
  InternalPointer = PointsOnBoundary.begin();
  LastTriangle = NULL;
  TriangleFilesWritten = 0;
}
;

/** Destructor of class Tesselation.
 * We have to free all points, lines and triangles.
 */
Tesselation::~Tesselation()
{
  cout << Verbose(1) << "Free'ing TesselStruct ... " << endl;
  for (TriangleMap::iterator runner = TrianglesOnBoundary.begin(); runner != TrianglesOnBoundary.end(); runner++) {
    if (runner->second != NULL) {
      delete (runner->second);
      runner->second = NULL;
    } else
      cerr << "ERROR: The triangle " << runner->first << " has already been free'd." << endl;
  }
  cout << "This envelope was written to file " << TriangleFilesWritten << " times(s)." << endl;
}
;

/** PointCloud implementation of GetCenter
 * Uses PointsOnBoundary and STL stuff.
 */   
Vector * Tesselation::GetCenter(ofstream *out)
{
  Vector *Center = new Vector(0.,0.,0.);
  int num=0;
  for (GoToFirst(); (!IsEnd()); GoToNext()) {
    Center->AddVector(GetPoint()->node);
    num++;
  }
  Center->Scale(1./num);
  return Center;
};

/** PointCloud implementation of GoPoint
 * Uses PointsOnBoundary and STL stuff.
 */   
TesselPoint * Tesselation::GetPoint()
{
  return (InternalPointer->second->node);
};

/** PointCloud implementation of GetTerminalPoint.
 * Uses PointsOnBoundary and STL stuff.
 */   
TesselPoint * Tesselation::GetTerminalPoint()
{
  PointMap::iterator Runner = PointsOnBoundary.end();
  Runner--;
  return (Runner->second->node);
};

/** PointCloud implementation of GoToNext.
 * Uses PointsOnBoundary and STL stuff.
 */   
void Tesselation::GoToNext()
{
  if (InternalPointer != PointsOnBoundary.end())
    InternalPointer++;
};

/** PointCloud implementation of GoToPrevious.
 * Uses PointsOnBoundary and STL stuff.
 */   
void Tesselation::GoToPrevious()
{
  if (InternalPointer != PointsOnBoundary.begin())
    InternalPointer--;
};

/** PointCloud implementation of GoToFirst.
 * Uses PointsOnBoundary and STL stuff.
 */   
void Tesselation::GoToFirst()
{
  InternalPointer = PointsOnBoundary.begin();
};

/** PointCloud implementation of GoToLast.
 * Uses PointsOnBoundary and STL stuff.
 */   
void Tesselation::GoToLast()
{
  InternalPointer = PointsOnBoundary.end();
  InternalPointer--;
};

/** PointCloud implementation of IsEmpty.
 * Uses PointsOnBoundary and STL stuff.
 */   
bool Tesselation::IsEmpty()
{
  return (PointsOnBoundary.empty());
};

/** PointCloud implementation of IsLast.
 * Uses PointsOnBoundary and STL stuff.
 */   
bool Tesselation::IsEnd()
{
  return (InternalPointer == PointsOnBoundary.end());
};


/** Gueses first starting triangle of the convex envelope.
 * We guess the starting triangle by taking the smallest distance between two points and looking for a fitting third.
 * \param *out output stream for debugging
 * \param PointsOnBoundary set of boundary points defining the convex envelope of the cluster
 */
void
Tesselation::GuessStartingTriangle(ofstream *out)
{
  // 4b. create a starting triangle
  // 4b1. create all distances
  DistanceMultiMap DistanceMMap;
  double distance, tmp;
  Vector PlaneVector, TrialVector;
  PointMap::iterator A, B, C; // three nodes of the first triangle
  A = PointsOnBoundary.begin(); // the first may be chosen arbitrarily

  // with A chosen, take each pair B,C and sort
  if (A != PointsOnBoundary.end())
    {
      B = A;
      B++;
      for (; B != PointsOnBoundary.end(); B++)
        {
          C = B;
          C++;
          for (; C != PointsOnBoundary.end(); C++)
            {
              tmp = A->second->node->node->DistanceSquared(B->second->node->node);
              distance = tmp * tmp;
              tmp = A->second->node->node->DistanceSquared(C->second->node->node);
              distance += tmp * tmp;
              tmp = B->second->node->node->DistanceSquared(C->second->node->node);
              distance += tmp * tmp;
              DistanceMMap.insert(DistanceMultiMapPair(distance, pair<PointMap::iterator, PointMap::iterator> (B, C)));
            }
        }
    }
  //    // listing distances
  //    *out << Verbose(1) << "Listing DistanceMMap:";
  //    for(DistanceMultiMap::iterator runner = DistanceMMap.begin(); runner != DistanceMMap.end(); runner++) {
  //      *out << " " << runner->first << "(" << *runner->second.first->second << ", " << *runner->second.second->second << ")";
  //    }
  //    *out << endl;
  // 4b2. pick three baselines forming a triangle
  // 1. we take from the smallest sum of squared distance as the base line BC (with peak A) onward as the triangle candidate
  DistanceMultiMap::iterator baseline = DistanceMMap.begin();
  for (; baseline != DistanceMMap.end(); baseline++)
    {
      // we take from the smallest sum of squared distance as the base line BC (with peak A) onward as the triangle candidate
      // 2. next, we have to check whether all points reside on only one side of the triangle
      // 3. construct plane vector
      PlaneVector.MakeNormalVector(A->second->node->node,
          baseline->second.first->second->node->node,
          baseline->second.second->second->node->node);
      *out << Verbose(2) << "Plane vector of candidate triangle is ";
      PlaneVector.Output(out);
      *out << endl;
      // 4. loop over all points
      double sign = 0.;
      PointMap::iterator checker = PointsOnBoundary.begin();
      for (; checker != PointsOnBoundary.end(); checker++)
        {
          // (neglecting A,B,C)
          if ((checker == A) || (checker == baseline->second.first) || (checker
              == baseline->second.second))
            continue;
          // 4a. project onto plane vector
          TrialVector.CopyVector(checker->second->node->node);
          TrialVector.SubtractVector(A->second->node->node);
          distance = TrialVector.ScalarProduct(&PlaneVector);
          if (fabs(distance) < 1e-4) // we need to have a small epsilon around 0 which is still ok
            continue;
          *out << Verbose(3) << "Projection of " << checker->second->node->Name
              << " yields distance of " << distance << "." << endl;
          tmp = distance / fabs(distance);
          // 4b. Any have different sign to than before? (i.e. would lie outside convex hull with this starting triangle)
          if ((sign != 0) && (tmp != sign))
            {
              // 4c. If so, break 4. loop and continue with next candidate in 1. loop
              *out << Verbose(2) << "Current candidates: "
                  << A->second->node->Name << ","
                  << baseline->second.first->second->node->Name << ","
                  << baseline->second.second->second->node->Name << " leaves "
                  << checker->second->node->Name << " outside the convex hull."
                  << endl;
              break;
            }
          else
            { // note the sign for later
              *out << Verbose(2) << "Current candidates: "
                  << A->second->node->Name << ","
                  << baseline->second.first->second->node->Name << ","
                  << baseline->second.second->second->node->Name << " leave "
                  << checker->second->node->Name << " inside the convex hull."
                  << endl;
              sign = tmp;
            }
          // 4d. Check whether the point is inside the triangle (check distance to each node
          tmp = checker->second->node->node->DistanceSquared(A->second->node->node);
          int innerpoint = 0;
          if ((tmp < A->second->node->node->DistanceSquared(
              baseline->second.first->second->node->node)) && (tmp
              < A->second->node->node->DistanceSquared(
                  baseline->second.second->second->node->node)))
            innerpoint++;
          tmp = checker->second->node->node->DistanceSquared(
              baseline->second.first->second->node->node);
          if ((tmp < baseline->second.first->second->node->node->DistanceSquared(
              A->second->node->node)) && (tmp
              < baseline->second.first->second->node->node->DistanceSquared(
                  baseline->second.second->second->node->node)))
            innerpoint++;
          tmp = checker->second->node->node->DistanceSquared(
              baseline->second.second->second->node->node);
          if ((tmp < baseline->second.second->second->node->node->DistanceSquared(
              baseline->second.first->second->node->node)) && (tmp
              < baseline->second.second->second->node->node->DistanceSquared(
                  A->second->node->node)))
            innerpoint++;
          // 4e. If so, break 4. loop and continue with next candidate in 1. loop
          if (innerpoint == 3)
            break;
        }
      // 5. come this far, all on same side? Then break 1. loop and construct triangle
      if (checker == PointsOnBoundary.end())
        {
          *out << "Looks like we have a candidate!" << endl;
          break;
        }
    }
  if (baseline != DistanceMMap.end())
    {
      BPS[0] = baseline->second.first->second;
      BPS[1] = baseline->second.second->second;
      BLS[0] = new class BoundaryLineSet(BPS, LinesOnBoundaryCount);
      BPS[0] = A->second;
      BPS[1] = baseline->second.second->second;
      BLS[1] = new class BoundaryLineSet(BPS, LinesOnBoundaryCount);
      BPS[0] = baseline->second.first->second;
      BPS[1] = A->second;
      BLS[2] = new class BoundaryLineSet(BPS, LinesOnBoundaryCount);

      // 4b3. insert created triangle
      BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
      TrianglesOnBoundary.insert(TrianglePair(TrianglesOnBoundaryCount, BTS));
      TrianglesOnBoundaryCount++;
      for (int i = 0; i < NDIM; i++)
        {
          LinesOnBoundary.insert(LinePair(LinesOnBoundaryCount, BTS->lines[i]));
          LinesOnBoundaryCount++;
        }

      *out << Verbose(1) << "Starting triangle is " << *BTS << "." << endl;
    }
  else
    {
      *out << Verbose(1) << "No starting triangle found." << endl;
      exit(255);
    }
}
;

/** Tesselates the convex envelope of a cluster from a single starting triangle.
 * The starting triangle is made out of three baselines. Each line in the final tesselated cluster may belong to at most
 * 2 triangles. Hence, we go through all current lines:
 * -# if the lines contains to only one triangle
 * -# We search all points in the boundary
 *    -# if the triangle is in forward direction of the baseline (at most 90 degrees angle between vector orthogonal to
 *       baseline in triangle plane pointing out of the triangle and normal vector of new triangle)
 *    -# if the triangle with the baseline and the current point has the smallest of angles (comparison between normal vectors)
 *    -# then we have a new triangle, whose baselines we again add (or increase their TriangleCount)
 * \param *out output stream for debugging
 * \param *configuration for IsAngstroem
 * \param *cloud cluster of points
 */
void Tesselation::TesselateOnBoundary(ofstream *out, PointCloud *cloud)
{
  bool flag;
  PointMap::iterator winner;
  class BoundaryPointSet *peak = NULL;
  double SmallestAngle, TempAngle;
  Vector NormalVector, VirtualNormalVector, CenterVector, TempVector, helper, PropagationVector, *Center = NULL;
  LineMap::iterator LineChecker[2];

  Center = cloud->GetCenter(out);
  // create a first tesselation with the given BoundaryPoints
  do {
    flag = false;
    for (LineMap::iterator baseline = LinesOnBoundary.begin(); baseline != LinesOnBoundary.end(); baseline++)
      if (baseline->second->triangles.size() == 1) {
        // 5a. go through each boundary point if not _both_ edges between either endpoint of the current line and this point exist (and belong to 2 triangles)
        SmallestAngle = M_PI;

        // get peak point with respect to this base line's only triangle
        BTS = baseline->second->triangles.begin()->second; // there is only one triangle so far
        *out << Verbose(2) << "Current baseline is between " << *(baseline->second) << "." << endl;
        for (int i = 0; i < 3; i++)
          if ((BTS->endpoints[i] != baseline->second->endpoints[0]) && (BTS->endpoints[i] != baseline->second->endpoints[1]))
            peak = BTS->endpoints[i];
        *out << Verbose(3) << " and has peak " << *peak << "." << endl;

        // prepare some auxiliary vectors
        Vector BaseLineCenter, BaseLine;
        BaseLineCenter.CopyVector(baseline->second->endpoints[0]->node->node);
        BaseLineCenter.AddVector(baseline->second->endpoints[1]->node->node);
        BaseLineCenter.Scale(1. / 2.); // points now to center of base line
        BaseLine.CopyVector(baseline->second->endpoints[0]->node->node);
        BaseLine.SubtractVector(baseline->second->endpoints[1]->node->node);

        // offset to center of triangle
        CenterVector.Zero();
        for (int i = 0; i < 3; i++)
          CenterVector.AddVector(BTS->endpoints[i]->node->node);
        CenterVector.Scale(1. / 3.);
        *out << Verbose(4) << "CenterVector of base triangle is " << CenterVector << endl;

        // normal vector of triangle
        NormalVector.CopyVector(Center);
        NormalVector.SubtractVector(&CenterVector);
        BTS->GetNormalVector(NormalVector);
        NormalVector.CopyVector(&BTS->NormalVector);
        *out << Verbose(4) << "NormalVector of base triangle is " << NormalVector << endl;

        // vector in propagation direction (out of triangle)
        // project center vector onto triangle plane (points from intersection plane-NormalVector to plane-CenterVector intersection)
        PropagationVector.MakeNormalVector(&BaseLine, &NormalVector);
        TempVector.CopyVector(&CenterVector);
        TempVector.SubtractVector(baseline->second->endpoints[0]->node->node); // TempVector is vector on triangle plane pointing from one baseline egde towards center!
        //*out << Verbose(2) << "Projection of propagation onto temp: " << PropagationVector.Projection(&TempVector) << "." << endl;
        if (PropagationVector.ScalarProduct(&TempVector) > 0) // make sure normal propagation vector points outward from baseline
          PropagationVector.Scale(-1.);
        *out << Verbose(4) << "PropagationVector of base triangle is " << PropagationVector << endl;
        winner = PointsOnBoundary.end();

        // loop over all points and calculate angle between normal vector of new and present triangle
        for (PointMap::iterator target = PointsOnBoundary.begin(); target != PointsOnBoundary.end(); target++) {
          if ((target->second != baseline->second->endpoints[0]) && (target->second != baseline->second->endpoints[1])) { // don't take the same endpoints
            *out << Verbose(3) << "Target point is " << *(target->second) << ":" << endl;

            // first check direction, so that triangles don't intersect
            VirtualNormalVector.CopyVector(target->second->node->node);
            VirtualNormalVector.SubtractVector(&BaseLineCenter); // points from center of base line to target
            VirtualNormalVector.ProjectOntoPlane(&NormalVector);
            TempAngle = VirtualNormalVector.Angle(&PropagationVector);
            *out << Verbose(4) << "VirtualNormalVector is " << VirtualNormalVector << " and PropagationVector is " << PropagationVector << "." << endl;
            if (TempAngle > (M_PI/2.)) { // no bends bigger than Pi/2 (90 degrees)
              *out << Verbose(4) << "Angle on triangle plane between propagation direction and base line to " << *(target->second) << " is " << TempAngle << ", bad direction!" << endl;
              continue;
            } else
              *out << Verbose(4) << "Angle on triangle plane between propagation direction and base line to " << *(target->second) << " is " << TempAngle << ", good direction!" << endl;

            // check first and second endpoint (if any connecting line goes to target has at least not more than 1 triangle)
            LineChecker[0] = baseline->second->endpoints[0]->lines.find(target->first);
            LineChecker[1] = baseline->second->endpoints[1]->lines.find(target->first);
            if (((LineChecker[0] != baseline->second->endpoints[0]->lines.end()) && (LineChecker[0]->second->triangles.size() == 2))) {
              *out << Verbose(4) << *(baseline->second->endpoints[0]) << " has line " << *(LineChecker[0]->second) << " to " << *(target->second) << " as endpoint with " << LineChecker[0]->second->triangles.size() << " triangles." << endl;
              continue;
            }
            if (((LineChecker[1] != baseline->second->endpoints[1]->lines.end()) && (LineChecker[1]->second->triangles.size() == 2))) {
              *out << Verbose(4) << *(baseline->second->endpoints[1]) << " has line " << *(LineChecker[1]->second) << " to " << *(target->second) << " as endpoint with " << LineChecker[1]->second->triangles.size() << " triangles." << endl;
              continue;
            }

            // check whether the envisaged triangle does not already exist (if both lines exist and have same endpoint)
            if ((((LineChecker[0] != baseline->second->endpoints[0]->lines.end()) && (LineChecker[1] != baseline->second->endpoints[1]->lines.end()) && (GetCommonEndpoint(LineChecker[0]->second, LineChecker[1]->second) == peak)))) {
              *out << Verbose(4) << "Current target is peak!" << endl;
              continue;
            }

            // check for linear dependence
            TempVector.CopyVector(baseline->second->endpoints[0]->node->node);
            TempVector.SubtractVector(target->second->node->node);
            helper.CopyVector(baseline->second->endpoints[1]->node->node);
            helper.SubtractVector(target->second->node->node);
            helper.ProjectOntoPlane(&TempVector);
            if (fabs(helper.NormSquared()) < MYEPSILON) {
              *out << Verbose(4) << "Chosen set of vectors is linear dependent." << endl;
              continue;
            }

            // in case NOT both were found, create virtually this triangle, get its normal vector, calculate angle
            flag = true;
            VirtualNormalVector.MakeNormalVector(baseline->second->endpoints[0]->node->node, baseline->second->endpoints[1]->node->node, target->second->node->node);
            TempVector.CopyVector(baseline->second->endpoints[0]->node->node);
            TempVector.AddVector(baseline->second->endpoints[1]->node->node);
            TempVector.AddVector(target->second->node->node);
            TempVector.Scale(1./3.);
            TempVector.SubtractVector(Center);
            // make it always point outward
            if (VirtualNormalVector.ScalarProduct(&TempVector) < 0)
              VirtualNormalVector.Scale(-1.);
            // calculate angle
            TempAngle = NormalVector.Angle(&VirtualNormalVector);
            *out << Verbose(4) << "NormalVector is " << VirtualNormalVector << " and the angle is " << TempAngle << "." << endl;
            if ((SmallestAngle - TempAngle) > MYEPSILON) { // set to new possible winner
              SmallestAngle = TempAngle;
              winner = target;
              *out << Verbose(4) << "New winner " << *winner->second->node << " due to smaller angle between normal vectors." << endl;
            } else if (fabs(SmallestAngle - TempAngle) < MYEPSILON) { // check the angle to propagation, both possible targets are in one plane! (their normals have same angle)
              // hence, check the angles to some normal direction from our base line but in this common plane of both targets...
              helper.CopyVector(target->second->node->node);
              helper.SubtractVector(&BaseLineCenter);
              helper.ProjectOntoPlane(&BaseLine);
              // ...the one with the smaller angle is the better candidate
              TempVector.CopyVector(target->second->node->node);
              TempVector.SubtractVector(&BaseLineCenter);
              TempVector.ProjectOntoPlane(&VirtualNormalVector);
              TempAngle = TempVector.Angle(&helper);
              TempVector.CopyVector(winner->second->node->node);
              TempVector.SubtractVector(&BaseLineCenter);
              TempVector.ProjectOntoPlane(&VirtualNormalVector);
              if (TempAngle < TempVector.Angle(&helper)) {
                TempAngle = NormalVector.Angle(&VirtualNormalVector);
                SmallestAngle = TempAngle;
                winner = target;
                *out << Verbose(4) << "New winner " << *winner->second->node << " due to smaller angle " << TempAngle << " to propagation direction." << endl;
              } else
                *out << Verbose(4) << "Keeping old winner " << *winner->second->node << " due to smaller angle to propagation direction." << endl;
            } else
              *out << Verbose(4) << "Keeping old winner " << *winner->second->node << " due to smaller angle between normal vectors." << endl;
          }
        } // end of loop over all boundary points

        // 5b. The point of the above whose triangle has the greatest angle with the triangle the current line belongs to (it only belongs to one, remember!): New triangle
        if (winner != PointsOnBoundary.end()) {
          *out << Verbose(2) << "Winning target point is " << *(winner->second) << " with angle " << SmallestAngle << "." << endl;
          // create the lins of not yet present
          BLS[0] = baseline->second;
          // 5c. add lines to the line set if those were new (not yet part of a triangle), delete lines that belong to two triangles)
          LineChecker[0] = baseline->second->endpoints[0]->lines.find(winner->first);
          LineChecker[1] = baseline->second->endpoints[1]->lines.find(winner->first);
          if (LineChecker[0] == baseline->second->endpoints[0]->lines.end()) { // create
            BPS[0] = baseline->second->endpoints[0];
            BPS[1] = winner->second;
            BLS[1] = new class BoundaryLineSet(BPS, LinesOnBoundaryCount);
            LinesOnBoundary.insert(LinePair(LinesOnBoundaryCount, BLS[1]));
            LinesOnBoundaryCount++;
          } else
            BLS[1] = LineChecker[0]->second;
          if (LineChecker[1] == baseline->second->endpoints[1]->lines.end()) { // create
            BPS[0] = baseline->second->endpoints[1];
            BPS[1] = winner->second;
            BLS[2] = new class BoundaryLineSet(BPS, LinesOnBoundaryCount);
            LinesOnBoundary.insert(LinePair(LinesOnBoundaryCount, BLS[2]));
            LinesOnBoundaryCount++;
          } else
            BLS[2] = LineChecker[1]->second;
          BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
          BTS->GetCenter(&helper);
          helper.SubtractVector(Center);
          helper.Scale(-1);
          BTS->GetNormalVector(helper);
          TrianglesOnBoundary.insert(TrianglePair(TrianglesOnBoundaryCount, BTS));
          TrianglesOnBoundaryCount++;
        } else {
          *out << Verbose(1) << "I could not determine a winner for this baseline " << *(baseline->second) << "." << endl;
        }

        // 5d. If the set of lines is not yet empty, go to 5. and continue
      } else
        *out << Verbose(2) << "Baseline candidate " << *(baseline->second) << " has a triangle count of " << baseline->second->triangles.size() << "." << endl;
  } while (flag);

  // exit
  delete(Center);
};

/** Inserts all points outside of the tesselated surface into it by adding new triangles.
 * \param *out output stream for debugging
 * \param *cloud cluster of points
 * \param *LC LinkedCell structure to find nearest point quickly
 * \return true - all straddling points insert, false - something went wrong
 */
bool Tesselation::InsertStraddlingPoints(ofstream *out, PointCloud *cloud, LinkedCell *LC)
{
  Vector Intersection, Normal;
  TesselPoint *Walker = NULL;
  Vector *Center = cloud->GetCenter(out);
  list<BoundaryTriangleSet*> *triangles = NULL;
  bool AddFlag = false;
  LinkedCell *BoundaryPoints = NULL;

  *out << Verbose(1) << "Begin of InsertStraddlingPoints" << endl;

  cloud->GoToFirst();
  BoundaryPoints = new LinkedCell(this, 5.);
  while (!cloud->IsEnd()) {  // we only have to go once through all points, as boundary can become only bigger
    if (AddFlag) {
      delete(BoundaryPoints);
      BoundaryPoints = new LinkedCell(this, 5.);
      AddFlag = false;
    }
    Walker = cloud->GetPoint();
    *out << Verbose(2) << "Current point is " << *Walker << "." << endl;
    // get the next triangle
    triangles = FindClosestTrianglesToPoint(out, Walker->node, BoundaryPoints);
    BTS = triangles->front();
    if ((triangles == NULL) || (BTS->ContainsBoundaryPoint(Walker))) {
      *out << Verbose(2) << "No triangles found, probably a tesselation point itself." << endl;
      cloud->GoToNext();
      continue;
    } else {
    }
    *out << Verbose(2) << "Closest triangle is " << *BTS << "." << endl;
    // get the intersection point
    if (BTS->GetIntersectionInsideTriangle(out, Center, Walker->node, &Intersection)) {
      *out << Verbose(2) << "We have an intersection at " << Intersection << "." << endl;
      // we have the intersection, check whether in- or outside of boundary
      if ((Center->DistanceSquared(Walker->node) - Center->DistanceSquared(&Intersection)) < -MYEPSILON) {
        // inside, next!
        *out << Verbose(2) << *Walker << " is inside wrt triangle " << *BTS << "." << endl;
      } else {
        // outside!
        *out << Verbose(2) << *Walker << " is outside wrt triangle " << *BTS << "." << endl;
        class BoundaryLineSet *OldLines[3], *NewLines[3];
        class BoundaryPointSet *OldPoints[3], *NewPoint;
        // store the three old lines and old points
        for (int i=0;i<3;i++) {
          OldLines[i] = BTS->lines[i];
          OldPoints[i] = BTS->endpoints[i];
        }
        Normal.CopyVector(&BTS->NormalVector);
        // add Walker to boundary points
        *out << Verbose(2) << "Adding " << *Walker << " to BoundaryPoints." << endl;
        AddFlag = true;
        if (AddBoundaryPoint(Walker,0))
          NewPoint = BPS[0];
        else
          continue;
        // remove triangle
        *out << Verbose(2) << "Erasing triangle " << *BTS << "." << endl;
        TrianglesOnBoundary.erase(BTS->Nr);
        delete(BTS);
        // create three new boundary lines
        for (int i=0;i<3;i++) {
          BPS[0] = NewPoint;
          BPS[1] = OldPoints[i];
          NewLines[i] = new class BoundaryLineSet(BPS, LinesOnBoundaryCount);
          *out << Verbose(3) << "Creating new line " << *NewLines[i] << "." << endl;
          LinesOnBoundary.insert(LinePair(LinesOnBoundaryCount, NewLines[i])); // no need for check for unique insertion as BPS[0] is definitely a new one
          LinesOnBoundaryCount++;
        }
        // create three new triangle with new point
        for (int i=0;i<3;i++) { // find all baselines
          BLS[0] = OldLines[i];
          int n = 1;
          for (int j=0;j<3;j++) {
            if (NewLines[j]->IsConnectedTo(BLS[0])) {
              if (n>2) {
                *out << Verbose(1) << "ERROR: " << BLS[0] << " connects to all of the new lines?!" << endl;
                return false;
              } else
                BLS[n++] = NewLines[j];
            }
          }
          // create the triangle
          BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
          Normal.Scale(-1.);
          BTS->GetNormalVector(Normal);
          Normal.Scale(-1.);
          *out << Verbose(2) << "Created new triangle " << *BTS << "." << endl;
          TrianglesOnBoundary.insert(TrianglePair(TrianglesOnBoundaryCount, BTS));
          TrianglesOnBoundaryCount++;
        }
      }
    } else { // something is wrong with FindClosestTriangleToPoint!
      *out << Verbose(1) << "ERROR: The closest triangle did not produce an intersection!" << endl;
      return false;
    }
    cloud->GoToNext();
  }

  // exit
  delete(Center);
  *out << Verbose(1) << "End of InsertStraddlingPoints" << endl;
  return true;
};

/** Adds a point to the tesselation::PointsOnBoundary list.
 * \param *Walker point to add
 * \param n TesselStruct::BPS index to put pointer into
 * \return true - new point was added, false - point already present
 */
bool
Tesselation::AddBoundaryPoint(TesselPoint *Walker, int n)
{
  PointTestPair InsertUnique;
  BPS[n] = new class BoundaryPointSet(Walker);
  InsertUnique = PointsOnBoundary.insert(PointPair(Walker->nr, BPS[n]));
  if (InsertUnique.second) { // if new point was not present before, increase counter
    PointsOnBoundaryCount++;
    return true;
  } else {
    delete(BPS[n]);
    BPS[n] = InsertUnique.first->second;
    return false;
  }
}
;

/** Adds point to Tesselation::PointsOnBoundary if not yet present.
 * Tesselation::TPS is set to either this new BoundaryPointSet or to the existing one of not unique.
 * @param Candidate point to add
 * @param n index for this point in Tesselation::TPS array
 */
void
Tesselation::AddTesselationPoint(TesselPoint* Candidate, int n)
{
  PointTestPair InsertUnique;
  TPS[n] = new class BoundaryPointSet(Candidate);
  InsertUnique = PointsOnBoundary.insert(PointPair(Candidate->nr, TPS[n]));
  if (InsertUnique.second) { // if new point was not present before, increase counter
    PointsOnBoundaryCount++;
  } else {
    delete TPS[n];
    cout << Verbose(4) << "Node " << *((InsertUnique.first)->second->node) << " is already present in PointsOnBoundary." << endl;
    TPS[n] = (InsertUnique.first)->second;
  }
}
;

/** Function tries to add line from current Points in BPS to BoundaryLineSet.
 * If successful it raises the line count and inserts the new line into the BLS,
 * if unsuccessful, it writes the line which had been present into the BLS, deleting the new constructed one.
 * @param *a first endpoint
 * @param *b second endpoint
 * @param n index of Tesselation::BLS giving the line with both endpoints
 */
void Tesselation::AddTesselationLine(class BoundaryPointSet *a, class BoundaryPointSet *b, int n) {
  bool insertNewLine = true;

  if (a->lines.find(b->node->nr) != a->lines.end()) {
    LineMap::iterator FindLine = a->lines.find(b->node->nr);
    pair<LineMap::iterator,LineMap::iterator> FindPair;
    FindPair = a->lines.equal_range(b->node->nr);
    cout << Verbose(5) << "INFO: There is at least one line between " << *a << " and " << *b << ": " << *(FindLine->second) << "." << endl;

    for (FindLine = FindPair.first; FindLine != FindPair.second; FindLine++) {
      // If there is a line with less than two attached triangles, we don't need a new line.
      if (FindLine->second->triangles.size() < 2) {
        insertNewLine = false;
        cout << Verbose(4) << "Using existing line " << *FindLine->second << endl;

        BPS[0] = FindLine->second->endpoints[0];
        BPS[1] = FindLine->second->endpoints[1];
        BLS[n] = FindLine->second;

        break;
      }
    }
  }

  if (insertNewLine) {
    AlwaysAddTesselationTriangleLine(a, b, n);
  }
}
;

/**
 * Adds lines from each of the current points in the BPS to BoundaryLineSet.
 * Raises the line count and inserts the new line into the BLS.
 *
 * @param *a first endpoint
 * @param *b second endpoint
 * @param n index of Tesselation::BLS giving the line with both endpoints
 */
void Tesselation::AlwaysAddTesselationTriangleLine(class BoundaryPointSet *a, class BoundaryPointSet *b, int n)
{
  cout << Verbose(4) << "Adding line [" << LinesOnBoundaryCount << "|" << *(a->node) << " and " << *(b->node) << "." << endl;
  BPS[0] = a;
  BPS[1] = b;
  BLS[n] = new class BoundaryLineSet(BPS, LinesOnBoundaryCount);  // this also adds the line to the local maps
  // add line to global map
  LinesOnBoundary.insert(LinePair(LinesOnBoundaryCount, BLS[n]));
  // increase counter
  LinesOnBoundaryCount++;
};

/** Function adds triangle to global list.
 * Furthermore, the triangle receives the next free id and id counter \a TrianglesOnBoundaryCount is increased.
 */
void Tesselation::AddTesselationTriangle()
{
  cout << Verbose(1) << "Adding triangle to global TrianglesOnBoundary map." << endl;

  // add triangle to global map
  TrianglesOnBoundary.insert(TrianglePair(TrianglesOnBoundaryCount, BTS));
  TrianglesOnBoundaryCount++;

  // set as last new triangle
  LastTriangle = BTS;

  // NOTE: add triangle to local maps is done in constructor of BoundaryTriangleSet
};

/** Function adds triangle to global list.
 * Furthermore, the triangle number is set to \a nr.
 * \param nr triangle number
 */
void Tesselation::AddTesselationTriangle(int nr)
{
  cout << Verbose(1) << "Adding triangle to global TrianglesOnBoundary map." << endl;

  // add triangle to global map
  TrianglesOnBoundary.insert(TrianglePair(nr, BTS));

  // set as last new triangle
  LastTriangle = BTS;

  // NOTE: add triangle to local maps is done in constructor of BoundaryTriangleSet
};

/** Removes a triangle from the tesselation.
 * Removes itself from the TriangleMap's of its lines, calls for them RemoveTriangleLine() if they are no more connected.
 * Removes itself from memory.
 * \param *triangle to remove
 */
void Tesselation::RemoveTesselationTriangle(class BoundaryTriangleSet *triangle)
{
  if (triangle == NULL)
    return;
  for (int i = 0; i < 3; i++) {
    if (triangle->lines[i] != NULL) {
      cout << Verbose(5) << "Removing triangle Nr." << triangle->Nr << " in line " << *triangle->lines[i] << "." << endl;
      triangle->lines[i]->triangles.erase(triangle->Nr);
      if (triangle->lines[i]->triangles.empty()) {
          cout << Verbose(5) << *triangle->lines[i] << " is no more attached to any triangle, erasing." << endl;
          RemoveTesselationLine(triangle->lines[i]);
      } else {
        cout << Verbose(5) << *triangle->lines[i] << " is still attached to another triangle: ";
        for(TriangleMap::iterator TriangleRunner = triangle->lines[i]->triangles.begin(); TriangleRunner != triangle->lines[i]->triangles.end(); TriangleRunner++)
          cout << "[" << (TriangleRunner->second)->Nr << "|" << *((TriangleRunner->second)->endpoints[0]) << ", " << *((TriangleRunner->second)->endpoints[1]) << ", " << *((TriangleRunner->second)->endpoints[2]) << "] \t";
        cout << endl;
//        for (int j=0;j<2;j++) {
//          cout << Verbose(5) << "Lines of endpoint " << *(triangle->lines[i]->endpoints[j]) << ": ";
//          for(LineMap::iterator LineRunner = triangle->lines[i]->endpoints[j]->lines.begin(); LineRunner != triangle->lines[i]->endpoints[j]->lines.end(); LineRunner++)
//            cout << "[" << *(LineRunner->second) << "] \t";
//          cout << endl;
//        }
      }
      triangle->lines[i] = NULL;  // free'd or not: disconnect
    } else
      cerr << "ERROR: This line " << i << " has already been free'd." << endl;
  }

  if (TrianglesOnBoundary.erase(triangle->Nr))
    cout << Verbose(5) << "Removing triangle Nr. " << triangle->Nr << "." << endl;
  delete(triangle);
};

/** Removes a line from the tesselation.
 * Removes itself from each endpoints' LineMap, then removes itself from global LinesOnBoundary list and free's the line.
 * \param *line line to remove
 */
void Tesselation::RemoveTesselationLine(class BoundaryLineSet *line)
{
  int Numbers[2];

  if (line == NULL)
    return;
  // get other endpoint number for finding copies of same line
  if (line->endpoints[1] != NULL)
    Numbers[0] = line->endpoints[1]->Nr;
  else
    Numbers[0] = -1;
  if (line->endpoints[0] != NULL)
    Numbers[1] = line->endpoints[0]->Nr;
  else
    Numbers[1] = -1;

  for (int i = 0; i < 2; i++) {
    if (line->endpoints[i] != NULL) {
      if (Numbers[i] != -1) { // as there may be multiple lines with same endpoints, we have to go through each and find in the endpoint's line list this line set
        pair<LineMap::iterator, LineMap::iterator> erasor = line->endpoints[i]->lines.equal_range(Numbers[i]);
        for (LineMap::iterator Runner = erasor.first; Runner != erasor.second; Runner++)
          if ((*Runner).second == line) {
            cout << Verbose(5) << "Removing Line Nr. " << line->Nr << " in boundary point " << *line->endpoints[i] << "." << endl;
            line->endpoints[i]->lines.erase(Runner);
            break;
          }
      } else { // there's just a single line left
        if (line->endpoints[i]->lines.erase(line->Nr))
          cout << Verbose(5) << "Removing Line Nr. " << line->Nr << " in boundary point " << *line->endpoints[i] << "." << endl;
      }
      if (line->endpoints[i]->lines.empty()) {
        cout << Verbose(5) << *line->endpoints[i] << " has no more lines it's attached to, erasing." << endl;
        RemoveTesselationPoint(line->endpoints[i]);
      } else {
        cout << Verbose(5) << *line->endpoints[i] << " has still lines it's attached to: ";
        for(LineMap::iterator LineRunner = line->endpoints[i]->lines.begin(); LineRunner != line->endpoints[i]->lines.end(); LineRunner++)
          cout << "[" << *(LineRunner->second) << "] \t";
        cout << endl;
      }
      line->endpoints[i] = NULL;  // free'd or not: disconnect
    } else
      cerr << "ERROR: Endpoint " << i << " has already been free'd." << endl;
  }
  if (!line->triangles.empty())
    cerr << "WARNING: Memory Leak! I " << *line << " am still connected to some triangles." << endl;

  if (LinesOnBoundary.erase(line->Nr))
    cout << Verbose(5) << "Removing line Nr. " << line->Nr << "." << endl;
  delete(line);
};

/** Removes a point from the tesselation.
 * Checks whether there are still lines connected, removes from global PointsOnBoundary list, then free's the point.
 * \note If a point should be removed, while keep the tesselated surface intact (i.e. closed), use RemovePointFromTesselatedSurface()
 * \param *point point to remove
 */
void Tesselation::RemoveTesselationPoint(class BoundaryPointSet *point)
{
  if (point == NULL)
    return;
  if (PointsOnBoundary.erase(point->Nr))
    cout << Verbose(5) << "Removing point Nr. " << point->Nr << "." << endl;
  delete(point);
};

/** Checks whether the triangle consisting of the three points is already present.
 * Searches for the points in Tesselation::PointsOnBoundary and checks their
 * lines. If any of the three edges already has two triangles attached, false is
 * returned.
 * \param *out output stream for debugging
 * \param *Candidates endpoints of the triangle candidate
 * \return integer 0 if no triangle exists, 1 if one triangle exists, 2 if two
 *                 triangles exist which is the maximum for three points
 */
int Tesselation::CheckPresenceOfTriangle(ofstream *out, TesselPoint *Candidates[3]) {
  int adjacentTriangleCount = 0;
  class BoundaryPointSet *Points[3];

  *out << Verbose(2) << "Begin of CheckPresenceOfTriangle" << endl;
  // builds a triangle point set (Points) of the end points
  for (int i = 0; i < 3; i++) {
    PointMap::iterator FindPoint = PointsOnBoundary.find(Candidates[i]->nr);
    if (FindPoint != PointsOnBoundary.end()) {
      Points[i] = FindPoint->second;
    } else {
      Points[i] = NULL;
    }
  }

  // checks lines between the points in the Points for their adjacent triangles
  for (int i = 0; i < 3; i++) {
    if (Points[i] != NULL) {
      for (int j = i; j < 3; j++) {
        if (Points[j] != NULL) {
          LineMap::iterator FindLine = Points[i]->lines.find(Points[j]->node->nr);
          for (; (FindLine != Points[i]->lines.end()) && (FindLine->first == Points[j]->node->nr); FindLine++) {
            TriangleMap *triangles = &FindLine->second->triangles;
            *out << Verbose(3) << "Current line is " << FindLine->first << ": " << *(FindLine->second) << " with triangles " << triangles << "." << endl;
            for (TriangleMap::iterator FindTriangle = triangles->begin(); FindTriangle != triangles->end(); FindTriangle++) {
              if (FindTriangle->second->IsPresentTupel(Points)) {
                adjacentTriangleCount++;
              }
            }
            *out << Verbose(3) << "end." << endl;
          }
          // Only one of the triangle lines must be considered for the triangle count.
          //*out << Verbose(2) << "Found " << adjacentTriangleCount << " adjacent triangles for the point set." << endl;
          //return adjacentTriangleCount;
        }
      }
    }
  }

  *out << Verbose(2) << "Found " << adjacentTriangleCount << " adjacent triangles for the point set." << endl;
  *out << Verbose(2) << "End of CheckPresenceOfTriangle" << endl;
  return adjacentTriangleCount;
};

/** Checks whether the triangle consisting of the three points is already present.
 * Searches for the points in Tesselation::PointsOnBoundary and checks their
 * lines. If any of the three edges already has two triangles attached, false is
 * returned.
 * \param *out output stream for debugging
 * \param *Candidates endpoints of the triangle candidate
 * \return NULL - none found or pointer to triangle
 */
class BoundaryTriangleSet * Tesselation::GetPresentTriangle(ofstream *out, TesselPoint *Candidates[3])
{
  class BoundaryTriangleSet *triangle = NULL;
  class BoundaryPointSet *Points[3];

  // builds a triangle point set (Points) of the end points
  for (int i = 0; i < 3; i++) {
    PointMap::iterator FindPoint = PointsOnBoundary.find(Candidates[i]->nr);
    if (FindPoint != PointsOnBoundary.end()) {
      Points[i] = FindPoint->second;
    } else {
      Points[i] = NULL;
    }
  }

  // checks lines between the points in the Points for their adjacent triangles
  for (int i = 0; i < 3; i++) {
    if (Points[i] != NULL) {
      for (int j = i; j < 3; j++) {
        if (Points[j] != NULL) {
          LineMap::iterator FindLine = Points[i]->lines.find(Points[j]->node->nr);
          for (; (FindLine != Points[i]->lines.end()) && (FindLine->first == Points[j]->node->nr); FindLine++) {
            TriangleMap *triangles = &FindLine->second->triangles;
            for (TriangleMap::iterator FindTriangle = triangles->begin(); FindTriangle != triangles->end(); FindTriangle++) {
              if (FindTriangle->second->IsPresentTupel(Points)) {
                if ((triangle == NULL) || (triangle->Nr > FindTriangle->second->Nr))
                  triangle = FindTriangle->second;
              }
            }
          }
          // Only one of the triangle lines must be considered for the triangle count.
          //*out << Verbose(2) << "Found " << adjacentTriangleCount << " adjacent triangles for the point set." << endl;
          //return adjacentTriangleCount;
        }
      }
    }
  }

  return triangle;
};


/** Finds the starting triangle for FindNonConvexBorder().
 * Looks at the outermost point per axis, then FindSecondPointForTesselation()
 * for the second and FindNextSuitablePointViaAngleOfSphere() for the third
 * point are called.
 * \param *out output stream for debugging
 * \param RADIUS radius of virtual rolling sphere
 * \param *LC LinkedCell structure with neighbouring TesselPoint's
 */
void Tesselation::FindStartingTriangle(ofstream *out, const double RADIUS, LinkedCell *LC)
{
  cout << Verbose(1) << "Begin of FindStartingTriangle\n";
  int i = 0;
  LinkedNodes *List = NULL;
  TesselPoint* FirstPoint = NULL;
  TesselPoint* SecondPoint = NULL;
  TesselPoint* MaxPoint[NDIM];
  double maxCoordinate[NDIM];
  Vector Oben;
  Vector helper;
  Vector Chord;
  Vector SearchDirection;

  Oben.Zero();

  for (i = 0; i < 3; i++) {
    MaxPoint[i] = NULL;
    maxCoordinate[i] = -1;
  }

  // 1. searching topmost point with respect to each axis
  for (int i=0;i<NDIM;i++) { // each axis
    LC->n[i] = LC->N[i]-1; // current axis is topmost cell
    for (LC->n[(i+1)%NDIM]=0;LC->n[(i+1)%NDIM]<LC->N[(i+1)%NDIM];LC->n[(i+1)%NDIM]++)
      for (LC->n[(i+2)%NDIM]=0;LC->n[(i+2)%NDIM]<LC->N[(i+2)%NDIM];LC->n[(i+2)%NDIM]++) {
        List = LC->GetCurrentCell();
        //cout << Verbose(2) << "Current cell is " << LC->n[0] << ", " << LC->n[1] << ", " << LC->n[2] << " with No. " << LC->index << "." << endl;
        if (List != NULL) {
          for (LinkedNodes::iterator Runner = List->begin();Runner != List->end();Runner++) {
            if ((*Runner)->node->x[i] > maxCoordinate[i]) {
              cout << Verbose(2) << "New maximal for axis " << i << " node is " << *(*Runner) << " at " << *(*Runner)->node << "." << endl;
              maxCoordinate[i] = (*Runner)->node->x[i];
              MaxPoint[i] = (*Runner);
            }
          }
        } else {
          cerr << "ERROR: The current cell " << LC->n[0] << "," << LC->n[1] << "," << LC->n[2] << " is invalid!" << endl;
        }
      }
  }

  cout << Verbose(2) << "Found maximum coordinates: ";
  for (int i=0;i<NDIM;i++)
    cout << i << ": " << *MaxPoint[i] << "\t";
  cout << endl;

  BTS = NULL;
  CandidateList *OptCandidates = new CandidateList();
  for (int k=0;k<NDIM;k++) {
    Oben.Zero();
    Oben.x[k] = 1.;
    FirstPoint = MaxPoint[k];
    cout << Verbose(1) << "Coordinates of start node at " << *FirstPoint->node << "." << endl;

    double ShortestAngle;
    TesselPoint* OptCandidate = NULL;
    ShortestAngle = 999999.; // This will contain the angle, which will be always positive (when looking for second point), when looking for third point this will be the quadrant.

    FindSecondPointForTesselation(FirstPoint, Oben, OptCandidate, &ShortestAngle, RADIUS, LC); // we give same point as next candidate as its bonds are looked into in find_second_...
    SecondPoint = OptCandidate;
    if (SecondPoint == NULL)  // have we found a second point?
      continue;

    helper.CopyVector(FirstPoint->node);
    helper.SubtractVector(SecondPoint->node);
    helper.Normalize();
    Oben.ProjectOntoPlane(&helper);
    Oben.Normalize();
    helper.VectorProduct(&Oben);
    ShortestAngle = 2.*M_PI; // This will indicate the quadrant.

    Chord.CopyVector(FirstPoint->node); // bring into calling function
    Chord.SubtractVector(SecondPoint->node);
    double radius = Chord.ScalarProduct(&Chord);
    double CircleRadius = sqrt(RADIUS*RADIUS - radius/4.);
    helper.CopyVector(&Oben);
    helper.Scale(CircleRadius);
    // Now, oben and helper are two orthonormalized vectors in the plane defined by Chord (not normalized)

    // look in one direction of baseline for initial candidate
    SearchDirection.MakeNormalVector(&Chord, &Oben);  // whether we look "left" first or "right" first is not important ...

    // adding point 1 and point 2 and add the line between them
    cout << Verbose(1) << "Coordinates of start node at " << *FirstPoint->node << "." << endl;
    AddTesselationPoint(FirstPoint, 0);
    cout << Verbose(1) << "Found second point is at " << *SecondPoint->node << ".\n";
    AddTesselationPoint(SecondPoint, 1);
    AddTesselationLine(TPS[0], TPS[1], 0);

    //cout << Verbose(2) << "INFO: OldSphereCenter is at " << helper << ".\n";
    FindThirdPointForTesselation(
      Oben, SearchDirection, helper, BLS[0], NULL, *&OptCandidates, &ShortestAngle, RADIUS, LC
    );
    cout << Verbose(1) << "List of third Points is ";
    for (CandidateList::iterator it = OptCandidates->begin(); it != OptCandidates->end(); ++it) {
        cout << " " << *(*it)->point;
    }
    cout << endl;

    for (CandidateList::iterator it = OptCandidates->begin(); it != OptCandidates->end(); ++it) {
      // add third triangle point
      AddTesselationPoint((*it)->point, 2);
      // add the second and third line
      AddTesselationLine(TPS[1], TPS[2], 1);
      AddTesselationLine(TPS[0], TPS[2], 2);
      // ... and triangles to the Maps of the Tesselation class
      BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
      AddTesselationTriangle();
      // ... and calculate its normal vector (with correct orientation)
      (*it)->OptCenter.Scale(-1.);
      cout << Verbose(2) << "Anti-Oben is currently " << (*it)->OptCenter << "." << endl;
      BTS->GetNormalVector((*it)->OptCenter);  // vector to compare with should point inwards
      cout << Verbose(0) << "==> Found starting triangle consists of " << *FirstPoint << ", " << *SecondPoint << " and "
      << *(*it)->point << " with normal vector " << BTS->NormalVector << ".\n";

      // if we do not reach the end with the next step of iteration, we need to setup a new first line
      if (it != OptCandidates->end()--) {
        FirstPoint = (*it)->BaseLine->endpoints[0]->node;
        SecondPoint = (*it)->point;
        // adding point 1 and point 2 and the line between them
        AddTesselationPoint(FirstPoint, 0);
        AddTesselationPoint(SecondPoint, 1);
        AddTesselationLine(TPS[0], TPS[1], 0);
      }
      cout << Verbose(2) << "Projection is " << BTS->NormalVector.ScalarProduct(&Oben) << "." << endl;
    }
    if (BTS != NULL) // we have created one starting triangle
      break;
    else {
      // remove all candidates from the list and then the list itself
      class CandidateForTesselation *remover = NULL;
      for (CandidateList::iterator it = OptCandidates->begin(); it != OptCandidates->end(); ++it) {
        remover = *it;
        delete(remover);
      }
      OptCandidates->clear();
    }
  }

  // remove all candidates from the list and then the list itself
  class CandidateForTesselation *remover = NULL;
  for (CandidateList::iterator it = OptCandidates->begin(); it != OptCandidates->end(); ++it) {
    remover = *it;
    delete(remover);
  }
  delete(OptCandidates);
  cout << Verbose(1) << "End of FindStartingTriangle\n";
};


/** This function finds a triangle to a line, adjacent to an existing one.
 * @param out output stream for debugging
 * @param Line current baseline to search from
 * @param T current triangle which \a Line is edge of
 * @param RADIUS radius of the rolling ball
 * @param N number of found triangles
 * @param *LC LinkedCell structure with neighbouring points
 */
bool Tesselation::FindNextSuitableTriangle(ofstream *out, BoundaryLineSet &Line, BoundaryTriangleSet &T, const double& RADIUS, LinkedCell *LC)
{
  cout << Verbose(0) << "Begin of FindNextSuitableTriangle\n";
  bool result = true;
  CandidateList *OptCandidates = new CandidateList();

  Vector CircleCenter;
  Vector CirclePlaneNormal;
  Vector OldSphereCenter;
  Vector SearchDirection;
  Vector helper;
  TesselPoint *ThirdNode = NULL;
  LineMap::iterator testline;
  double ShortestAngle = 2.*M_PI; // This will indicate the quadrant.
  double radius, CircleRadius;

  cout << Verbose(1) << "Current baseline is " << Line << " of triangle " << T << "." << endl;
  for (int i=0;i<3;i++)
    if ((T.endpoints[i]->node != Line.endpoints[0]->node) && (T.endpoints[i]->node != Line.endpoints[1]->node))
      ThirdNode = T.endpoints[i]->node;

  // construct center of circle
  CircleCenter.CopyVector(Line.endpoints[0]->node->node);
  CircleCenter.AddVector(Line.endpoints[1]->node->node);
  CircleCenter.Scale(0.5);

  // construct normal vector of circle
  CirclePlaneNormal.CopyVector(Line.endpoints[0]->node->node);
  CirclePlaneNormal.SubtractVector(Line.endpoints[1]->node->node);

  // calculate squared radius of circle
  radius = CirclePlaneNormal.ScalarProduct(&CirclePlaneNormal);
  if (radius/4. < RADIUS*RADIUS) {
    CircleRadius = RADIUS*RADIUS - radius/4.;
    CirclePlaneNormal.Normalize();
    //cout << Verbose(2) << "INFO: CircleCenter is at " << CircleCenter << ", CirclePlaneNormal is " << CirclePlaneNormal << " with circle radius " << sqrt(CircleRadius) << "." << endl;

    // construct old center
    GetCenterofCircumcircle(&OldSphereCenter, T.endpoints[0]->node->node, T.endpoints[1]->node->node, T.endpoints[2]->node->node);
    helper.CopyVector(&T.NormalVector);  // normal vector ensures that this is correct center of the two possible ones
    radius = Line.endpoints[0]->node->node->DistanceSquared(&OldSphereCenter);
    helper.Scale(sqrt(RADIUS*RADIUS - radius));
    OldSphereCenter.AddVector(&helper);
    OldSphereCenter.SubtractVector(&CircleCenter);
    //cout << Verbose(2) << "INFO: OldSphereCenter is at " << OldSphereCenter << "." << endl;

    // construct SearchDirection
    SearchDirection.MakeNormalVector(&T.NormalVector, &CirclePlaneNormal);
    helper.CopyVector(Line.endpoints[0]->node->node);
    helper.SubtractVector(ThirdNode->node);
    if (helper.ScalarProduct(&SearchDirection) < -HULLEPSILON)// ohoh, SearchDirection points inwards!
      SearchDirection.Scale(-1.);
    SearchDirection.ProjectOntoPlane(&OldSphereCenter);
    SearchDirection.Normalize();
    cout << Verbose(2) << "INFO: SearchDirection is " << SearchDirection << "." << endl;
    if (fabs(OldSphereCenter.ScalarProduct(&SearchDirection)) > HULLEPSILON) {
      // rotated the wrong way!
      cerr << "ERROR: SearchDirection and RelativeOldSphereCenter are still not orthogonal!" << endl;
    }

    // add third point
    FindThirdPointForTesselation(
      T.NormalVector, SearchDirection, OldSphereCenter, &Line, ThirdNode, OptCandidates,
      &ShortestAngle, RADIUS, LC
    );

  } else {
    cout << Verbose(1) << "Circumcircle for base line " << Line << " and base triangle " << T << " is too big!" << endl;
  }

  if (OptCandidates->begin() == OptCandidates->end()) {
    cerr << "WARNING: Could not find a suitable candidate." << endl;
    return false;
  }
  cout << Verbose(1) << "Third Points are ";
  for (CandidateList::iterator it = OptCandidates->begin(); it != OptCandidates->end(); ++it) {
    cout << " " << *(*it)->point;
  }
  cout << endl;

  BoundaryLineSet *BaseRay = &Line;
  for (CandidateList::iterator it = OptCandidates->begin(); it != OptCandidates->end(); ++it) {
    cout << Verbose(1) << " Third point candidate is " << *(*it)->point
    << " with circumsphere's center at " << (*it)->OptCenter << "." << endl;
    cout << Verbose(1) << " Baseline is " << *BaseRay << endl;

    // check whether all edges of the new triangle still have space for one more triangle (i.e. TriangleCount <2)
    TesselPoint *PointCandidates[3];
    PointCandidates[0] = (*it)->point;
    PointCandidates[1] = BaseRay->endpoints[0]->node;
    PointCandidates[2] = BaseRay->endpoints[1]->node;
    int existentTrianglesCount = CheckPresenceOfTriangle(out, PointCandidates);

    BTS = NULL;
    // If there is no triangle, add it regularly.
    if (existentTrianglesCount == 0) {
      AddTesselationPoint((*it)->point, 0);
      AddTesselationPoint(BaseRay->endpoints[0]->node, 1);
      AddTesselationPoint(BaseRay->endpoints[1]->node, 2);

      if (CheckLineCriteriaForDegeneratedTriangle(TPS)) {
        AddTesselationLine(TPS[0], TPS[1], 0);
        AddTesselationLine(TPS[0], TPS[2], 1);
        AddTesselationLine(TPS[1], TPS[2], 2);

        BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
        AddTesselationTriangle();
        (*it)->OptCenter.Scale(-1.);
        BTS->GetNormalVector((*it)->OptCenter);
        (*it)->OptCenter.Scale(-1.);

        cout << "--> New triangle with " << *BTS << " and normal vector " << BTS->NormalVector
          << " for this triangle ... " << endl;
      //cout << Verbose(1) << "We have "<< TrianglesOnBoundaryCount << " for line " << *BaseRay << "." << endl;
      } else {
        cout << Verbose(1) << "WARNING: This triangle consisting of ";
        cout << *(*it)->point << ", ";
        cout << *BaseRay->endpoints[0]->node << " and ";
        cout << *BaseRay->endpoints[1]->node << " ";
        cout << "exists and is not added, as it does not seem helpful!" << endl;
        result = false;
      }
    } else if ((existentTrianglesCount >= 1) && (existentTrianglesCount <= 3)) { // If there is a planar region within the structure, we need this triangle a second time.
        AddTesselationPoint((*it)->point, 0);
        AddTesselationPoint(BaseRay->endpoints[0]->node, 1);
        AddTesselationPoint(BaseRay->endpoints[1]->node, 2);

        // We demand that at most one new degenerate line is created and that this line also already exists (which has to be the case due to existentTrianglesCount == 1)
        // i.e. at least one of the three lines must be present with TriangleCount <= 1
        if (CheckLineCriteriaForDegeneratedTriangle(TPS)) {
          AddTesselationLine(TPS[0], TPS[1], 0);
          AddTesselationLine(TPS[0], TPS[2], 1);
          AddTesselationLine(TPS[1], TPS[2], 2);

          BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
          AddTesselationTriangle();  // add to global map

          (*it)->OtherOptCenter.Scale(-1.);
          BTS->GetNormalVector((*it)->OtherOptCenter);
          (*it)->OtherOptCenter.Scale(-1.);

          cout << "--> WARNING: Special new triangle with " << *BTS << " and normal vector " << BTS->NormalVector
          << " for this triangle ... " << endl;
          cout << Verbose(1) << "We have "<< BaseRay->triangles.size() << " for line " << BaseRay << "." << endl;
        } else {
          cout << Verbose(1) << "WARNING: This triangle consisting of ";
          cout << *(*it)->point << ", ";
          cout << *BaseRay->endpoints[0]->node << " and ";
          cout << *BaseRay->endpoints[1]->node << " ";
          cout << "exists and is not added, as it does not seem helpful!" << endl;
          result = false;
        }
    } else {
      cout << Verbose(1) << "This triangle consisting of ";
      cout << *(*it)->point << ", ";
      cout << *BaseRay->endpoints[0]->node << " and ";
      cout << *BaseRay->endpoints[1]->node << " ";
      cout << "is invalid!" << endl;
      result = false;
    }

    // set baseline to new ray from ref point (here endpoints[0]->node) to current candidate (here (*it)->point))
    BaseRay = BLS[0];
    if ((BTS != NULL) && (BTS->NormalVector.NormSquared() < MYEPSILON)) {
      *out << Verbose(1) << "CRITICAL: Triangle " << *BTS << " has zero normal vector!" << endl;
      exit(255);
    }

  }

  // remove all candidates from the list and then the list itself
  class CandidateForTesselation *remover = NULL;
  for (CandidateList::iterator it = OptCandidates->begin(); it != OptCandidates->end(); ++it) {
    remover = *it;
    delete(remover);
  }
  delete(OptCandidates);
  cout << Verbose(0) << "End of FindNextSuitableTriangle\n";
  return result;
};

/** Checks whether the quadragon of the two triangles connect to \a *Base is convex.
 * We look whether the closest point on \a *Base with respect to the other baseline is outside
 * of the segment formed by both endpoints (concave) or not (convex).
 * \param *out output stream for debugging
 * \param *Base line to be flipped
 * \return NULL - convex, otherwise endpoint that makes it concave
 */
class BoundaryPointSet *Tesselation::IsConvexRectangle(ofstream *out, class BoundaryLineSet *Base)
{
  class BoundaryPointSet *Spot = NULL;
  class BoundaryLineSet *OtherBase;
  Vector *ClosestPoint;

  int m=0;
  for(TriangleMap::iterator runner = Base->triangles.begin(); runner != Base->triangles.end(); runner++)
    for (int j=0;j<3;j++) // all of their endpoints and baselines
      if (!Base->ContainsBoundaryPoint(runner->second->endpoints[j])) // and neither of its endpoints
        BPS[m++] = runner->second->endpoints[j];
  OtherBase = new class BoundaryLineSet(BPS,-1);

  *out << Verbose(3) << "INFO: Current base line is " << *Base << "." << endl;
  *out << Verbose(3) << "INFO: Other base line is " << *OtherBase << "." << endl;

  // get the closest point on each line to the other line
  ClosestPoint = GetClosestPointBetweenLine(out, Base, OtherBase);

  // delete the temporary other base line
  delete(OtherBase);

  // get the distance vector from Base line to OtherBase line
  Vector DistanceToIntersection[2], BaseLine;
  double distance[2];
  BaseLine.CopyVector(Base->endpoints[1]->node->node);
  BaseLine.SubtractVector(Base->endpoints[0]->node->node);
  for (int i=0;i<2;i++) {
    DistanceToIntersection[i].CopyVector(ClosestPoint);
    DistanceToIntersection[i].SubtractVector(Base->endpoints[i]->node->node);
    distance[i] = BaseLine.ScalarProduct(&DistanceToIntersection[i]);
  }
  delete(ClosestPoint);
  if ((distance[0] * distance[1]) > 0)  { // have same sign?
    *out << Verbose(3) << "REJECT: Both SKPs have same sign: " << distance[0] << " and " << distance[1]  << ". " << *Base << "' rectangle is concave." << endl;
    if (distance[0] < distance[1]) {
      Spot = Base->endpoints[0];
    } else {
      Spot = Base->endpoints[1];
    }
    return Spot;
  } else {  // different sign, i.e. we are in between
    *out << Verbose(3) << "ACCEPT: Rectangle of triangles of base line " << *Base << " is convex." << endl;
    return NULL;
  }

};

void Tesselation::PrintAllBoundaryPoints(ofstream *out)
{
  // print all lines
  *out << Verbose(1) << "Printing all boundary points for debugging:" << endl;
  for (PointMap::iterator PointRunner = PointsOnBoundary.begin();PointRunner != PointsOnBoundary.end(); PointRunner++)
    *out << Verbose(2) << *(PointRunner->second) << endl;
};

void Tesselation::PrintAllBoundaryLines(ofstream *out)
{
  // print all lines
  *out << Verbose(1) << "Printing all boundary lines for debugging:" << endl;
  for (LineMap::iterator LineRunner = LinesOnBoundary.begin(); LineRunner != LinesOnBoundary.end(); LineRunner++)
    *out << Verbose(2) << *(LineRunner->second) << endl;
};

void Tesselation::PrintAllBoundaryTriangles(ofstream *out)
{
  // print all triangles
  *out << Verbose(1) << "Printing all boundary triangles for debugging:" << endl;
  for (TriangleMap::iterator TriangleRunner = TrianglesOnBoundary.begin(); TriangleRunner != TrianglesOnBoundary.end(); TriangleRunner++)
    *out << Verbose(2) << *(TriangleRunner->second) << endl;
};

/** For a given boundary line \a *Base and its two triangles, picks the central baseline that is "higher".
 * \param *out output stream for debugging
 * \param *Base line to be flipped
 * \return volume change due to flipping (0 - then no flipped occured)
 */
double Tesselation::PickFarthestofTwoBaselines(ofstream *out, class BoundaryLineSet *Base)
{
  class BoundaryLineSet *OtherBase;
  Vector *ClosestPoint[2];
  double volume;

  int m=0;
  for(TriangleMap::iterator runner = Base->triangles.begin(); runner != Base->triangles.end(); runner++)
    for (int j=0;j<3;j++) // all of their endpoints and baselines
      if (!Base->ContainsBoundaryPoint(runner->second->endpoints[j])) // and neither of its endpoints
        BPS[m++] = runner->second->endpoints[j];
  OtherBase = new class BoundaryLineSet(BPS,-1);

  *out << Verbose(3) << "INFO: Current base line is " << *Base << "." << endl;
  *out << Verbose(3) << "INFO: Other base line is " << *OtherBase << "." << endl;

  // get the closest point on each line to the other line
  ClosestPoint[0] = GetClosestPointBetweenLine(out, Base, OtherBase);
  ClosestPoint[1] = GetClosestPointBetweenLine(out, OtherBase, Base);

  // get the distance vector from Base line to OtherBase line
  Vector Distance;
  Distance.CopyVector(ClosestPoint[1]);
  Distance.SubtractVector(ClosestPoint[0]);

  // calculate volume
  volume = CalculateVolumeofGeneralTetraeder(Base->endpoints[1]->node->node, OtherBase->endpoints[0]->node->node, OtherBase->endpoints[1]->node->node, Base->endpoints[0]->node->node);

  // delete the temporary other base line and the closest points
  delete(ClosestPoint[0]);
  delete(ClosestPoint[1]);
  delete(OtherBase);

  if (Distance.NormSquared() < MYEPSILON) { // check for intersection
    *out << Verbose(3) << "REJECT: Both lines have an intersection: Nothing to do." << endl;
    return false;
  } else { // check for sign against BaseLineNormal
    Vector BaseLineNormal;
    BaseLineNormal.Zero();
    if (Base->triangles.size() < 2) {
      *out << Verbose(2) << "ERROR: Less than two triangles are attached to this baseline!" << endl;
      return 0.;
    }
    for (TriangleMap::iterator runner = Base->triangles.begin(); runner != Base->triangles.end(); runner++) {
      *out << Verbose(4) << "INFO: Adding NormalVector " << runner->second->NormalVector << " of triangle " << *(runner->second) << "." << endl;
      BaseLineNormal.AddVector(&(runner->second->NormalVector));
    }
    BaseLineNormal.Scale(1./2.);

    if (Distance.ScalarProduct(&BaseLineNormal) > MYEPSILON) { // Distance points outwards, hence OtherBase higher than Base -> flip
      *out << Verbose(2) << "ACCEPT: Other base line would be higher: Flipping baseline." << endl;
      // calculate volume summand as a general tetraeder
      return volume;
    } else {  // Base higher than OtherBase -> do nothing
      *out << Verbose(2) << "REJECT: Base line is higher: Nothing to do." << endl;
      return 0.;
    }
  }
};

/** For a given baseline and its two connected triangles, flips the baseline.
 * I.e. we create the new baseline between the other two endpoints of these four
 * endpoints and reconstruct the two triangles accordingly.
 * \param *out output stream for debugging
 * \param *Base line to be flipped
 * \return pointer to allocated new baseline - flipping successful, NULL - something went awry
 */
class BoundaryLineSet * Tesselation::FlipBaseline(ofstream *out, class BoundaryLineSet *Base)
{
  class BoundaryLineSet *OldLines[4], *NewLine;
  class BoundaryPointSet *OldPoints[2];
  Vector BaseLineNormal;
  int OldTriangleNrs[2], OldBaseLineNr;
  int i,m;

  *out << Verbose(1) << "Begin of FlipBaseline" << endl;

  // calculate NormalVector for later use
  BaseLineNormal.Zero();
  if (Base->triangles.size() < 2) {
    *out << Verbose(2) << "ERROR: Less than two triangles are attached to this baseline!" << endl;
    return NULL;
  }
  for (TriangleMap::iterator runner = Base->triangles.begin(); runner != Base->triangles.end(); runner++) {
    *out << Verbose(4) << "INFO: Adding NormalVector " << runner->second->NormalVector << " of triangle " << *(runner->second) << "." << endl;
    BaseLineNormal.AddVector(&(runner->second->NormalVector));
  }
  BaseLineNormal.Scale(-1./2.); // has to point inside for BoundaryTriangleSet::GetNormalVector()

  // get the two triangles
  // gather four endpoints and four lines
  for (int j=0;j<4;j++)
    OldLines[j] = NULL;
  for (int j=0;j<2;j++)
    OldPoints[j] = NULL;
  i=0;
  m=0;
  *out << Verbose(3) << "The four old lines are: ";
  for(TriangleMap::iterator runner = Base->triangles.begin(); runner != Base->triangles.end(); runner++)
    for (int j=0;j<3;j++) // all of their endpoints and baselines
      if (runner->second->lines[j] != Base) { // pick not the central baseline
        OldLines[i++] = runner->second->lines[j];
        *out << *runner->second->lines[j] << "\t";
      }
  *out << endl;
  *out << Verbose(3) << "The two old points are: ";
  for(TriangleMap::iterator runner = Base->triangles.begin(); runner != Base->triangles.end(); runner++)
    for (int j=0;j<3;j++) // all of their endpoints and baselines
      if (!Base->ContainsBoundaryPoint(runner->second->endpoints[j])) { // and neither of its endpoints
        OldPoints[m++] = runner->second->endpoints[j];
        *out << *runner->second->endpoints[j] << "\t";
      }
  *out << endl;

  // check whether everything is in place to create new lines and triangles
  if (i<4) {
    *out << Verbose(1) << "ERROR: We have not gathered enough baselines!" << endl;
    return NULL;
  }
  for (int j=0;j<4;j++)
    if (OldLines[j] == NULL) {
      *out << Verbose(1) << "ERROR: We have not gathered enough baselines!" << endl;
      return NULL;
    }
  for (int j=0;j<2;j++)
    if (OldPoints[j] == NULL) {
      *out << Verbose(1) << "ERROR: We have not gathered enough endpoints!" << endl;
      return NULL;
    }

  // remove triangles and baseline removes itself
  *out << Verbose(3) << "INFO: Deleting baseline " << *Base << " from global list." << endl;
  OldBaseLineNr = Base->Nr;
  m=0;
  for(TriangleMap::iterator runner = Base->triangles.begin(); runner != Base->triangles.end(); runner++) {
    *out << Verbose(3) << "INFO: Deleting triangle " << *(runner->second) << "." << endl;
    OldTriangleNrs[m++] = runner->second->Nr;
    RemoveTesselationTriangle(runner->second);
  }

  // construct new baseline (with same number as old one)
  BPS[0] = OldPoints[0];
  BPS[1] = OldPoints[1];
  NewLine = new class BoundaryLineSet(BPS, OldBaseLineNr);
  LinesOnBoundary.insert(LinePair(OldBaseLineNr, NewLine)); // no need for check for unique insertion as NewLine is definitely a new one
  *out << Verbose(3) << "INFO: Created new baseline " << *NewLine << "." << endl;

  // construct new triangles with flipped baseline
  i=-1;
  if (OldLines[0]->IsConnectedTo(OldLines[2]))
    i=2;
  if (OldLines[0]->IsConnectedTo(OldLines[3]))
    i=3;
  if (i!=-1) {
    BLS[0] = OldLines[0];
    BLS[1] = OldLines[i];
    BLS[2] = NewLine;
    BTS = new class BoundaryTriangleSet(BLS, OldTriangleNrs[0]);
    BTS->GetNormalVector(BaseLineNormal);
    AddTesselationTriangle(OldTriangleNrs[0]);
    *out << Verbose(3) << "INFO: Created new triangle " << *BTS << "." << endl;

    BLS[0] = (i==2 ? OldLines[3] : OldLines[2]);
    BLS[1] = OldLines[1];
    BLS[2] = NewLine;
    BTS = new class BoundaryTriangleSet(BLS, OldTriangleNrs[1]);
    BTS->GetNormalVector(BaseLineNormal);
    AddTesselationTriangle(OldTriangleNrs[1]);
    *out << Verbose(3) << "INFO: Created new triangle " << *BTS << "." << endl;
  } else {
    *out << Verbose(1) << "The four old lines do not connect, something's utterly wrong here!" << endl;
    return NULL;
  }

  *out << Verbose(1) << "End of FlipBaseline" << endl;
  return NewLine;
};


/** Finds the second point of starting triangle.
 * \param *a first node
 * \param Oben vector indicating the outside
 * \param OptCandidate reference to recommended candidate on return
 * \param Storage[3] array storing angles and other candidate information
 * \param RADIUS radius of virtual sphere
 * \param *LC LinkedCell structure with neighbouring points
 */
void Tesselation::FindSecondPointForTesselation(TesselPoint* a, Vector Oben, TesselPoint*& OptCandidate, double Storage[3], double RADIUS, LinkedCell *LC)
{
  cout << Verbose(2) << "Begin of FindSecondPointForTesselation" << endl;
  Vector AngleCheck;
  class TesselPoint* Candidate = NULL;
  double norm = -1., angle;
  LinkedNodes *List = NULL;
  int N[NDIM], Nlower[NDIM], Nupper[NDIM];

  if (LC->SetIndexToNode(a)) {  // get cell for the starting point
    for(int i=0;i<NDIM;i++) // store indices of this cell
      N[i] = LC->n[i];
  } else {
    cerr << "ERROR: Point " << *a << " is not found in cell " << LC->index << "." << endl;
    return;
  }
  // then go through the current and all neighbouring cells and check the contained points for possible candidates
  cout << Verbose(3) << "LC Intervals from [";
  for (int i=0;i<NDIM;i++) {
  cout << " " << N[i] << "<->" << LC->N[i];
  }
  cout << "] :";
  for (int i=0;i<NDIM;i++) {
    Nlower[i] = ((N[i]-1) >= 0) ? N[i]-1 : 0;
    Nupper[i] = ((N[i]+1) < LC->N[i]) ? N[i]+1 : LC->N[i]-1;
    cout << " [" << Nlower[i] << "," << Nupper[i] << "] ";
  }
  cout << endl;


  for (LC->n[0] = Nlower[0]; LC->n[0] <= Nupper[0]; LC->n[0]++)
    for (LC->n[1] = Nlower[1]; LC->n[1] <= Nupper[1]; LC->n[1]++)
      for (LC->n[2] = Nlower[2]; LC->n[2] <= Nupper[2]; LC->n[2]++) {
        List = LC->GetCurrentCell();
        //cout << Verbose(2) << "Current cell is " << LC->n[0] << ", " << LC->n[1] << ", " << LC->n[2] << " with No. " << LC->index << "." << endl;
        if (List != NULL) {
          for (LinkedNodes::iterator Runner = List->begin(); Runner != List->end(); Runner++) {
            Candidate = (*Runner);
            // check if we only have one unique point yet ...
            if (a != Candidate) {
              // Calculate center of the circle with radius RADIUS through points a and Candidate
              Vector OrthogonalizedOben, aCandidate, Center;
              double distance, scaleFactor;

              OrthogonalizedOben.CopyVector(&Oben);
              aCandidate.CopyVector(a->node);
              aCandidate.SubtractVector(Candidate->node);
              OrthogonalizedOben.ProjectOntoPlane(&aCandidate);
              OrthogonalizedOben.Normalize();
              distance = 0.5 * aCandidate.Norm();
              scaleFactor = sqrt(((RADIUS * RADIUS) - (distance * distance)));
              OrthogonalizedOben.Scale(scaleFactor);

              Center.CopyVector(Candidate->node);
              Center.AddVector(a->node);
              Center.Scale(0.5);
              Center.AddVector(&OrthogonalizedOben);

              AngleCheck.CopyVector(&Center);
              AngleCheck.SubtractVector(a->node);
              norm = aCandidate.Norm();
              // second point shall have smallest angle with respect to Oben vector
              if (norm < RADIUS*2.) {
                angle = AngleCheck.Angle(&Oben);
                if (angle < Storage[0]) {
                  //cout << Verbose(3) << "Old values of Storage: %lf %lf \n", Storage[0], Storage[1]);
                  cout << Verbose(3) << "Current candidate is " << *Candidate << ": Is a better candidate with distance " << norm << " and angle " << angle << " to oben " << Oben << ".\n";
                  OptCandidate = Candidate;
                  Storage[0] = angle;
                  //cout << Verbose(3) << "Changing something in Storage: %lf %lf. \n", Storage[0], Storage[2]);
                } else {
                  //cout << Verbose(3) << "Current candidate is " << *Candidate << ": Looses with angle " << angle << " to a better candidate " << *OptCandidate << endl;
                }
              } else {
                //cout << Verbose(3) << "Current candidate is " << *Candidate << ": Refused due to Radius " << norm << endl;
              }
            } else {
              //cout << Verbose(3) << "Current candidate is " << *Candidate << ": Candidate is equal to first endpoint." << *a << "." << endl;
            }
          }
        } else {
          cout << Verbose(3) << "Linked cell list is empty." << endl;
        }
      }
  cout << Verbose(2) << "End of FindSecondPointForTesselation" << endl;
};


/** This recursive function finds a third point, to form a triangle with two given ones.
 * Note that this function is for the starting triangle.
 * The idea is as follows: A sphere with fixed radius is (almost) uniquely defined in space by three points
 * that sit on its boundary. Hence, when two points are given and we look for the (next) third point, then
 * the center of the sphere is still fixed up to a single parameter. The band of possible values
 * describes a circle in 3D-space. The old center of the sphere for the current base triangle gives
 * us the "null" on this circle, the new center of the candidate point will be some way along this
 * circle. The shorter the way the better is the candidate. Note that the direction is clearly given
 * by the normal vector of the base triangle that always points outwards by construction.
 * Hence, we construct a Center of this circle which sits right in the middle of the current base line.
 * We construct the normal vector that defines the plane this circle lies in, it is just in the
 * direction of the baseline. And finally, we need the radius of the circle, which is given by the rest
 * with respect to the length of the baseline and the sphere's fixed \a RADIUS.
 * Note that there is one difficulty: The circumcircle is uniquely defined, but for the circumsphere's center
 * there are two possibilities which becomes clear from the construction as seen below. Hence, we must check
 * both.
 * Note also that the acos() function is not unique on [0, 2.*M_PI). Hence, we need an additional check
 * to decide for one of the two possible angles. Therefore we need a SearchDirection and to make this check
 * sensible we need OldSphereCenter to be orthogonal to it. Either we construct SearchDirection orthogonal
 * right away, or -- what we do here -- we rotate the relative sphere centers such that this orthogonality
 * holds. Then, the normalized projection onto the SearchDirection is either +1 or -1 and thus states whether
 * the angle is uniquely in either (0,M_PI] or [M_PI, 2.*M_PI).
 * @param NormalVector normal direction of the base triangle (here the unit axis vector, \sa FindStartingTriangle())
 * @param SearchDirection general direction where to search for the next point, relative to center of BaseLine
 * @param OldSphereCenter center of sphere for base triangle, relative to center of BaseLine, giving null angle for the parameter circle
 * @param BaseLine BoundaryLineSet with the current base line
 * @param ThirdNode third point to avoid in search
 * @param candidates list of equally good candidates to return
 * @param ShortestAngle the current path length on this circle band for the current OptCandidate
 * @param RADIUS radius of sphere
 * @param *LC LinkedCell structure with neighbouring points
 */
void Tesselation::FindThirdPointForTesselation(Vector NormalVector, Vector SearchDirection, Vector OldSphereCenter, class BoundaryLineSet *BaseLine, class TesselPoint  *ThirdNode, CandidateList* &candidates, double *ShortestAngle, const double RADIUS, LinkedCell *LC)
{
  Vector CircleCenter;  // center of the circle, i.e. of the band of sphere's centers
  Vector CirclePlaneNormal; // normal vector defining the plane this circle lives in
  Vector SphereCenter;
  Vector NewSphereCenter;   // center of the sphere defined by the two points of BaseLine and the one of Candidate, first possibility
  Vector OtherNewSphereCenter;   // center of the sphere defined by the two points of BaseLine and the one of Candidate, second possibility
  Vector NewNormalVector;   // normal vector of the Candidate's triangle
  Vector helper, OptCandidateCenter, OtherOptCandidateCenter;
  LinkedNodes *List = NULL;
  double CircleRadius; // radius of this circle
  double radius;
  double alpha, Otheralpha; // angles (i.e. parameter for the circle).
  int N[NDIM], Nlower[NDIM], Nupper[NDIM];
  TesselPoint *Candidate = NULL;
  CandidateForTesselation *optCandidate = NULL;

  cout << Verbose(1) << "Begin of FindThirdPointForTesselation" << endl;

  cout << Verbose(2) << "INFO: NormalVector of BaseTriangle is " << NormalVector << "." << endl;

  // construct center of circle
  CircleCenter.CopyVector(BaseLine->endpoints[0]->node->node);
  CircleCenter.AddVector(BaseLine->endpoints[1]->node->node);
  CircleCenter.Scale(0.5);

  // construct normal vector of circle
  CirclePlaneNormal.CopyVector(BaseLine->endpoints[0]->node->node);
  CirclePlaneNormal.SubtractVector(BaseLine->endpoints[1]->node->node);

  // calculate squared radius TesselPoint *ThirdNode,f circle
  radius = CirclePlaneNormal.ScalarProduct(&CirclePlaneNormal);
  if (radius/4. < RADIUS*RADIUS) {
    CircleRadius = RADIUS*RADIUS - radius/4.;
    CirclePlaneNormal.Normalize();
    //cout << Verbose(2) << "INFO: CircleCenter is at " << CircleCenter << ", CirclePlaneNormal is " << CirclePlaneNormal << " with circle radius " << sqrt(CircleRadius) << "." << endl;

    // test whether old center is on the band's plane
    if (fabs(OldSphereCenter.ScalarProduct(&CirclePlaneNormal)) > HULLEPSILON) {
      cerr << "ERROR: Something's very wrong here: OldSphereCenter is not on the band's plane as desired by " << fabs(OldSphereCenter.ScalarProduct(&CirclePlaneNormal)) << "!" << endl;
      OldSphereCenter.ProjectOntoPlane(&CirclePlaneNormal);
    }
    radius = OldSphereCenter.ScalarProduct(&OldSphereCenter);
    if (fabs(radius - CircleRadius) < HULLEPSILON) {
      //cout << Verbose(2) << "INFO: OldSphereCenter is at " << OldSphereCenter << "." << endl;

      // check SearchDirection
      //cout << Verbose(2) << "INFO: SearchDirection is " << SearchDirection << "." << endl;
      if (fabs(OldSphereCenter.ScalarProduct(&SearchDirection)) > HULLEPSILON) {  // rotated the wrong way!
        cerr << "ERROR: SearchDirection and RelativeOldSphereCenter are not orthogonal!" << endl;
      }

      // get cell for the starting point
      if (LC->SetIndexToVector(&CircleCenter)) {
        for(int i=0;i<NDIM;i++) // store indices of this cell
        N[i] = LC->n[i];
        //cout << Verbose(2) << "INFO: Center cell is " << N[0] << ", " << N[1] << ", " << N[2] << " with No. " << LC->index << "." << endl;
      } else {
        cerr << "ERROR: Vector " << CircleCenter << " is outside of LinkedCell's bounding box." << endl;
        return;
      }
      // then go through the current and all neighbouring cells and check the contained points for possible candidates
      //cout << Verbose(2) << "LC Intervals:";
      for (int i=0;i<NDIM;i++) {
        Nlower[i] = ((N[i]-1) >= 0) ? N[i]-1 : 0;
        Nupper[i] = ((N[i]+1) < LC->N[i]) ? N[i]+1 : LC->N[i]-1;
        //cout << " [" << Nlower[i] << "," << Nupper[i] << "] ";
      }
      //cout << endl;
      for (LC->n[0] = Nlower[0]; LC->n[0] <= Nupper[0]; LC->n[0]++)
        for (LC->n[1] = Nlower[1]; LC->n[1] <= Nupper[1]; LC->n[1]++)
          for (LC->n[2] = Nlower[2]; LC->n[2] <= Nupper[2]; LC->n[2]++) {
            List = LC->GetCurrentCell();
            //cout << Verbose(2) << "Current cell is " << LC->n[0] << ", " << LC->n[1] << ", " << LC->n[2] << " with No. " << LC->index << "." << endl;
            if (List != NULL) {
              for (LinkedNodes::iterator Runner = List->begin(); Runner != List->end(); Runner++) {
                Candidate = (*Runner);

                // check for three unique points
                //cout << Verbose(2) << "INFO: Current Candidate is " << *Candidate << " at " << Candidate->node << "." << endl;
                if ((Candidate != BaseLine->endpoints[0]->node) && (Candidate != BaseLine->endpoints[1]->node) ){

                  // construct both new centers
                  GetCenterofCircumcircle(&NewSphereCenter, BaseLine->endpoints[0]->node->node, BaseLine->endpoints[1]->node->node, Candidate->node);
                  OtherNewSphereCenter.CopyVector(&NewSphereCenter);

                  if ((NewNormalVector.MakeNormalVector(BaseLine->endpoints[0]->node->node, BaseLine->endpoints[1]->node->node, Candidate->node))
                  && (fabs(NewNormalVector.ScalarProduct(&NewNormalVector)) > HULLEPSILON)
                  ) {
                    helper.CopyVector(&NewNormalVector);
                    //cout << Verbose(2) << "INFO: NewNormalVector is " << NewNormalVector << "." << endl;
                    radius = BaseLine->endpoints[0]->node->node->DistanceSquared(&NewSphereCenter);
                    if (radius < RADIUS*RADIUS) {
                      helper.Scale(sqrt(RADIUS*RADIUS - radius));
                      //cout << Verbose(2) << "INFO: Distance of NewCircleCenter to NewSphereCenter is " << helper.Norm() << " with sphere radius " << RADIUS << "." << endl;
                      NewSphereCenter.AddVector(&helper);
                      NewSphereCenter.SubtractVector(&CircleCenter);
                      //cout << Verbose(2) << "INFO: NewSphereCenter is at " << NewSphereCenter << "." << endl;

                      // OtherNewSphereCenter is created by the same vector just in the other direction
                      helper.Scale(-1.);
                      OtherNewSphereCenter.AddVector(&helper);
                      OtherNewSphereCenter.SubtractVector(&CircleCenter);
                      //cout << Verbose(2) << "INFO: OtherNewSphereCenter is at " << OtherNewSphereCenter << "." << endl;

                      alpha = GetPathLengthonCircumCircle(CircleCenter, CirclePlaneNormal, CircleRadius, NewSphereCenter, OldSphereCenter, NormalVector, SearchDirection);
                      Otheralpha = GetPathLengthonCircumCircle(CircleCenter, CirclePlaneNormal, CircleRadius, OtherNewSphereCenter, OldSphereCenter, NormalVector, SearchDirection);
                      alpha = min(alpha, Otheralpha);
                      // if there is a better candidate, drop the current list and add the new candidate
                      // otherwise ignore the new candidate and keep the list
                      if (*ShortestAngle > (alpha - HULLEPSILON)) {
                        optCandidate = new CandidateForTesselation(Candidate, BaseLine, OptCandidateCenter, OtherOptCandidateCenter);
                        if (fabs(alpha - Otheralpha) > MYEPSILON) {
                          optCandidate->OptCenter.CopyVector(&NewSphereCenter);
                          optCandidate->OtherOptCenter.CopyVector(&OtherNewSphereCenter);
                        } else {
                          optCandidate->OptCenter.CopyVector(&OtherNewSphereCenter);
                          optCandidate->OtherOptCenter.CopyVector(&NewSphereCenter);
                        }
                        // if there is an equal candidate, add it to the list without clearing the list
                        if ((*ShortestAngle - HULLEPSILON) < alpha) {
                          candidates->push_back(optCandidate);
                          cout << Verbose(2) << "ACCEPT: We have found an equally good candidate: " << *(optCandidate->point) << " with "
                            << alpha << " and circumsphere's center at " << optCandidate->OptCenter << "." << endl;
                        } else {
                          // remove all candidates from the list and then the list itself
                          class CandidateForTesselation *remover = NULL;
                          for (CandidateList::iterator it = candidates->begin(); it != candidates->end(); ++it) {
                            remover = *it;
                            delete(remover);
                          }
                          candidates->clear();
                          candidates->push_back(optCandidate);
                          cout << Verbose(2) << "ACCEPT: We have found a better candidate: " << *(optCandidate->point) << " with "
                            << alpha << " and circumsphere's center at " << optCandidate->OptCenter << "." << endl;
                        }
                        *ShortestAngle = alpha;
                        //cout << Verbose(2) << "INFO: There are " << candidates->size() << " candidates in the list now." << endl;
                      } else {
                        if ((optCandidate != NULL) && (optCandidate->point != NULL)) {
                          //cout << Verbose(2) << "REJECT: Old candidate " << *(optCandidate->point) << " with " << *ShortestAngle << " is better than new one " << *Candidate << " with " << alpha << " ." << endl;
                        } else {
                          //cout << Verbose(2) << "REJECT: Candidate " << *Candidate << " with " << alpha << " was rejected." << endl;
                        }
                      }

                    } else {
                      //cout << Verbose(2) << "REJECT: NewSphereCenter " << NewSphereCenter << " for " << *Candidate << " is too far away: " << radius << "." << endl;
                    }
                  } else {
                    //cout << Verbose(2) << "REJECT: Three points from " << *BaseLine << " and Candidate " << *Candidate << " are linear-dependent." << endl;
                  }
                } else {
                  if (ThirdNode != NULL) {
                    //cout << Verbose(2) << "REJECT: Base triangle " << *BaseLine << " and " << *ThirdNode << " contains Candidate " << *Candidate << "." << endl;
                  } else {
                    //cout << Verbose(2) << "REJECT: Base triangle " << *BaseLine << " contains Candidate " << *Candidate << "." << endl;
                  }
                }
              }
            }
          }
    } else {
      cerr << Verbose(2) << "ERROR: The projected center of the old sphere has radius " << radius << " instead of " << CircleRadius << "." << endl;
    }
  } else {
    if (ThirdNode != NULL)
      cout << Verbose(2) << "Circumcircle for base line " << *BaseLine << " and third node " << *ThirdNode << " is too big!" << endl;
    else
      cout << Verbose(2) << "Circumcircle for base line " << *BaseLine << " is too big!" << endl;
  }

  //cout << Verbose(2) << "INFO: Sorting candidate list ..." << endl;
  if (candidates->size() > 1) {
    candidates->unique();
    candidates->sort(SortCandidates);
  }

  cout << Verbose(1) << "End of FindThirdPointForTesselation" << endl;
};

/** Finds the endpoint two lines are sharing.
 * \param *line1 first line
 * \param *line2 second line
 * \return point which is shared or NULL if none
 */
class BoundaryPointSet *Tesselation::GetCommonEndpoint(class BoundaryLineSet * line1, class BoundaryLineSet * line2)
{
  class BoundaryLineSet * lines[2] =
    { line1, line2 };
  class BoundaryPointSet *node = NULL;
  map<int, class BoundaryPointSet *> OrderMap;
  pair<map<int, class BoundaryPointSet *>::iterator, bool> OrderTest;
  for (int i = 0; i < 2; i++)
    // for both lines
    for (int j = 0; j < 2; j++)
      { // for both endpoints
        OrderTest = OrderMap.insert(pair<int, class BoundaryPointSet *> (
            lines[i]->endpoints[j]->Nr, lines[i]->endpoints[j]));
        if (!OrderTest.second)
          { // if insertion fails, we have common endpoint
            node = OrderTest.first->second;
            cout << Verbose(5) << "Common endpoint of lines " << *line1
                << " and " << *line2 << " is: " << *node << "." << endl;
            j = 2;
            i = 2;
            break;
          }
      }
  return node;
};

/** Finds the triangle that is closest to a given Vector \a *x.
 * \param *out output stream for debugging
 * \param *x Vector to look from
 * \return list of BoundaryTriangleSet of nearest triangles or NULL in degenerate case.
 */
list<BoundaryTriangleSet*> * Tesselation::FindClosestTrianglesToPoint(ofstream *out, Vector *x, LinkedCell* LC)
{
  TesselPoint *trianglePoints[3];
  TesselPoint *SecondPoint = NULL;
  list<BoundaryTriangleSet*> *triangles = NULL;

  if (LinesOnBoundary.empty()) {
    *out << Verbose(0) << "Error: There is no tesselation structure to compare the point with, please create one first.";
    return NULL;
  }

  trianglePoints[0] = FindClosestPoint(x, SecondPoint, LC);
  
  // check whether closest point is "too close" :), then it's inside
  if (trianglePoints[0] == NULL) {
    *out << Verbose(2) << "Is the only point, no one else is closeby." << endl;
    return NULL;
  }
  if (trianglePoints[0]->node->DistanceSquared(x) < MYEPSILON) {
    *out << Verbose(3) << "Point is right on a tesselation point, no nearest triangle." << endl;
    PointMap::iterator PointRunner = PointsOnBoundary.find(trianglePoints[0]->nr);
    triangles = new list<BoundaryTriangleSet*>;
    if (PointRunner != PointsOnBoundary.end()) {
      for(LineMap::iterator LineRunner = PointRunner->second->lines.begin(); LineRunner != PointRunner->second->lines.end(); LineRunner++)
        for(TriangleMap::iterator TriangleRunner = LineRunner->second->triangles.begin(); TriangleRunner != LineRunner->second->triangles.end(); TriangleRunner++)
          triangles->push_back(TriangleRunner->second);
      triangles->sort();
      triangles->unique();
    } else {
      PointRunner = PointsOnBoundary.find(SecondPoint->nr);
      trianglePoints[0] = SecondPoint;
      if (PointRunner != PointsOnBoundary.end()) {
        for(LineMap::iterator LineRunner = PointRunner->second->lines.begin(); LineRunner != PointRunner->second->lines.end(); LineRunner++)
          for(TriangleMap::iterator TriangleRunner = LineRunner->second->triangles.begin(); TriangleRunner != LineRunner->second->triangles.end(); TriangleRunner++)
            triangles->push_back(TriangleRunner->second);
        triangles->sort();
        triangles->unique();
      } else {
        *out << Verbose(1) << "ERROR: I cannot find a boundary point to the tessel point " << *trianglePoints[0] << "." << endl;
        return NULL;
      }
    }
  } else {
    list<TesselPoint*> *connectedClosestPoints = GetCircleOfConnectedPoints(out, trianglePoints[0], x);
    trianglePoints[1] = connectedClosestPoints->front();
    trianglePoints[2] = connectedClosestPoints->back();
    for (int i=0;i<3;i++) {
      if (trianglePoints[i] == NULL) {
        *out << Verbose(1) << "ERROR: IsInnerPoint encounters serious error, point " << i << " not found." << endl;
      }
      //*out << Verbose(2) << "List of triangle points:" << endl;
      //*out << Verbose(3) << *trianglePoints[i] << endl;
    }

    triangles = FindTriangles(trianglePoints);
    *out << Verbose(2) << "List of possible triangles:" << endl;
    for(list<BoundaryTriangleSet*>::iterator Runner = triangles->begin(); Runner != triangles->end(); Runner++)
      *out << Verbose(3) << **Runner << endl;

    delete(connectedClosestPoints);
  }
  
  if (triangles->empty()) {
    *out << Verbose(0) << "ERROR: There is no nearest triangle. Please check the tesselation structure.";
    delete(triangles);
    return NULL;
  } else
    return triangles;
};

/** Finds closest triangle to a point.
 * This basically just takes care of the degenerate case, which is not handled in FindClosestTrianglesToPoint().
 * \param *out output stream for debugging
 * \param *x Vector to look from
 * \return list of BoundaryTriangleSet of nearest triangles or NULL.
 */
class BoundaryTriangleSet * Tesselation::FindClosestTriangleToPoint(ofstream *out, Vector *x, LinkedCell* LC)
{
  class BoundaryTriangleSet *result = NULL;
  list<BoundaryTriangleSet*> *triangles = FindClosestTrianglesToPoint(out, x, LC);
  Vector Center;

  if (triangles == NULL)
    return NULL;

  if (triangles->size() == 1) { // there is no degenerate case
    result = triangles->front();
    *out << Verbose(2) << "Normal Vector of this triangle is " << result->NormalVector << "." << endl;
  } else {
    result = triangles->front();
    result->GetCenter(&Center);
    Center.SubtractVector(x);
    *out << Verbose(2) << "Normal Vector of this front side is " << result->NormalVector << "." << endl;
    if (Center.ScalarProduct(&result->NormalVector) < 0) {
      result = triangles->back();
      *out << Verbose(2) << "Normal Vector of this back side is " << result->NormalVector << "." << endl;
      if (Center.ScalarProduct(&result->NormalVector) < 0) {
        *out << Verbose(1) << "ERROR: Front and back side yield NormalVector in wrong direction!" << endl;
      }
    }
  }
  delete(triangles);
  return result;
};

/** Checks whether the provided Vector is within the tesselation structure.
 *
 * @param point of which to check the position
 * @param *LC LinkedCell structure
 *
 * @return true if the point is inside the tesselation structure, false otherwise
 */
bool Tesselation::IsInnerPoint(ofstream *out, Vector Point, LinkedCell* LC)
{
  class BoundaryTriangleSet *result = FindClosestTriangleToPoint(out, &Point, LC);
  Vector Center;

  if (result == NULL) {// is boundary point or only point in point cloud?
    *out << Verbose(1) << Point << " is the only point in vicinity." << endl;
    return false;
  }

  result->GetCenter(&Center);
  *out << Verbose(3) << "INFO: Central point of the triangle is " << Center << "." << endl;
  Center.SubtractVector(&Point);
  *out << Verbose(3) << "INFO: Vector from center to point to test is " << Center << "." << endl;
  if (Center.ScalarProduct(&result->NormalVector) > -MYEPSILON) {
    *out << Verbose(1) << Point << " is an inner point." << endl;
    return true;
  } else {
    *out << Verbose(1) << Point << " is NOT an inner point." << endl;
    return false;
  }
}

/** Checks whether the provided TesselPoint is within the tesselation structure.
 *
 * @param *Point of which to check the position
 * @param *LC Linked Cell structure
 *
 * @return true if the point is inside the tesselation structure, false otherwise
 */
bool Tesselation::IsInnerPoint(ofstream *out, TesselPoint *Point, LinkedCell* LC)
{
  return IsInnerPoint(out, *(Point->node), LC);
}

/** Gets all points connected to the provided point by triangulation lines.
 *
 * @param *Point of which get all connected points
 *
 * @return set of the all points linked to the provided one
 */
set<TesselPoint*> * Tesselation::GetAllConnectedPoints(ofstream *out, TesselPoint* Point)
{
  set<TesselPoint*> *connectedPoints = new set<TesselPoint*>;
  class BoundaryPointSet *ReferencePoint = NULL;
  TesselPoint* current;
  bool takePoint = false;

  // find the respective boundary point
  PointMap::iterator PointRunner = PointsOnBoundary.find(Point->nr);
  if (PointRunner != PointsOnBoundary.end()) {
    ReferencePoint = PointRunner->second;
  } else {
    *out << Verbose(2) << "GetAllConnectedPoints() could not find the BoundaryPoint belonging to " << *Point << "." << endl;
    ReferencePoint = NULL;
  }

  // little trick so that we look just through lines connect to the BoundaryPoint
  // OR fall-back to look through all lines if there is no such BoundaryPoint
  LineMap *Lines = &LinesOnBoundary;
  if (ReferencePoint != NULL)
    Lines = &(ReferencePoint->lines);
  LineMap::iterator findLines = Lines->begin();
  while (findLines != Lines->end()) {
   takePoint = false;

   if (findLines->second->endpoints[0]->Nr == Point->nr) {
     takePoint = true;
     current = findLines->second->endpoints[1]->node;
   } else if (findLines->second->endpoints[1]->Nr == Point->nr) {
     takePoint = true;
     current = findLines->second->endpoints[0]->node;
   }

   if (takePoint) {
     *out << Verbose(5) << "INFO: Endpoint " << *current << " of line " << *(findLines->second) << " is enlisted." << endl;
     connectedPoints->insert(current);
   }

   findLines++;
  }

  if (connectedPoints->size() == 0) { // if have not found any points
    *out << Verbose(1) << "ERROR: We have not found any connected points to " << *Point<< "." << endl;
    return NULL;
  }

  return connectedPoints;
};


/** Gets all points connected to the provided point by triangulation lines, ordered such that we have the circle round the point.
 * Maps them down onto the plane designated by the axis \a *Point and \a *Reference. The center of all points
 * connected in the tesselation to \a *Point is mapped to spherical coordinates with the zero angle being given
 * by the mapped down \a *Reference. Hence, the biggest and the smallest angles are those of the two shanks of the
 * triangle we are looking for.
 *
 * @param *out output stream for debugging
 * @param *Point of which get all connected points
 * @param *Reference Reference vector for zero angle or NULL for no preference
 * @return list of the all points linked to the provided one
 */
list<TesselPoint*> * Tesselation::GetCircleOfConnectedPoints(ofstream *out, TesselPoint* Point, Vector *Reference)
{
  map<double, TesselPoint*> anglesOfPoints;
  set<TesselPoint*> *connectedPoints = GetAllConnectedPoints(out, Point);
  list<TesselPoint*> *connectedCircle = new list<TesselPoint*>;
  Vector center;
  Vector PlaneNormal;
  Vector AngleZero;
  Vector OrthogonalVector;
  Vector helper;

  // calculate central point
  for (set<TesselPoint*>::iterator TesselRunner = connectedPoints->begin(); TesselRunner != connectedPoints->end(); TesselRunner++)
    center.AddVector((*TesselRunner)->node);
  //*out << "Summed vectors " << center << "; number of points " << connectedPoints.size()
  //  << "; scale factor " << 1.0/connectedPoints.size();
  center.Scale(1.0/connectedPoints->size());
  *out << Verbose(4) << "INFO: Calculated center of all circle points is " << center << "." << endl;

  // projection plane of the circle is at the closes Point and normal is pointing away from center of all circle points
  PlaneNormal.CopyVector(Point->node);
  PlaneNormal.SubtractVector(&center);
  PlaneNormal.Normalize();
  *out << Verbose(4) << "INFO: Calculated plane normal of circle is " << PlaneNormal << "." << endl;

  // construct one orthogonal vector
  if (Reference != NULL)
    AngleZero.CopyVector(Reference);
  else
    AngleZero.CopyVector((*connectedPoints->begin())->node);
  AngleZero.SubtractVector(Point->node);
  AngleZero.ProjectOntoPlane(&PlaneNormal);
  *out << Verbose(4) << "INFO: Reference vector on this plane representing angle 0 is " << AngleZero << "." << endl;
  OrthogonalVector.MakeNormalVector(&PlaneNormal, &AngleZero);
  *out << Verbose(4) << "INFO: OrthogonalVector on plane is " << OrthogonalVector << "." << endl;

  // go through all connected points and calculate angle
  for (set<TesselPoint*>::iterator listRunner = connectedPoints->begin(); listRunner != connectedPoints->end(); listRunner++) {
    helper.CopyVector((*listRunner)->node);
    helper.SubtractVector(Point->node);
    helper.ProjectOntoPlane(&PlaneNormal);
    double angle = GetAngle(helper, AngleZero, OrthogonalVector);
    *out << Verbose(3) << "INFO: Calculated angle is " << angle << " for point " << **listRunner << "." << endl;
    anglesOfPoints.insert(pair<double, TesselPoint*>(angle, (*listRunner)));
  }

  for(map<double, TesselPoint*>::iterator AngleRunner = anglesOfPoints.begin(); AngleRunner != anglesOfPoints.end(); AngleRunner++) {
    connectedCircle->push_back(AngleRunner->second);
  }

  delete(connectedPoints);
  return connectedCircle;
}

/** Gets all points connected to the provided point by triangulation lines, ordered such that we walk along a closed path.
 *
 * @param *out output stream for debugging
 * @param *Point of which get all connected points
 * @return list of the all points linked to the provided one
 */
list<list<TesselPoint*> *> * Tesselation::GetPathsOfConnectedPoints(ofstream *out, TesselPoint* Point)
{
  map<double, TesselPoint*> anglesOfPoints;
  list<list<TesselPoint*> *> *ListOfPaths = new list<list<TesselPoint*> *>;
  list<TesselPoint*> *connectedPath = NULL;
  Vector center;
  Vector PlaneNormal;
  Vector AngleZero;
  Vector OrthogonalVector;
  Vector helper;
  class BoundaryPointSet *ReferencePoint = NULL;
  class BoundaryPointSet *CurrentPoint = NULL;
  class BoundaryTriangleSet *triangle = NULL;
  class BoundaryLineSet *CurrentLine = NULL;
  class BoundaryLineSet *StartLine = NULL;

  // find the respective boundary point
  PointMap::iterator PointRunner = PointsOnBoundary.find(Point->nr);
  if (PointRunner != PointsOnBoundary.end()) {
    ReferencePoint = PointRunner->second;
  } else {
    *out << Verbose(2) << "ERROR: GetPathOfConnectedPoints() could not find the BoundaryPoint belonging to " << *Point << "." << endl;
    return NULL;
  }

  map <class BoundaryLineSet *, bool> TouchedLine;
  map <class BoundaryTriangleSet *, bool> TouchedTriangle;
  map <class BoundaryLineSet *, bool>::iterator LineRunner;
  map <class BoundaryTriangleSet *, bool>::iterator TriangleRunner;
  for (LineMap::iterator Runner = ReferencePoint->lines.begin(); Runner != ReferencePoint->lines.end(); Runner++) {
    TouchedLine.insert( pair <class BoundaryLineSet *, bool>(Runner->second, false) );
    for (TriangleMap::iterator Sprinter = Runner->second->triangles.begin(); Sprinter != Runner->second->triangles.end(); Sprinter++)
      TouchedTriangle.insert( pair <class BoundaryTriangleSet *, bool>(Sprinter->second, false) );
  }
  if (!ReferencePoint->lines.empty()) {
    for (LineMap::iterator runner = ReferencePoint->lines.begin(); runner != ReferencePoint->lines.end(); runner++) {
      LineRunner = TouchedLine.find(runner->second);
      if (LineRunner == TouchedLine.end()) {
        *out << Verbose(2) << "ERROR: I could not find " << *runner->second << " in the touched list." << endl;
      } else if (!LineRunner->second) {
        LineRunner->second = true;
        connectedPath = new list<TesselPoint*>;
        triangle = NULL;
        CurrentLine = runner->second;
        StartLine = CurrentLine;
        CurrentPoint = CurrentLine->GetOtherEndpoint(ReferencePoint);
        *out << Verbose(3)<< "INFO: Beginning path retrieval at " << *CurrentPoint << " of line " << *CurrentLine << "." << endl;
        do {
          // push current one
          *out << Verbose(3) << "INFO: Putting " << *CurrentPoint << " at end of path." << endl;
          connectedPath->push_back(CurrentPoint->node);

          // find next triangle
          for (TriangleMap::iterator Runner = CurrentLine->triangles.begin(); Runner != CurrentLine->triangles.end(); Runner++) {
            *out << Verbose(3) << "INFO: Inspecting triangle " << *Runner->second << "." << endl;
            if ((Runner->second != triangle)) { // look for first triangle not equal to old one
              triangle = Runner->second;
              TriangleRunner = TouchedTriangle.find(triangle);
              if (TriangleRunner != TouchedTriangle.end()) {
                if (!TriangleRunner->second) {
                  TriangleRunner->second = true;
                  *out << Verbose(3) << "INFO: Connecting triangle is " << *triangle << "." << endl;
                  break;
                } else {
                  *out << Verbose(3) << "INFO: Skipping " << *triangle << ", as we have already visited it." << endl;
                  triangle = NULL;
                }
              } else {
                *out << Verbose(2) << "ERROR: I could not find " << *triangle << " in the touched list." << endl;
                triangle = NULL;
              }
            }
          }
          if (triangle == NULL)
            break;
          // find next line
          for (int i=0;i<3;i++) {
            if ((triangle->lines[i] != CurrentLine) && (triangle->lines[i]->ContainsBoundaryPoint(ReferencePoint))) { // not the current line and still containing Point
              CurrentLine = triangle->lines[i];
              *out << Verbose(3) << "INFO: Connecting line is " << *CurrentLine << "." << endl;
              break;
            }
          }
          LineRunner = TouchedLine.find(CurrentLine);
          if (LineRunner == TouchedLine.end())
            *out << Verbose(2) << "ERROR: I could not find " << *CurrentLine << " in the touched list." << endl;
          else
            LineRunner->second = true;
          // find next point
          CurrentPoint = CurrentLine->GetOtherEndpoint(ReferencePoint);

        } while (CurrentLine != StartLine);
        // last point is missing, as it's on start line
        *out << Verbose(3) << "INFO: Putting " << *CurrentPoint << " at end of path." << endl;
        if (StartLine->GetOtherEndpoint(ReferencePoint)->node != connectedPath->back())
          connectedPath->push_back(StartLine->GetOtherEndpoint(ReferencePoint)->node);

        ListOfPaths->push_back(connectedPath);
      } else {
        *out << Verbose(3) << "INFO: Skipping " << *runner->second << ", as we have already visited it." << endl;
      }
    }
  } else {
    *out << Verbose(1) << "ERROR: There are no lines attached to " << *ReferencePoint << "." << endl;
  }

  return ListOfPaths;
}

/** Gets all closed paths on the circle of points connected to the provided point by triangulation lines, if this very point is removed.
 * From GetPathsOfConnectedPoints() extracts all single loops of intracrossing paths in the list of closed paths.
 * @param *out output stream for debugging
 * @param *Point of which get all connected points
 * @return list of the closed paths
 */
list<list<TesselPoint*> *> * Tesselation::GetClosedPathsOfConnectedPoints(ofstream *out, TesselPoint* Point)
{
  list<list<TesselPoint*> *> *ListofPaths = GetPathsOfConnectedPoints(out, Point);
  list<list<TesselPoint*> *> *ListofClosedPaths = new list<list<TesselPoint*> *>;
  list<TesselPoint*> *connectedPath = NULL;
  list<TesselPoint*> *newPath = NULL;
  int count = 0;


  list<TesselPoint*>::iterator CircleRunner;
  list<TesselPoint*>::iterator CircleStart;

  for(list<list<TesselPoint*> *>::iterator ListRunner = ListofPaths->begin(); ListRunner != ListofPaths->end(); ListRunner++) {
    connectedPath = *ListRunner;

    *out << Verbose(2) << "INFO: Current path is " << connectedPath << "." << endl;

    // go through list, look for reappearance of starting Point and count
    CircleStart = connectedPath->begin();

    // go through list, look for reappearance of starting Point and create list
    list<TesselPoint*>::iterator Marker = CircleStart;
    for (CircleRunner = CircleStart; CircleRunner != connectedPath->end(); CircleRunner++) {
      if ((*CircleRunner == *CircleStart) && (CircleRunner != CircleStart)) { // is not the very first point
        // we have a closed circle from Marker to new Marker
        *out << Verbose(3) << count+1 << ". closed path consists of: ";
        newPath = new list<TesselPoint*>;
        list<TesselPoint*>::iterator CircleSprinter = Marker;
        for (; CircleSprinter != CircleRunner; CircleSprinter++) {
          newPath->push_back(*CircleSprinter);
          *out << (**CircleSprinter) << " <-> ";
        }
        *out << ".." << endl;
        count++;
        Marker = CircleRunner;

        // add to list
        ListofClosedPaths->push_back(newPath);
      }
    }
  }
  *out << Verbose(3) << "INFO: " << count << " closed additional path(s) have been created." << endl;

  // delete list of paths
  while (!ListofPaths->empty()) {
    connectedPath = *(ListofPaths->begin());
    ListofPaths->remove(connectedPath);
    delete(connectedPath);
  }
  delete(ListofPaths);

  // exit
  return ListofClosedPaths;
};


/** Gets all belonging triangles for a given BoundaryPointSet.
 * \param *out output stream for debugging
 * \param *Point BoundaryPoint
 * \return pointer to allocated list of triangles
 */
set<BoundaryTriangleSet*> *Tesselation::GetAllTriangles(ofstream *out, class BoundaryPointSet *Point)
{
  set<BoundaryTriangleSet*> *connectedTriangles = new set<BoundaryTriangleSet*>;

  if (Point == NULL) {
    *out << Verbose(1) << "ERROR: Point given is NULL." << endl;
  } else {
    // go through its lines and insert all triangles
    for (LineMap::iterator LineRunner = Point->lines.begin(); LineRunner != Point->lines.end(); LineRunner++)
      for (TriangleMap::iterator TriangleRunner = (LineRunner->second)->triangles.begin(); TriangleRunner != (LineRunner->second)->triangles.end(); TriangleRunner++) {
      connectedTriangles->insert(TriangleRunner->second);
    }
  }

  return connectedTriangles;
};


/** Removes a boundary point from the envelope while keeping it closed.
 * We remove the old triangles connected to the point and re-create new triangles to close the surface following this ansatz:
 *  -# a closed path(s) of boundary points surrounding the point to be removed is constructed
 *  -# on each closed path, we pick three adjacent points, create a triangle with them and subtract the middle point from the path
 *  -# we advance two points (i.e. the next triangle will start at the ending point of the last triangle) and continue as before
 *  -# the surface is closed, when the path is empty
 * Thereby, we (hopefully) make sure that the removed points remains beneath the surface (this is checked via IsInnerPoint eventually).
 * \param *out output stream for debugging
 * \param *point point to be removed
 * \return volume added to the volume inside the tesselated surface by the removal
 */
double Tesselation::RemovePointFromTesselatedSurface(ofstream *out, class BoundaryPointSet *point) {
  class BoundaryLineSet *line = NULL;
  class BoundaryTriangleSet *triangle = NULL;
  Vector OldPoint, NormalVector;
  double volume = 0;
  int count = 0;

  if (point == NULL) {
    *out << Verbose(1) << "ERROR: Cannot remove the point " << point << ", it's NULL!" << endl;
    return 0.;
  } else
    *out << Verbose(2) << "Removing point " << *point << " from tesselated boundary ..." << endl;

  // copy old location for the volume
  OldPoint.CopyVector(point->node->node);

  // get list of connected points
  if (point->lines.empty()) {
    *out << Verbose(1) << "ERROR: Cannot remove the point " << *point << ", it's connected to no lines!" << endl;
    return 0.;
  }

  list<list<TesselPoint*> *> *ListOfClosedPaths = GetClosedPathsOfConnectedPoints(out, point->node);
  list<TesselPoint*> *connectedPath = NULL;

  // gather all triangles
  for (LineMap::iterator LineRunner = point->lines.begin(); LineRunner != point->lines.end(); LineRunner++)
    count+=LineRunner->second->triangles.size();
  map<class BoundaryTriangleSet *, int> Candidates;
  for (LineMap::iterator LineRunner = point->lines.begin(); LineRunner != point->lines.end(); LineRunner++) {
    line = LineRunner->second;
    for (TriangleMap::iterator TriangleRunner = line->triangles.begin(); TriangleRunner != line->triangles.end(); TriangleRunner++) {
      triangle = TriangleRunner->second;
      Candidates.insert( pair<class BoundaryTriangleSet *, int> (triangle, triangle->Nr) );
    }
  }

  // remove all triangles
  count=0;
  NormalVector.Zero();
  for (map<class BoundaryTriangleSet *, int>::iterator Runner = Candidates.begin(); Runner != Candidates.end(); Runner++) {
    *out << Verbose(3) << "INFO: Removing triangle " << *(Runner->first) << "." << endl;
    NormalVector.SubtractVector(&Runner->first->NormalVector); // has to point inward
    RemoveTesselationTriangle(Runner->first);
    count++;
  }
  *out << Verbose(1) << count << " triangles were removed." << endl;

  list<list<TesselPoint*> *>::iterator ListAdvance = ListOfClosedPaths->begin();
  list<list<TesselPoint*> *>::iterator ListRunner = ListAdvance;
  map<class BoundaryTriangleSet *, int>::iterator NumberRunner = Candidates.begin();
  list<TesselPoint*>::iterator StartNode, MiddleNode, EndNode;
  double angle;
  double smallestangle;
  Vector Point, Reference, OrthogonalVector;
  if (count > 2) {  // less than three triangles, then nothing will be created
    class TesselPoint *TriangleCandidates[3];
    count = 0;
    for ( ; ListRunner != ListOfClosedPaths->end(); ListRunner = ListAdvance) {  // go through all closed paths
      if (ListAdvance != ListOfClosedPaths->end())
        ListAdvance++;

      connectedPath = *ListRunner;

      // re-create all triangles by going through connected points list
      list<class BoundaryLineSet *> NewLines;
      for (;!connectedPath->empty();) {
        // search middle node with widest angle to next neighbours
        EndNode = connectedPath->end();
        smallestangle = 0.;
        for (MiddleNode = connectedPath->begin(); MiddleNode != connectedPath->end(); MiddleNode++) {
          cout << Verbose(3) << "INFO: MiddleNode is " << **MiddleNode << "." << endl;
          // construct vectors to next and previous neighbour
          StartNode = MiddleNode;
          if (StartNode == connectedPath->begin())
            StartNode = connectedPath->end();
          StartNode--;
          //cout << Verbose(3) << "INFO: StartNode is " << **StartNode << "." << endl;
          Point.CopyVector((*StartNode)->node);
          Point.SubtractVector((*MiddleNode)->node);
          StartNode = MiddleNode;
          StartNode++;
          if (StartNode == connectedPath->end())
            StartNode = connectedPath->begin();
          //cout << Verbose(3) << "INFO: EndNode is " << **StartNode << "." << endl;
          Reference.CopyVector((*StartNode)->node);
          Reference.SubtractVector((*MiddleNode)->node);
          OrthogonalVector.CopyVector((*MiddleNode)->node);
          OrthogonalVector.SubtractVector(&OldPoint);
          OrthogonalVector.MakeNormalVector(&Reference);
          angle = GetAngle(Point, Reference, OrthogonalVector);
          //if (angle < M_PI)  // no wrong-sided triangles, please?
            if(fabs(angle - M_PI) < fabs(smallestangle - M_PI)) {  // get straightest angle (i.e. construct those triangles with smallest area first)
              smallestangle = angle;
              EndNode = MiddleNode;
            }
        }
        MiddleNode = EndNode;
        if (MiddleNode == connectedPath->end()) {
          cout << Verbose(1) << "CRITICAL: Could not find a smallest angle!" << endl;
          exit(255);
        }
        StartNode = MiddleNode;
        if (StartNode == connectedPath->begin())
          StartNode = connectedPath->end();
        StartNode--;
        EndNode++;
        if (EndNode == connectedPath->end())
          EndNode = connectedPath->begin();
        cout << Verbose(4) << "INFO: StartNode is " << **StartNode << "." << endl;
        cout << Verbose(4) << "INFO: MiddleNode is " << **MiddleNode << "." << endl;
        cout << Verbose(4) << "INFO: EndNode is " << **EndNode << "." << endl;
        *out << Verbose(3) << "INFO: Attempting to create triangle " << (*StartNode)->Name << ", " << (*MiddleNode)->Name << " and " << (*EndNode)->Name << "." << endl;
        TriangleCandidates[0] = *StartNode;
        TriangleCandidates[1] = *MiddleNode;
        TriangleCandidates[2] = *EndNode;
        triangle = GetPresentTriangle(out, TriangleCandidates);
        if (triangle != NULL) {
          cout << Verbose(1) << "WARNING: New triangle already present, skipping!" << endl;
          StartNode++;
          MiddleNode++;
          EndNode++;
          if (StartNode == connectedPath->end())
            StartNode = connectedPath->begin();
          if (MiddleNode == connectedPath->end())
            MiddleNode = connectedPath->begin();
          if (EndNode == connectedPath->end())
            EndNode = connectedPath->begin();
          continue;
        }
        *out << Verbose(5) << "Adding new triangle points."<< endl;
        AddTesselationPoint(*StartNode, 0);
        AddTesselationPoint(*MiddleNode, 1);
        AddTesselationPoint(*EndNode, 2);
        *out << Verbose(5) << "Adding new triangle lines."<< endl;
        AddTesselationLine(TPS[0], TPS[1], 0);
        AddTesselationLine(TPS[0], TPS[2], 1);
        NewLines.push_back(BLS[1]);
        AddTesselationLine(TPS[1], TPS[2], 2);
        BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
        BTS->GetNormalVector(NormalVector);
        AddTesselationTriangle();
        // calculate volume summand as a general tetraeder
        volume += CalculateVolumeofGeneralTetraeder(TPS[0]->node->node, TPS[1]->node->node, TPS[2]->node->node, &OldPoint);
        // advance number
        count++;

        // prepare nodes for next triangle
        StartNode = EndNode;
        cout << Verbose(4) << "Removing " << **MiddleNode << " from closed path, remaining points: " << connectedPath->size() << "." << endl;
        connectedPath->remove(*MiddleNode); // remove the middle node (it is surrounded by triangles)
        if (connectedPath->size() == 2) { // we are done
          connectedPath->remove(*StartNode); // remove the start node
          connectedPath->remove(*EndNode); // remove the end node
          break;
        } else if (connectedPath->size() < 2) { // something's gone wrong!
          cout << Verbose(1) << "CRITICAL: There are only two endpoints left!" << endl;
          exit(255);
        } else {
          MiddleNode = StartNode;
          MiddleNode++;
          if (MiddleNode == connectedPath->end())
            MiddleNode = connectedPath->begin();
          EndNode = MiddleNode;
          EndNode++;
          if (EndNode == connectedPath->end())
            EndNode = connectedPath->begin();
        }
      }
      // maximize the inner lines (we preferentially created lines with a huge angle, which is for the tesselation not wanted though useful for the closing)
      if (NewLines.size() > 1) {
        list<class BoundaryLineSet *>::iterator Candidate;
        class BoundaryLineSet *OtherBase = NULL;
        double tmp, maxgain;
        do {
          maxgain = 0;
          for(list<class BoundaryLineSet *>::iterator Runner = NewLines.begin(); Runner != NewLines.end(); Runner++) {
            tmp = PickFarthestofTwoBaselines(out, *Runner);
            if (maxgain < tmp) {
              maxgain = tmp;
              Candidate = Runner;
            }
          }
          if (maxgain != 0) {
            volume += maxgain;
            cout << Verbose(3) << "Flipping baseline with highest volume" << **Candidate << "." << endl;
            OtherBase = FlipBaseline(out, *Candidate);
            NewLines.erase(Candidate);
            NewLines.push_back(OtherBase);
          }
        } while (maxgain != 0.);
      }

      ListOfClosedPaths->remove(connectedPath);
      delete(connectedPath);
    }
    *out << Verbose(1) << count << " triangles were created." << endl;
  } else {
    while (!ListOfClosedPaths->empty()) {
      ListRunner = ListOfClosedPaths->begin();
      connectedPath = *ListRunner;
      ListOfClosedPaths->remove(connectedPath);
      delete(connectedPath);
    }
    *out << Verbose(1) << "No need to create any triangles." << endl;
  }
  delete(ListOfClosedPaths);

  *out << Verbose(1) << "Removed volume is " << volume << "." << endl;

  return volume;
};



/**
 * Finds triangles belonging to the three provided points.
 *
 * @param *Points[3] list, is expected to contain three points
 *
 * @return triangles which belong to the provided points, will be empty if there are none,
 *         will usually be one, in case of degeneration, there will be two
 */
list<BoundaryTriangleSet*> *Tesselation::FindTriangles(TesselPoint* Points[3]) 
{
  list<BoundaryTriangleSet*> *result = new list<BoundaryTriangleSet*>;
  LineMap::iterator FindLine;
  PointMap::iterator FindPoint;
  TriangleMap::iterator FindTriangle;
  class BoundaryPointSet *TrianglePoints[3];

  for (int i = 0; i < 3; i++) {
    FindPoint = PointsOnBoundary.find(Points[i]->nr);
    if (FindPoint != PointsOnBoundary.end()) {
      TrianglePoints[i] = FindPoint->second;
    } else {
      TrianglePoints[i] = NULL;
    }
  }

  // checks lines between the points in the Points for their adjacent triangles
  for (int i = 0; i < 3; i++) {
    if (TrianglePoints[i] != NULL) {
      for (int j = i; j < 3; j++) {
        if (TrianglePoints[j] != NULL) {
          FindLine = TrianglePoints[i]->lines.find(TrianglePoints[j]->node->nr);
          if (FindLine != TrianglePoints[i]->lines.end()) {
            for (; FindLine->first == TrianglePoints[j]->node->nr; FindLine++) {
              FindTriangle = FindLine->second->triangles.begin();
              for (; FindTriangle != FindLine->second->triangles.end(); FindTriangle++) {
                if (FindTriangle->second->IsPresentTupel(TrianglePoints)) {
                  result->push_back(FindTriangle->second);
                }
              }
            }
            // Is it sufficient to consider one of the triangle lines for this.
            return result;

          }
        }
      }
    }
  }

  return result;
}

/**
 * Finds all degenerated lines within the tesselation structure.
 *
 * @return map of keys of degenerated line pairs, each line occurs twice
 *         in the list, once as key and once as value
 */
map<int, int> * Tesselation::FindAllDegeneratedLines()
{
  map<int, class BoundaryLineSet *> AllLines;
  map<int, int> * DegeneratedLines = new map<int, int>;

  // sanity check
  if (LinesOnBoundary.empty()) {
    cout << Verbose(1) << "Warning: FindAllDegeneratedTriangles() was called without any tesselation structure.";
    return DegeneratedLines;
  }

  LineMap::iterator LineRunner1;
  pair<LineMap::iterator, bool> tester;
  for (LineRunner1 = LinesOnBoundary.begin(); LineRunner1 != LinesOnBoundary.end(); ++LineRunner1) {
    tester = AllLines.insert( pair<int,BoundaryLineSet *> (LineRunner1->second->endpoints[0]->Nr, LineRunner1->second) );
    if ((!tester.second) && (tester.first->second->endpoints[1]->Nr == LineRunner1->second->endpoints[1]->Nr)) { // found degenerated line
      DegeneratedLines->insert ( pair<int, int> (LineRunner1->second->Nr, tester.first->second->Nr) );
      DegeneratedLines->insert ( pair<int, int> (tester.first->second->Nr, LineRunner1->second->Nr) );
    }
  }

  AllLines.clear();

  cout << Verbose(1) << "FindAllDegeneratedLines() found " << DegeneratedLines->size() << " lines." << endl;
  map<int,int>::iterator it;
  for (it = DegeneratedLines->begin(); it != DegeneratedLines->end(); it++)
      cout << Verbose(2) << (*it).first << " => " << (*it).second << endl;

  return DegeneratedLines;
}

/**
 * Finds all degenerated triangles within the tesselation structure.
 *
 * @return map of keys of degenerated triangle pairs, each triangle occurs twice
 *         in the list, once as key and once as value
 */
map<int, int> * Tesselation::FindAllDegeneratedTriangles()
{
  map<int, int> * DegeneratedLines = FindAllDegeneratedLines();
  map<int, int> * DegeneratedTriangles = new map<int, int>;

  TriangleMap::iterator TriangleRunner1, TriangleRunner2;
  LineMap::iterator Liner;
  class BoundaryLineSet *line1 = NULL, *line2 = NULL;

  for (map<int, int>::iterator LineRunner = DegeneratedLines->begin(); LineRunner != DegeneratedLines->end(); ++LineRunner) {
    // run over both lines' triangles
    Liner = LinesOnBoundary.find(LineRunner->first);
    if (Liner != LinesOnBoundary.end())
      line1 = Liner->second;
    Liner = LinesOnBoundary.find(LineRunner->second);
    if (Liner != LinesOnBoundary.end())
      line2 = Liner->second;
    for (TriangleRunner1 = line1->triangles.begin(); TriangleRunner1 != line1->triangles.end(); ++TriangleRunner1) {
      for (TriangleRunner2 = line2->triangles.begin(); TriangleRunner2 != line2->triangles.end(); ++TriangleRunner2) {
        if ((TriangleRunner1->second != TriangleRunner2->second)
          && (TriangleRunner1->second->IsPresentTupel(TriangleRunner2->second))) {
          DegeneratedTriangles->insert( pair<int, int> (TriangleRunner1->second->Nr, TriangleRunner2->second->Nr) );
          DegeneratedTriangles->insert( pair<int, int> (TriangleRunner2->second->Nr, TriangleRunner1->second->Nr) );
        }
      }
    }
  }
  delete(DegeneratedLines);

  cout << Verbose(1) << "FindAllDegeneratedTriangles() found " << DegeneratedTriangles->size() << " triangles:" << endl;
  map<int,int>::iterator it;
  for (it = DegeneratedTriangles->begin(); it != DegeneratedTriangles->end(); it++)
      cout << Verbose(2) << (*it).first << " => " << (*it).second << endl;

  return DegeneratedTriangles;
}

/**
 * Purges degenerated triangles from the tesselation structure if they are not
 * necessary to keep a single point within the structure.
 */
void Tesselation::RemoveDegeneratedTriangles()
{
  map<int, int> * DegeneratedTriangles = FindAllDegeneratedTriangles();
  TriangleMap::iterator finder;
  BoundaryTriangleSet *triangle = NULL, *partnerTriangle = NULL;
  int count  = 0;

  cout << Verbose(1) << "Begin of RemoveDegeneratedTriangles" << endl;

  for (map<int, int>::iterator TriangleKeyRunner = DegeneratedTriangles->begin();
    TriangleKeyRunner != DegeneratedTriangles->end(); ++TriangleKeyRunner
  ) {
    finder = TrianglesOnBoundary.find(TriangleKeyRunner->first);
    if (finder != TrianglesOnBoundary.end())
      triangle = finder->second;
    else
      break;
    finder = TrianglesOnBoundary.find(TriangleKeyRunner->second);
    if (finder != TrianglesOnBoundary.end())
      partnerTriangle = finder->second;
    else
      break;

    bool trianglesShareLine = false;
    for (int i = 0; i < 3; ++i)
      for (int j = 0; j < 3; ++j)
        trianglesShareLine = trianglesShareLine || triangle->lines[i] == partnerTriangle->lines[j];

    if (trianglesShareLine
      && (triangle->endpoints[1]->LinesCount > 2)
      && (triangle->endpoints[2]->LinesCount > 2)
      && (triangle->endpoints[0]->LinesCount > 2)
    ) {
      // check whether we have to fix lines
      BoundaryTriangleSet *Othertriangle = NULL;
      BoundaryTriangleSet *OtherpartnerTriangle = NULL;
      TriangleMap::iterator TriangleRunner;
      for (int i = 0; i < 3; ++i)
        for (int j = 0; j < 3; ++j)
          if (triangle->lines[i] != partnerTriangle->lines[j]) {
            // get the other two triangles
            for (TriangleRunner = triangle->lines[i]->triangles.begin(); TriangleRunner != triangle->lines[i]->triangles.end(); ++TriangleRunner)
              if (TriangleRunner->second != triangle) {
                Othertriangle = TriangleRunner->second;
              }
            for (TriangleRunner = partnerTriangle->lines[i]->triangles.begin(); TriangleRunner != partnerTriangle->lines[i]->triangles.end(); ++TriangleRunner)
              if (TriangleRunner->second != partnerTriangle) {
                OtherpartnerTriangle = TriangleRunner->second;
              }
            /// interchanges their lines so that triangle->lines[i] == partnerTriangle->lines[j]
            // the line of triangle receives the degenerated ones
            triangle->lines[i]->triangles.erase(Othertriangle->Nr);
            triangle->lines[i]->triangles.insert( TrianglePair( partnerTriangle->Nr, partnerTriangle) );
            for (int k=0;k<3;k++)
              if (triangle->lines[i] == Othertriangle->lines[k]) {
                Othertriangle->lines[k] = partnerTriangle->lines[j];
                break;
              }
            // the line of partnerTriangle receives the non-degenerated ones
            partnerTriangle->lines[j]->triangles.erase( partnerTriangle->Nr);
            partnerTriangle->lines[j]->triangles.insert( TrianglePair( Othertriangle->Nr, Othertriangle) );
            partnerTriangle->lines[j] = triangle->lines[i];
          }

      // erase the pair
      count += (int) DegeneratedTriangles->erase(triangle->Nr);
      cout << Verbose(1) << "RemoveDegeneratedTriangles() removes triangle " << *triangle << "." << endl;
      RemoveTesselationTriangle(triangle);
      count += (int) DegeneratedTriangles->erase(partnerTriangle->Nr);
      cout << Verbose(1) << "RemoveDegeneratedTriangles() removes triangle " << *partnerTriangle << "." << endl;
      RemoveTesselationTriangle(partnerTriangle);
    } else {
      cout << Verbose(1) << "RemoveDegeneratedTriangles() does not remove triangle " << *triangle
        << " and its partner " << *partnerTriangle << " because it is essential for at"
        << " least one of the endpoints to be kept in the tesselation structure." << endl;
    }
  }
  delete(DegeneratedTriangles);

  cout << Verbose(1) << "RemoveDegeneratedTriangles() removed " << count << " triangles:" << endl;
  cout << Verbose(1) << "End of RemoveDegeneratedTriangles" << endl;
}

/** Adds an outside Tesselpoint to the envelope via (two) degenerated triangles.
 * We look for the closest point on the boundary, we look through its connected boundary lines and
 * seek the one with the minimum angle between its center point and the new point and this base line.
 * We open up the line by adding a degenerated triangle, whose other side closes the base line again.
 * \param *out output stream for debugging
 * \param *point point to add
 * \param *LC Linked Cell structure to find nearest point
 */
void Tesselation::AddBoundaryPointByDegeneratedTriangle(ofstream *out, class TesselPoint *point, LinkedCell *LC)
{
  *out << Verbose(2) << "Begin of AddBoundaryPointByDegeneratedTriangle" << endl;

  // find nearest boundary point
  class TesselPoint *BackupPoint = NULL;
  class TesselPoint *NearestPoint = FindClosestPoint(point->node, BackupPoint, LC);
  class BoundaryPointSet *NearestBoundaryPoint = NULL;
  PointMap::iterator PointRunner;

  if (NearestPoint == point)
    NearestPoint = BackupPoint;
  PointRunner = PointsOnBoundary.find(NearestPoint->nr);
  if (PointRunner != PointsOnBoundary.end()) {
    NearestBoundaryPoint = PointRunner->second;
  } else {
    *out << Verbose(1) << "ERROR: I cannot find the boundary point." << endl;
    return;
  }
  *out << Verbose(2) << "Nearest point on boundary is " << NearestPoint->Name << "." << endl;

  // go through its lines and find the best one to split
  Vector CenterToPoint;
  Vector BaseLine;
  double angle, BestAngle = 0.;
  class BoundaryLineSet *BestLine = NULL;
  for (LineMap::iterator Runner = NearestBoundaryPoint->lines.begin(); Runner != NearestBoundaryPoint->lines.end(); Runner++) {
    BaseLine.CopyVector(Runner->second->endpoints[0]->node->node);
    BaseLine.SubtractVector(Runner->second->endpoints[1]->node->node);
    CenterToPoint.CopyVector(Runner->second->endpoints[0]->node->node);
    CenterToPoint.AddVector(Runner->second->endpoints[1]->node->node);
    CenterToPoint.Scale(0.5);
    CenterToPoint.SubtractVector(point->node);
    angle = CenterToPoint.Angle(&BaseLine);
    if (fabs(angle - M_PI/2.) < fabs(BestAngle - M_PI/2.)) {
      BestAngle = angle;
      BestLine = Runner->second;
    }
  }

  // remove one triangle from the chosen line
  class BoundaryTriangleSet *TempTriangle = (BestLine->triangles.begin())->second;
  BestLine->triangles.erase(TempTriangle->Nr);
  int nr = -1;
  for (int i=0;i<3; i++) {
    if (TempTriangle->lines[i] == BestLine) {
      nr = i;
      break;
    }
  }

  // create new triangle to connect point (connects automatically with the missing spot of the chosen line)
  *out << Verbose(5) << "Adding new triangle points."<< endl;
  AddTesselationPoint((BestLine->endpoints[0]->node), 0);
  AddTesselationPoint((BestLine->endpoints[1]->node), 1);
  AddTesselationPoint(point, 2);
  *out << Verbose(5) << "Adding new triangle lines."<< endl;
  AddTesselationLine(TPS[0], TPS[1], 0);
  AddTesselationLine(TPS[0], TPS[2], 1);
  AddTesselationLine(TPS[1], TPS[2], 2);
  BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
  BTS->GetNormalVector(TempTriangle->NormalVector);
  BTS->NormalVector.Scale(-1.);
  *out << Verbose(3) << "INFO: NormalVector of new triangle is " << BTS->NormalVector << "." << endl;
  AddTesselationTriangle();

  // create other side of this triangle and close both new sides of the first created triangle
  *out << Verbose(5) << "Adding new triangle points."<< endl;
  AddTesselationPoint((BestLine->endpoints[0]->node), 0);
  AddTesselationPoint((BestLine->endpoints[1]->node), 1);
  AddTesselationPoint(point, 2);
  *out << Verbose(5) << "Adding new triangle lines."<< endl;
  AddTesselationLine(TPS[0], TPS[1], 0);
  AddTesselationLine(TPS[0], TPS[2], 1);
  AddTesselationLine(TPS[1], TPS[2], 2);
  BTS = new class BoundaryTriangleSet(BLS, TrianglesOnBoundaryCount);
  BTS->GetNormalVector(TempTriangle->NormalVector);
  *out << Verbose(3) << "INFO: NormalVector of other new triangle is " << BTS->NormalVector << "." << endl;
  AddTesselationTriangle();

  // add removed triangle to the last open line of the second triangle
  for (int i=0;i<3;i++) { // look for the same line as BestLine (only it's its degenerated companion)
    if ((BTS->lines[i]->ContainsBoundaryPoint(BestLine->endpoints[0])) && (BTS->lines[i]->ContainsBoundaryPoint(BestLine->endpoints[1]))) {
      if (BestLine == BTS->lines[i]){
        *out << Verbose(1) << "CRITICAL: BestLine is same as found line, something's wrong here!" << endl;
        exit(255);
      }
      BTS->lines[i]->triangles.insert( pair<int, class BoundaryTriangleSet *> (TempTriangle->Nr, TempTriangle) );
      TempTriangle->lines[nr] = BTS->lines[i];
      break;
    }
  }

  // exit
  *out << Verbose(2) << "End of AddBoundaryPointByDegeneratedTriangle" << endl;
};

/** Writes the envelope to file.
 * \param *out otuput stream for debugging
 * \param *filename basename of output file
 * \param *cloud PointCloud structure with all nodes
 */
void Tesselation::Output(ofstream *out, const char *filename, PointCloud *cloud)
{
  ofstream *tempstream = NULL;
  string NameofTempFile;
  char NumberName[255];

  if (LastTriangle != NULL) {
    sprintf(NumberName, "-%04d-%s_%s_%s", (int)TrianglesOnBoundary.size(), LastTriangle->endpoints[0]->node->Name, LastTriangle->endpoints[1]->node->Name, LastTriangle->endpoints[2]->node->Name);
    if (DoTecplotOutput) {
      string NameofTempFile(filename);
      NameofTempFile.append(NumberName);
      for(size_t npos = NameofTempFile.find_first_of(' '); npos != string::npos; npos = NameofTempFile.find(' ', npos))
      NameofTempFile.erase(npos, 1);
      NameofTempFile.append(TecplotSuffix);
      *out << Verbose(1) << "Writing temporary non convex hull to file " << NameofTempFile << ".\n";
      tempstream = new ofstream(NameofTempFile.c_str(), ios::trunc);
      WriteTecplotFile(out, tempstream, this, cloud, TriangleFilesWritten);
      tempstream->close();
      tempstream->flush();
      delete(tempstream);
    }

    if (DoRaster3DOutput) {
      string NameofTempFile(filename);
      NameofTempFile.append(NumberName);
      for(size_t npos = NameofTempFile.find_first_of(' '); npos != string::npos; npos = NameofTempFile.find(' ', npos))
      NameofTempFile.erase(npos, 1);
      NameofTempFile.append(Raster3DSuffix);
      *out << Verbose(1) << "Writing temporary non convex hull to file " << NameofTempFile << ".\n";
      tempstream = new ofstream(NameofTempFile.c_str(), ios::trunc);
      WriteRaster3dFile(out, tempstream, this, cloud);
      IncludeSphereinRaster3D(out, tempstream, this, cloud);
      tempstream->close();
      tempstream->flush();
      delete(tempstream);
    }
  }
  if (DoTecplotOutput || DoRaster3DOutput)
    TriangleFilesWritten++;
};
